% Generated by roxygen2 (4.1.0): do not edit by hand
% Please edit documentation in R/fit.meta.GMCM.R
\name{fit.meta.GMCM}
\alias{fit.meta.GMCM}
\alias{fit.meta.gmcm}
\title{Reproducibility/meta analysis using GMCMs}
\usage{
fit.meta.GMCM(u, init.par, method = c("NM", "SANN", "L-BFGS", "L-BFGS-B",
  "PEM"), max.ite = 1000, verbose = TRUE, positive.rho = TRUE,
  trace.theta = FALSE, ...)
}
\arguments{
\item{u}{An \code{n} by \code{d} matrix of test statistics. Rows correspond
to features and columns to experiments. Large values are assumed to be
indicative of reproducible genes.}

\item{init.par}{A 4-dimensional vector of the initial parameters where,
\code{init.par[1]} is the mixture proportion of spurious signals,
\code{init.par[2]} is the mean, \code{init.par[3]} is the standard
deviation, \code{init.par[4]} is the correlation.}

\item{method}{A character vector of length \eqn{1}{1}. The optimization
method used. Should be either \code{"NM"}, \code{"SANN"}, \code{"L-BFGS"},
\code{"L-BFGS-B"}, or \code{"PEM"} which are abbreviations of Nelder-Mead,
Simulated Annealing, limited-memory quasi-Newton method, limited-memory
quasi-Newton method with box constraints, and the pseudo EM algorithm,
respectively. Default is \code{"NM"}. See \code{\link{optim}} for further
details.}

\item{max.ite}{The maximum number of iterations.  If the \code{method} is
\code{"SANN"} this is the number of iterations as there is no other
stopping criterion. (See \code{\link{optim}})}

\item{verbose}{Logical. If \code{TRUE}, the log-likelihood values are
printed.}

\item{positive.rho}{\code{logical}. If \code{TRUE}, the correlation parameter
is restricted to be positive.}

\item{trace.theta}{\code{logical}. Extra convergence information is appended
as a list to the output returned if \code{TRUE}. The exact behavior is
dependent on the value of \code{method}. If \code{method} equals
\code{"PEM"}, the argument is passed to \code{trace.theta} in
\code{\link{PseudoEMAlgorithm}}. Otherwise it is passed to the control
argument \code{trace} in \code{\link{optim}}.}

\item{\dots}{Arguments passed to the \code{control}-list in
\code{\link{optim}} or \code{\link{PseudoEMAlgorithm}} if \code{method} is
\code{"PEM"}.}
}
\value{
A vector \code{par} of length 4 of the fitted parameters where
  \code{par[1]} is the probability of being from the first (or null)
  component, \code{par[2]} is the mean, \code{par[3]} is the standard
  deviation, and \code{par[4]} is the correlation.
}
\description{
This function performs reproducibility (or meta) analysis using GMCMs.
It features various optimization routines to identify the maximum likelihood
estimate of the special Gaussian mixture copula model proposed by
Li et. al. (2011).
}
\details{
The \code{"L-BFGS-B"} method does not perform a transformation of the
parameters.
}
\note{
Simulated annealing is strongly dependent on the initial values and
  the cooling scheme.

  See \code{\link{optim}} for further details.
}
\examples{
set.seed(1)

# True parameters
true.par <- c(0.9, 2, 0.7, 0.6)
# Simulation of data from the GMCM model
data <- SimulateGMCMData(n = 1000, par = true.par)
uhat <- Uhat(data$u) # Ranked observed data

init.par <- c(0.5, 1, 0.5, 0.9)  # Initial parameters

# Optimization with Nelder-Mead
nm.par   <- fit.meta.GMCM(uhat, init.par = init.par, method = "NM")

\dontrun{
# Comparison with other optimization methods
# Optimization with simulated annealing
sann.par <- fit.meta.GMCM(uhat, init.par = init.par, method = "SANN",
                          max.ite = 3000, temp = 1)
# Optimization with the Pseudo EM algorithm
pem.par  <- fit.meta.GMCM(uhat, init.par = init.par, method = "PEM")

# The estimates agree nicely
rbind("True" = true.par, "Start" = init.par,
      "NM" = nm.par, "SANN" = sann.par, "PEM" = pem.par)
}

# Get estimated cluster
Khat <- get.IDR(x = uhat, par = nm.par)$Khat
plot(uhat, col = Khat, main = "Clustering\\nIDR < 0.05")
}
\author{
Anders Ellern Bilgrau (abilgrau@math.aau.dk)
}
\references{
Li, Q., Brown, J. B. J. B., Huang, H., & Bickel, P. J. (2011).
  Measuring reproducibility of high-throughput experiments. The Annals of
  Applied Statistics, 5(3), 1752-1779. doi:10.1214/11-AOAS466
}
\seealso{
\code{\link{optim}}
}

