% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/chapmanRobson.R
\name{chapmanRobson}
\alias{chapmanRobson}
\alias{chapmanRobson.default}
\alias{chapmanRobson.formula}
\alias{plot.chapmanRobson}
\alias{summary.chapmanRobson}
\alias{confint.chapmanRobson}
\alias{coef.chapmanRobson}
\title{Computes Chapman-Robson estimates of S and Z.}
\usage{
chapmanRobson(x, ...)

\method{chapmanRobson}{default}(
  x,
  catch,
  ages2use = age,
  zmethod = c("Smithetal", "Hoenigetal", "original"),
  ...
)

\method{chapmanRobson}{formula}(
  x,
  data,
  ages2use = age,
  zmethod = c("Smithetal", "Hoenigetal", "original"),
  ...
)

\method{summary}{chapmanRobson}(object, parm = c("all", "both", "Z", "S"), verbose = FALSE, ...)

\method{coef}{chapmanRobson}(object, parm = c("all", "both", "Z", "S"), ...)

\method{confint}{chapmanRobson}(
  object,
  parm = c("all", "both", "S", "Z"),
  level = conf.level,
  conf.level = 0.95,
  ...
)

\method{plot}{chapmanRobson}(
  x,
  pos.est = "topright",
  cex.est = 0.95,
  round.est = c(3, 1),
  ylab = "Catch",
  xlab = "Age",
  ylim = NULL,
  col.pt = "gray30",
  axis.age = c("both", "age", "recoded age"),
  ...
)
}
\arguments{
\item{x}{A numerical vector of the assigned ages in the catch curve or a formula of the form \code{catch~age} when used in \code{chapmanRobson}. An object saved from \code{chapmanRobson} (i.e., of class \code{chapmanRobson}) when used in the methods.}

\item{\dots}{Additional arguments for methods.}

\item{catch}{A numerical vector of the catches or CPUEs for the ages in the catch curve. Not used if \code{x} is a formula.}

\item{ages2use}{A numerical vector of the ages that define the descending limb of the catch curve.}

\item{zmethod}{A string that indicates the method to use for estimating Z. See details.}

\item{data}{A data frame from which the variables in the \code{x} formula can be found. Not used if \code{x} is not a formula.}

\item{object}{An object saved from the \code{chapmanRobson} call (i.e., of class \code{chapmanRobson}).}

\item{parm}{A numeric or string (of parameter names) vector that specifies which parameters are to be given confidence intervals  If missing, all parameters are considered.}

\item{verbose}{A logical that indicates whether the method should return just the estimate (\code{FALSE}; default) or a more verbose statement.}

\item{level}{Same as \code{conf.level}. Used for compatibility with the generic \code{confint} function.}

\item{conf.level}{A number representing the level of confidence to use for constructing confidence intervals.}

\item{pos.est}{A string to identify where to place the estimated mortality rates on the plot. Can be set to one of \code{"bottomright"}, \code{"bottom"}, \code{"bottomleft"}, \code{"left"}, \code{"topleft"}, \code{"top"}, \code{"topright"}, \code{"right"} or \code{"center"} for positioning the estimated mortality rates on the plot. Typically \code{"bottomleft"} (DEFAULT) and \code{"topright"} will be \dQuote{out-of-the-way} placements. Set \code{pos.est} to \code{NULL} to remove the estimated mortality rates from the plot.}

\item{cex.est}{A single numeric character expansion value for the estimated mortality rates on the plot.}

\item{round.est}{A numeric that indicates the number of decimal place to which Z (first value) and S (second value) should be rounded. If only one value then it will be used for both Z and S.}

\item{ylab}{A label for the y-axis (\code{"Catch"} is the default).}

\item{xlab}{A label for the x-axis (\code{"Age"} is the default).}

\item{ylim}{A numeric for the limits of the y-axis. If \code{NULL} then will default to 0 or the lowest catch and a maximum of the maximum catch. If a single value then it will be the maximum of the y-axis. If two values then these will the minimum and maximum values of the y-axis.}

\item{col.pt}{A string that indicates the color of the plotted points.}

\item{axis.age}{A string that indicates the type of x-axis to display. The \code{age} will display only the original ages, \code{recoded age} will display only the recoded ages, and \code{both} (DEFAULT) displays the original ages on the main axis and the recoded ages on the secondary axis.}
}
\value{
A list with the following items:
 \itemize{
   \item age the original vector of assigned ages.
   \item catch the original vector of observed catches or CPUEs.
   \item age.e a vector of assigned ages used to estimate mortalities.
   \item catch.e a vector of catches or CPUEs used to estimate mortalities.
   \item age.r a vector of recoded ages used to estimate mortalities. See references.
   \item n a numeric holding the intermediate calculation of n. See references.
   \item T a numeric holding the intermediate calculation of T. See references.
   \item est A 2x2 matrix that contains the estimates and standard errors for S and Z.
 }
}
\description{
Computes the Chapman-Robson estimates of annual survival rate (S) and instantaneous mortality rate (Z) from catch-at-age data on the descending limb of a catch-curve. Method functions extract estimates with associated standard errors and confidence intervals. A plot method highlights the descending-limb, shows the linear model on the descending limb, and, optionally, prints the estimated Z and A.
}
\details{
The default is to use all ages in the age vector. This is only appropriate if the age and catch vectors contain only the ages and catches on the descending limb of the catch curve. Use \code{ages2use} to isolate only the catch and ages on the descending limb.

The Chapman-Robson method provides an estimate of the annual survival rate, with the annual mortality rate (A) determined by 1-S. The instantaneous mortality rate is often computed as -log(S). However, Hoenig \emph{et al.} (1983) showed that this produced a biased (over)estimate of Z and provided a correction. The correction is applied by setting \code{zmethod="Hoenigetal"}. Smith \emph{et al.} (2012) showed that the Hoenig \emph{et al.} method should be corrected for a variance inflation factor. This correction is applied by setting \code{zmethod="Smithetal"} (which is the default behavior). Choose \code{zmethod="original"} to use the original estimates for Z and it's SE as provided by Chapman and Robson.
}
\section{Testing}{
 Tested the results of chapmanRobson against the results in Miranda and Bettoli (2007). The point estimates of S matched perfectly but the SE of S did not because Miranda and Bettoli used a rounded estimate of S in the calculation of the SE of S but chapmanRobson does not.

Tested the results against the results from \code{agesurv} in \pkg{fishmethods} using the \code{rockbass} data.frame in \pkg{fishmethods}. Results for Z and the SE of Z matched perfectly for non-bias-corrected results. The estimate of Z, but not the SE of Z, matched for the bias-corrected (following Smith \emph{et al.} (2012)) results. \pkg{FSA} uses equation 2 from Smith \emph{et al.} (2012) whereas \pkg{fishmethods} appears to use equation 5 from the same source to estimate the SE of Z.
}

\section{IFAR Chapter}{
 11-Mortality.
}

\examples{
plot(catch~age,data=BrookTroutTH,pch=19)

## demonstration of formula notation
cr1 <- chapmanRobson(catch~age,data=BrookTroutTH,ages2use=2:6)
summary(cr1)
summary(cr1,verbose=TRUE)
cbind(Est=coef(cr1),confint(cr1))
plot(cr1)
plot(cr1,axis.age="age")
plot(cr1,axis.age="recoded age")
summary(cr1,parm="Z")
cbind(Est=coef(cr1,parm="Z"),confint(cr1,parm="Z"))

## demonstration of excluding ages2use
cr2 <- chapmanRobson(catch~age,data=BrookTroutTH,ages2use=-c(0,1))
summary(cr2)
plot(cr2)

## demonstration of ability to work with missing age classes
age <- c(  2, 3, 4, 5, 7, 9,12)
ct  <- c(100,92,83,71,56,35, 1)
cr3 <- chapmanRobson(age,ct,4:12)
summary(cr3)
plot(cr3)

}
\references{
Ogle, D.H. 2016. \href{https://fishr-core-team.github.io/fishR/pages/books.html#introductory-fisheries-analyses-with-r}{Introductory Fisheries Analyses with R}. Chapman & Hall/CRC, Boca Raton, FL.

Chapman, D.G. and D.S. Robson. 1960. The analysis of a catch curve. Biometrics. 16:354-368.

Hoenig, J.M. and W.D. Lawing, and N.A. Hoenig. 1983. Using mean age, mean length and median length data to estimate the total mortality rate. International Council for the Exploration of the Sea, CM 1983/D:23, Copenhagen.

Ricker, W.E. 1975. Computation and interpretation of biological statistics of fish populations. Technical Report Bulletin 191, Bulletin of the Fisheries Research Board of Canada. [Was (is?) from http://www.dfo-mpo.gc.ca/Library/1485.pdf.]

Robson, D.S. and D.G. Chapman. 1961. Catch curves and mortality rates. Transactions of the American Fisheries Society. 90:181-189.

Smith, M.W., A.Y. Then, C. Wor, G. Ralph, K.H. Pollock, and J.M. Hoenig. 2012. Recommendations for catch-curve analysis. North American Journal of Fisheries Management. 32:956-967.
}
\seealso{
See \code{\link[fishmethods]{agesurv}} in \pkg{fishmethods} for similar functionality. See \code{\link{catchCurve}} and \code{\link[fishmethods]{agesurvcl}} in \pkg{fishmethods} for alternative methods. See \code{\link{metaM}} for empirical methods to estimate natural mortality.
}
\author{
Derek H. Ogle, \email{DerekOgle51@gmail.com}
}
\keyword{htest}
\keyword{manip}
