% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/alkSummaries.R
\name{alkMeanVar}
\alias{alkMeanVar}
\title{Mean Values-at-age from an Age-Length Key}
\usage{
alkMeanVar(key, formula, data, len.n, method = c("BettoliMiranda",
  "QuinnDeriso"))
}
\arguments{
\item{key}{A numeric matrix that contains the age-length key.  See details.}

\item{formula}{A formula of the form \code{var~lencat+age} where \code{var} generically represents the variable to be summarized (e.g., length, weight, fecundity), \code{lencat} generically represents the variable that contains the length intervals, and \code{age} generically represents the variable that contains the assigned ages.}

\item{data}{A data.frame that minimally contains the length intervals, assessed ages, and the variable to be summarized (i.e., this should be the aged sample) as given in \code{formula}.}

\item{len.n}{A vector of sample sizes for each length interval in the \emph{complete sample} (i.e., all fish regardles of whether they were aged or not).}

\item{method}{A string that indicates which method of calculation should be used.  See details.}
}
\value{
A data.frame with as many rows as ages (columns) present in \code{key} and the following three variables:
\itemize{
  \item age The ages.
  \item mean The mean value at each age.
  \item sd,se The SD if \code{method="BettoliMiranda"} or SE of the mean if \code{method="QuinnDeriso"} for the value at each age.
 }
}
\description{
Computes the mean value-at-age in a larger sample based on an age-length-key created from a subsample of ages through a two-stage random sampling design.  The mean values could be mean length-, weight-, or fecundity-at-age, for example.  The methods of Bettoli and Miranda (2001) or Quinn and Deriso (1999) are used.  A standard deviation is computed for the Bettoli and Miranda (2001) method and standard error for the Quinn and Deriso (1999) method.  See the testing section notes.
}
\details{
The age-length key \code{key} must have length intervals as rows and ages as columns.  The row names of \code{key} (i.e., \code{rownames(key)}) must contain the mininum values of each length interval (e.g., if an interval is 100-109, then the corresponding row name must be 100).  The column names of \code{key} (i.e., \code{colnames(key)}) must contain the age values (e.g., the columns can NOT be named with \dQuote{age.1}, for example).

The length intervals in the rows of \code{key} must contain all of the length intervals present in the larger sample.  Thus, the length of \code{len.n} must, at least, equal the number of rows in \code{key}.  If this constraint is not met, then the function will stop with an error message.

Note that the function will stop with an error if the formula in \code{formula} does not meet the specific criteria outlined in the parameter list above.
}
\section{Testing}{
 The results of these functions have not yet been rigorously tested.  The Bettoli and Miranda (2001) results appear, at least, approximately correct when compared to the results from \code{\link{alkIndivAge}}.  The Quinn and Deriso (1999) results appear at least approximately correct for the mean values, but do not appear to be correct for the SE values.  Thus, a note is returned with the Quinn and Deriso (1999) results that the SE should not be trusted.
}

\section{IFAR Chapter}{
 5-Age-Length Key.
}
\examples{
## Get data with length measurements and some assigned ages
data(WR79)

## Example -- Even breaks for length categories
WR1 <- WR79
# add length intervals (width=5)
WR1$LCat <- lencat(WR1$len,w=5)
# get number of fish in each length interval in the entire sample
len.n <- xtabs(~LCat,data=WR1)
# isolate aged sample
WR1.age <- subset(WR1, !is.na(age))
# create age-length key
raw <- xtabs(~LCat+age,data=WR1.age)
( WR1.key <- prop.table(raw, margin=1) )

## use age-length key to estimate mean length-at-age of all fish
# Bettoli-Miranda method
alkMeanVar(WR1.key,len~LCat+age,WR1.age,len.n)

# Quinn-Deriso method
alkMeanVar(WR1.key,len~LCat+age,WR1.age,len.n,method="QuinnDeriso")

}
\author{
Derek H. Ogle, \email{derek@derekogle.com}
}
\references{
Ogle, D.H.  2016.  \href{http://derekogle.com/IFAR}{Introductory Fisheries Analyses with R}.  Chapman & Hall/CRC, Boca Raton, FL.

Bettoli, P. W. and Miranda, L. E.  2001. A cautionary note about estimating mean length at age with subsampled data. North American Journal of Fisheries Management, 21:425-428.
 
Quinn, T. J. and R. B. Deriso. 1999. Quantitative Fish Dynamics. Oxford University Press, New York, New York. 542 pages
}
\seealso{
See \code{\link{alkIndivAge}} and related functions for a completely different methodology.  See \code{\link{alkAgeDist}} for a related method of determining the proportion of fish at each age.  See the \pkg{ALKr} package.
}
\keyword{manip}

