#' @name SOPC.EFM
#' @title SOPC Estimation Function for Elliptical Factor Model
#' @description This function processes Elliptical Factor Model (EFM) data using the Sparse Online Principal Component (SOPC) method.
#' @param data A numeric matrix containing the data used in the SOPC analysis.
#' @param m An integer specifying the number of subsets or common factors.
#' @param p An integer specifying the number of variables in the data.
#' @param A A numeric matrix representing the true factor loadings.
#' @param D A numeric matrix representing the true uniquenesses.
#' @usage SOPC.EFM(data, m, p, A, D)
#' @return A list containing the following metrics:
#' \item{Aso}{Estimated factor loadings matrix.}
#' \item{Dso}{Estimated uniquenesses matrix.}
#' \item{MSEA}{Mean squared error of the estimated factor loadings (Aso) compared to the true loadings (A).}
#' \item{MSED}{Mean squared error of the estimated uniquenesses (Dso) compared to the true uniquenesses (D).}
#' \item{LSA}{Loss metric for the estimated factor loadings (Aso), indicating the relative error compared to the true loadings (A).}
#' \item{LSD}{Loss metric for the estimated uniquenesses (Dso), indicating the relative error compared to the true uniquenesses (D).}
#' \item{tauA}{Proportion of zero factor loadings in the estimated loadings matrix (Aso), representing the sparsity.}
#' @examples
#' \donttest{
#' library(matrixcalc)
#' library(MASS)
#' 
#' n <- 100
#' p <- 10
#' m <- 5
#' mu <- t(matrix(rep(runif(p, 0, 1000), n), p, n))
#' mu0 <- as.matrix(runif(m, 0))
#' sigma0 <- diag(runif(m, 1))
#' F_matrix <- matrix(mvrnorm(n, mu0, sigma0), nrow = n)
#' A <- matrix(runif(p * m, -1, 1), nrow = p)
#' r <- rnorm(n * p, 0, 1)
#' epsilon <- matrix(r, nrow = n)
#' D <- diag(as.vector(apply(epsilon, 2, function(x) sum(x^2))))
#' data <- mu + F_matrix %*% t(A) + epsilon
#' results <- SOPC.EFM(data, m, p, A, D)
#' print(results)
#' }
#' @export
#' @importFrom matrixcalc frobenius.norm
SOPC.EFM <- function(data, m, p, A, D) {
  # 使用完全限定名调用SOPC函数
  sopc_result <- SOPC::SOPC(data, m = m, gamma = 0.1, eta = 0.8)
  Aso <- sopc_result$Aso
  Dso <- sopc_result$Dso
  MSEA <- frobenius.norm(Aso - A)^2 / (p^2)
  MSED <- frobenius.norm(Dso - D)^2 / (p^2)
  LSA <- frobenius.norm(Aso - A)^2 / frobenius.norm(A)^2
  LSD <- frobenius.norm(Dso - D)^2 / frobenius.norm(D)^2
  tauA <- as.vector(table(Aso == 0) / (p * m))[2]
  return(list(Aso = Aso,
              Dso = Dso,
              MSEA = MSEA,
              MSED = MSED,
              LSA = LSA,
              LSD = LSD,
              tauA = tauA))
}