% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/f_interface_clvdata.R
\name{clvdata}
\alias{clvdata}
\title{Create an object for transactional data required to estimate CLV}
\usage{
clvdata(
  data.transactions,
  date.format,
  time.unit,
  estimation.split = NULL,
  name.id = "Id",
  name.date = "Date",
  name.price = "Price"
)
}
\arguments{
\item{data.transactions}{Transaction data as \code{data.frame} or \code{data.table}. See details.}

\item{date.format}{Character string that indicates the format of the date variable in the data used. See details.}

\item{time.unit}{What time unit defines a period. May be abbreviated, capitalization is ignored. See details.}

\item{estimation.split}{Indicates the length of the estimation period. See details.}

\item{name.id}{Column name of the customer id in \code{data.transaction}.}

\item{name.date}{Column name of the transaction date in \code{data.transaction}.}

\item{name.price}{Column name of price in \code{data.transaction}. NULL if no spending data is present.}
}
\value{
An object of class \code{clv.data}.
See the class definition \linkS4class{clv.data}
for more details about the returned object.

The function \code{summary} can be used to obtain and print a summary of the data.
The generic accessor function \code{nobs} is available to read out the number of customers.
}
\description{
Creates a data object that contains the prepared transaction data and that is used as input for
model fitting. The transaction data may be split in an estimation and holdout sample if desired.
The model then will only be fit on the estimation sample.

If covariates should be used when fitting a model, covariate data can be added
to an object returned from this function.
}
\details{
\code{data.transactions} A \code{data.frame} or \code{data.table} with customers' purchase history.
Every transaction record consists of a purchase date and a customer id.
Optionally, the price of the transaction may be included to also allow for prediction
of future customer spending.

\code{time.unit} The definition of a single period. Currently available are \code{"hours"}, \code{"days"}, \code{"weeks"}, and \code{"years"}.
May be abbreviated.

\code{date.format} A single format to use when parsing any date that is given as character input. This includes
the dates given in \code{data.transaction}, \code{estimation.split}, or as an input to any other function at
a later point, such as \code{prediction.end} in \code{predict}.
The function \code{\link[lubridate]{parse_date_time}} of package \code{lubridate} is used to parse inputs
and hence all formats it accepts in argument \code{orders} can be used. For example, a date of format "year-month-day"
(i.e., "2010-06-17") is indicated with \code{"ymd"}. Other combinations such as \code{"dmy"}, \code{"dym"},
\code{"ymd HMS"}, or \code{"HMS dmy"} are possible as well.

\code{estimation.split} May be specified as either the number of periods since the first transaction or the timepoint
(either as character, Date, or POSIXct) at which the estimation period ends. The indicated timepoint itself will be part of the estimation sample.
If no value is provided or set to \code{NULL}, the whole dataset will used for fitting the model (no holdout sample).

\subsection{Aggregation of Transactions}{

Multiple transactions by the same customer that occur on the minimally representable temporal resolution are aggregated to a
single transaction with their spending summed. For time units \code{days} and any other coarser \code{Date}-based
time units (i.e. \code{weeks}, \code{years}), this means that transactions on the same day are combined.
When using finer time units such as \code{hours} which are based on \code{POSIXct}, transactions on the same second are aggregated.

For the definition of repeat-purchases, combined transactions are viewed as a single transaction. Hence, repeat-transactions
are determined from the aggregated transactions.
}
}
\examples{

data("cdnow")

# create clv data object with weekly periods
#    and no splitting
clv.data.cdnow <- clvdata(data.transactions = cdnow,
                          date.format="ymd",
                          time.unit = "weeks")

# same but split after 37 periods
clv.data.cdnow <- clvdata(data.transactions = cdnow,
                          date.format="ymd",
                          time.unit = "w",
                          estimation.split = 37)

# same but estimation end on the 15th Oct 1997
clv.data.cdnow <- clvdata(data.transactions = cdnow,
                          date.format="ymd",
                          time.unit = "w",
                          estimation.split = "1997-10-15")

\donttest{

# summary of the transaction data
summary(clv.data.cdnow)

# plot the transaction data
plot(clv.data.cdnow)

# create data with the weekly periods defined to
#   start on Mondays

\dontrun{
# set start of week to Monday
oldopts <- options("lubridate.week.start"=1)

# create clv.data while Monday is the beginning of the week
clv.data.cdnow <- clvdata(data.transactions = cdnow,
                          date.format="ymd",
                          time.unit = "weeks")

# Dynamic covariates now have to be supplied for every Monday

# set week start to what it was before
options(oldopts)
}

}


}
\seealso{
\code{\link[CLVTools:SetStaticCovariates]{SetStaticCovariates}} to add static covariates

\code{\link[CLVTools:SetDynamicCovariates]{SetDynamicCovariates}} for how to add dynamic covariates

\code{\link[CLVTools:plot.clv.data]{plot}} to plot the repeat transactions

\code{\link[CLVTools:summary.clv.data]{summary}} to summarize the transaction data

\code{\link[CLVTools:pnbd]{pnbd}} to fit Pareto/NBD models on a \code{clv.data} object
}
