# test-berman.R 20171001
context("berman")

# calculate properties for all available minerals at Tr,Pr
dir <- system.file("extdata/Berman/", package="CHNOSZ")
Ber88 <- read.csv(paste0(dir, "/Ber88.csv"), as.is=TRUE)
Ber90 <- read.csv(paste0(dir, "/Ber90.csv"), as.is=TRUE)
SHD91 <- read.csv(paste0(dir, "/SHD91.csv"), as.is=TRUE)
ZS92 <- read.csv(paste0(dir, "/ZS92.csv"), as.is=TRUE)
JUN92 <- read.csv(paste0(dir, "/JUN92.csv"), as.is=TRUE)
DS10 <- read.csv(paste0(dir, "/DS10.csv"), as.is=TRUE)
FDM14 <- read.csv(paste0(dir, "/FDM+14.csv"), as.is=TRUE)
BDat17 <- read.csv(paste0(dir, "/BDat17.csv"), as.is=TRUE)
# assemble the files and remove duplicates (keep the latest)
dat <- rbind(BDat17, FDM14, DS10, ZS92, SHD91, Ber90, Ber88)
dat <- dat[!duplicated(dat$name), ]
mineral <- unique(dat$name)
prop_Berman <- NULL

test_that("properties of all minerals are computed without warnings", {
  # running this without error means that:
  # - formulas for the minerals are found in thermo$obigt
  # - warnings are produced for mineral(s) with GfPrTr(calc) >= 1000 J/cal different from GfPrTr(table)
  expect_warning(properties <- lapply(mineral, berman, check.G=TRUE),
                 "fluortremolite", all=TRUE)
  # save the results so we can use them in the next tests
  assign("prop_Berman", properties, inherits=TRUE)
})

# assemble a data frame for Berman properties
prop_Berman <- do.call(rbind, prop_Berman)
# find the mineral data using Helgeson formulation
icr <- suppressMessages(info(mineral, "cr"))
# all of these except rutile (Robie et al., 1979) reference Helgeson et al., 1978
# NOTE: with check.it = TRUE (the default), this calculates Cp from the tabulated Maier-Kelley parameters
prop_Helgeson <- suppressMessages(info(icr))

# now we can compare Berman and Helgeson G, H, S, Cp, V
# minerals with missing properties are not matched here
# (i.e. fluorphlogopite, fluortremolite, glaucophane, and pyrope: no G and H in prop_Helgeson data)

test_that("Berman and Helgeson tabulated properties have large differences for few minerals", {
  # which minerals differ in DGf by more than 4 kcal/mol?
  idiffG <- which(abs(prop_Berman$G - prop_Helgeson$G) > 4000)
  expect_match(mineral[idiffG],
               "paragonite|anthophyllite|antigorite|Ca-Al-pyroxene|lawsonite|margarite|merwinite")

  # which minerals differ in DHf by more than 4 kcal/mol?
  idiffH <- which(abs(prop_Berman$H - prop_Helgeson$H) > 4000)
  # we get the above, plus phlogopite and clinozoisite:
  expect_match(mineral[idiffH],
               "paragonite|anthophyllite|antigorite|Ca-Al-pyroxene|lawsonite|margarite|merwinite|phlogopite|clinozoisite")

  # which minerals differ in S by more than 4 cal/K/mol?
  idiffS <- which(abs(prop_Berman$S - prop_Helgeson$S) > 4)
  expect_match(mineral[idiffS], "albite|annite|almandine|fluortremolite|andradite|grunerite")

  # which minerals differ in Cp by more than 4 cal/K/mol?
  idiffCp <- which(abs(prop_Berman$Cp - prop_Helgeson$Cp) > 4)
  expect_match(mineral[idiffCp], "glaucophane|antigorite|cristobalite,beta|K-feldspar|fluortremolite|grunerite")

  # which minerals differ in V by more than 1 cm^3/mol?
  idiffV <- which(abs(prop_Berman$V - prop_Helgeson$V) > 1)
  expect_match(mineral[idiffV], "glaucophane|anthophyllite|antigorite|chrysotile|merwinite|grunerite")
})

test_that("high-T,P calculated properties are similar to precalculated ones", {
  # Reference values for G were taken from the spreadsheet Berman_Gibbs_Free_Energies.xlsx
  #   (http://www.dewcommunity.org/uploads/4/1/7/6/41765907/sunday_afternoon_sessions__1_.zip accessed on 2017-10-03)
  T <- c(100, 100, 1000, 1000)
  P <- c(5000, 50000, 5000, 50000)

  # anadalusite: an uncomplicated mineral (no transitions)
  And_G <- c(-579368, -524987, -632421, -576834)
  And <- subcrt("andalusite", "cr_Berman", T=T, P=P)$out[[1]]
  expect_maxdiff(And$G, And_G, 7.5)

  # quartz: a mineral with polymorphic transitions
  aQz_G <- c(-202800, -179757, -223864, -200109)
  aQz <- subcrt("quartz", "cr_Berman", T=T, P=P)$out[[1]]
  expect_maxdiff(aQz$G[-2], aQz_G[-2], 1.2)
  # here, the high-P, low-T point suffers
  expect_maxdiff(aQz$G[2], aQz_G[2], 1250)

  # K-feldspar: this one has disordering effects
  Kfs_G <- c(-888115, -776324, -988950, -874777)
  Kfs <- subcrt("K-feldspar", "cr_Berman", T=T, P=P)$out[[1]]
  expect_maxdiff(Kfs$G[1:2], Kfs_G[1:2], 5)
  # we are less consistent with the reference values at high T
  expect_maxdiff(Kfs$G[3:4], Kfs_G[3:4], 350)
})
