#' Climatic vulnerability
#'
#' Calculates the climatic vulnerability of a species using a \code{cnfa} and
#' \code{departure} object.
#'
#' @param cnfa Object of class \code{cnfa}
#' @param dep Object of class \code{departure}
#' @param method character. What type of mean should be used to combine sensitivity
#'   and exposure. Choices are "arithmetic" and "geometric"
#' @param w numeric. Optional vector of length two specifying the relative weights of
#'   sensitivity and exposure. See Details
#' @param parallel logical. If \code{TRUE} then multiple cores are utilized
#' @param n numeric. Number of cores to use for calculation
#' @param filename character. Output filename (optional)
#' @param ... Additional arguments for file writing as for \code{\link[raster]{writeRaster}}
#'
#' @details
#' The values of the vulnerability raster are calculated by combining the sensitivity
#' \eqn{\sigma} and the exposure \eqn{\epsilon}. If \code{method = "arithmetic"},
#' they will be combined as
#'
#'   \eqn{\nu = (w_1\sigma + w_2\epsilon) / (\sum_i w_i).}
#'
#' If \code{method = "geometric"}, they will be combined as
#'
#'   \eqn{\nu = \sqrt(\sigma * \epsilon).}
#'
#' @examples
#' mod1 <- cnfa(x = climdat.hist, s.dat = ABPR, field = "CODE")
#' dep <- departure(x = climdat.hist, y = climdat.fut, s.dat = ABPR)
#' vuln <- vulnerability(cnfa = mod1, dep = dep)
#'
#' @return Returns an S4 object of class \code{vulnerability} with the following slots:
#' \describe{
#'   \item{call}{Original function call}
#'   \item{vf}{Vulnerability factor. Vector of length p that describes the amount of
#'    vulnerability in each climate variable}
#'   \item{vulnerability}{Magnitude of the vulnerability factor}
#'   \item{ras}{RasterLayer of climate vulnerability}
#'   \item{weights}{Raster layer of weights used for departure calculation}
#' }
#'
#' @seealso \code{\link{departure}}
#'
#' @export

setGeneric("vulnerability", function(cnfa, dep, method = "geometric", w, parallel = FALSE, n = 1, filename = "", ...) {
  standardGeneric("vulnerability")
})

#' @rdname vulnerability
setMethod("vulnerability",
          signature(cnfa = "cnfa", dep = "departure"),
          function(cnfa, dep, method = "geometric", w, parallel = FALSE, n = 1, filename = "", ...) {

            call <- sys.calls()[[1]]

            co <- cnfa@co
            p <- cnfa@p.spec
            d <- dep@df
            ras <- dep@ras

            v <- sqrt(as.numeric(diag(d) %*% abs(co) %*% p))
            names(v) <- names(d)
            V <- sqrt(as.numeric(t(v) %*% v))

            filename <- trim(filename)
            if (!canProcessInMemory(ras) && filename == '') {
              filename <- rasterTmpFile()
            }

            s.map <- sensitivity_map(cnfa, parallel = parallel, n = n)
            e.map <- exposure_map(dep, parallel = parallel, n = n)
            if (method == "arithmetic") {
              if (missing(w)) w <- c(1, 1)
              if (length(w) != 2) warning("more than two weights supplied; ignoring weights argument.")
              vuln.ras <- (w[1] * s.map + w[2] * e.map) / sum(w)
            } else if (method == "geometric") {
              vuln.ras <- sqrt(s.map * e.map)
            }
            if (filename != '') writeRaster(vuln.ras, filename = filename, ...)

            vuln <- methods::new("vulnerability", call = call, vf = v, vulnerability = V, ras = vuln.ras, weights = dep@weights)
            return(vuln)
          }
)


