% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sart.R
\name{sart}
\alias{sart}
\title{Estimating Tobit models with social interactions}
\usage{
sart(
  formula,
  Glist,
  starting = NULL,
  Ey0 = NULL,
  optimizer = "fastlbfgs",
  npl.ctr = list(),
  opt.ctr = list(),
  cov = TRUE,
  cinfo = TRUE,
  data
)
}
\arguments{
\item{formula}{a class object \link[stats]{formula}: a symbolic description of the model. \code{formula} must be as, for example, \code{y ~ x1 + x2 + gx1 + gx2}
where \code{y} is the endogenous vector and \code{x1}, \code{x2}, \code{gx1} and \code{gx2} are control variables, which can include contextual variables, i.e. averages among the peers.
Peer averages can be computed using the function \code{\link{peer.avg}}.}

\item{Glist}{The network matrix. For networks consisting of multiple subnets, \code{Glist} can be a list of subnets with the \code{m}-th element being an \code{ns*ns} adjacency matrix, where \code{ns} is the number of nodes in the \code{m}-th subnet.}

\item{starting}{(optional) a starting value for \eqn{\theta = (\lambda, \Gamma, \sigma)} (see the model specification in details).}

\item{Ey0}{(optional) a starting value for \eqn{E(y)}.}

\item{optimizer}{is either \code{fastlbfgs} (L-BFGS optimization method of the package \pkg{RcppNumerical}), \code{nlm} (referring to the function \link[stats]{nlm}), or \code{optim} (referring to the function \link[stats]{optim}).
Arguments for these functions such as, \code{control} and \code{method} can be set via the argument \code{opt.ctr}.}

\item{npl.ctr}{a list of controls for the NPL method (see details of the function \code{\link{cdnet}}).}

\item{opt.ctr}{a list of arguments to be passed in \code{optim_lbfgs} of the package \pkg{RcppNumerical}, \link[stats]{nlm} or \link[stats]{optim} (the solver set in \code{optimizer}), such as \code{maxit}, \code{eps_f}, \code{eps_g}, \code{control}, \code{method}, etc.}

\item{cov}{a Boolean indicating if the covariance must be computed.}

\item{cinfo}{a Boolean indicating whether information is complete (\code{cinfo = TRUE}) or incomplete (\code{cinfo = FALSE}). In the case of incomplete information, the model is defined under rational expectations.}

\item{data}{an optional data frame, list or environment (or object coercible by \link[base]{as.data.frame} to a data frame) containing the variables
in the model. If not found in data, the variables are taken from \code{environment(formula)}, typically the environment from which \code{sart} is called.}
}
\value{
A list consisting of:
\item{info}{a list of general information on the model.}
\item{estimate}{the Maximum Likelihood (ML) estimator.}
\item{Ey}{\eqn{E(y)}, the expectation of y.}
\item{GEy}{the average of \eqn{E(y)} friends.}
\item{cov}{a list including (if \code{cov == TRUE}) covariance matrices.}
\item{details}{outputs as returned by the optimizer.}
}
\description{
\code{sart} estimates Tobit models with social interactions (Xu and Lee, 2015).
}
\details{
For a complete information model, the outcome \eqn{y_i} is defined as:
\deqn{\begin{cases}y_i^{\ast} = \lambda \bar{y}_i + \mathbf{z}_i'\Gamma + \epsilon_i, \\ y_i = \max(0, y_i^{\ast}),\end{cases}}
where \eqn{\bar{y}_i} is the average of \eqn{y} among peers,
\eqn{\mathbf{z}_i} is a vector of control variables,
and \eqn{\epsilon_i \sim N(0, \sigma^2)}.
In the case of incomplete information modelswith rational expectations, \eqn{y_i} is defined as:
\deqn{\begin{cases}y_i^{\ast} = \lambda E(\bar{y}_i) + \mathbf{z}_i'\Gamma + \epsilon_i, \\ y_i = \max(0, y_i^{\ast}).\end{cases}}
}
\examples{
\donttest{
# Groups' size
set.seed(123)
M      <- 5 # Number of sub-groups
nvec   <- round(runif(M, 100, 200))
n      <- sum(nvec)

# Parameters
lambda <- 0.4
Gamma  <- c(2, -1.9, 0.8, 1.5, -1.2)
sigma  <- 1.5
theta  <- c(lambda, Gamma, sigma)

# X
X      <- cbind(rnorm(n, 1, 1), rexp(n, 0.4))

# Network
G      <- list()

for (m in 1:M) {
  nm           <- nvec[m]
  Gm           <- matrix(0, nm, nm)
  max_d        <- 30
  for (i in 1:nm) {
    tmp        <- sample((1:nm)[-i], sample(0:max_d, 1))
    Gm[i, tmp] <- 1
  }
  rs           <- rowSums(Gm); rs[rs == 0] <- 1
  Gm           <- Gm/rs
  G[[m]]       <- Gm
}

# Data
data   <- data.frame(X, peer.avg(G, cbind(x1 = X[,1], x2 =  X[,2])))
colnames(data) <- c("x1", "x2", "gx1", "gx2")

## Complete information game
ytmp    <- simsart(formula = ~ x1 + x2 + gx1 + gx2, Glist = G, theta = theta, 
                   data = data, cinfo = TRUE)
data$yc <- ytmp$y

## Incomplete information game
ytmp    <- simsart(formula = ~ x1 + x2 + gx1 + gx2, Glist = G, theta = theta, 
                   data = data, cinfo = FALSE)
data$yi <- ytmp$y

# Complete information estimation for yc
outc1   <- sart(formula = yc ~ x1 + x2 + gx1 + gx2, optimizer = "nlm",
                Glist = G, data = data, cinfo = TRUE)
summary(outc1)

# Complete information estimation for yi
outc1   <- sart(formula = yi ~ x1 + x2 + gx1 + gx2, optimizer = "nlm",
                Glist = G, data = data, cinfo = TRUE)
summary(outc1)

# Incomplete information estimation for yc
outi1   <- sart(formula = yc ~ x1 + x2 + gx1 + gx2, optimizer = "nlm",
                Glist = G, data = data, cinfo = FALSE)
summary(outi1)

# Incomplete information estimation for yi
outi1   <- sart(formula = yi ~ x1 + x2 + gx1 + gx2, optimizer = "nlm",
                Glist = G, data = data, cinfo = FALSE)
summary(outi1)
}
}
\references{
Xu, X., & Lee, L. F. (2015). Maximum likelihood estimation of a spatial autoregressive Tobit model. \emph{Journal of Econometrics}, 188(1), 264-280, \doi{10.1016/j.jeconom.2015.05.004}.
}
\seealso{
\code{\link{sar}}, \code{\link{cdnet}}, \code{\link{simsart}}.
}
