
#' Summary method for Bayesian VARs
#'
#' Retrieves several outputs of interest, including the median coefficient
#' matrix, the median variance-covariance matrix, and the log-likelihood.
#' Separate summary methods exist for impulse responses and forecasts.
#'
#' @param object A \code{bvar} object, obtained from \code{\link{bvar}}.
#' @param ... Not used.
#'
#' @return Returns a list of class \code{bvar_summary} with elements that can
#' can be accessed individually:
#' \itemize{
#'   \item \code{bvar} - the \code{bvar} object provided.
#'   \item \code{coef} - coefficient values from \code{\link{coef.bvar}}.
#'   \item \code{vcov} - VCOV values from \code{\link{vcov.bvar}}.
#'   \item \code{logLik} - the log-likelihood from \code{\link[stats]{logLik}}.
#' }
#'
#' @seealso \code{\link{bvar}};
#' \code{\link{predict.bvar}}; \code{\link{irf.bvar}}
#'
#' @keywords BVAR analysis
#'
#' @export
#'
#' @examples
#' \donttest{
#' # Access a subset of the fred_qd dataset
#' data <- fred_qd[, c("CPIAUCSL", "UNRATE", "FEDFUNDS")]
#' # Transform it to be stationary
#' data <- fred_transform(data, codes = c(5, 5, 1), lag = 4)
#'
#' # Estimate a BVAR using one lag, default settings and very few draws
#' x <- bvar(data, lags = 1, n_draw = 1000L, n_burn = 200L, verbose = FALSE)
#'
#' summary(x)
#' }
summary.bvar <- function(object, ...) {

  out <- structure(list(
    "bvar" = object, "coef" = coef.bvar(object), "vcov" = vcov.bvar(object),
    "logLik" = logLik.bvar(object)), class = "bvar_summary")

  return(out)
}


#' @export
print.bvar_summary <- function(x, ...) {

  print(x[["bvar"]])

  cat("\n"); print.bvar_coefs(x[["coef"]])
  cat("\n"); print.bvar_vcovs(x[["vcov"]])
  cat("\n"); cat("Log-Likelihood:", x[["logLik"]], "\n")

  return(invisible(x))
}
