% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/computeAGB.R
\name{computeAGB}
\alias{computeAGB}
\title{Computing tree above-ground biomass (AGB)}
\usage{
computeAGB(D, WD, H = NULL, coord = NULL, Dlim = NULL, useCache = FALSE)
}
\arguments{
\item{D}{Tree diameter (in cm), either a vector or a single value.}

\item{WD}{Wood density (in g/cm3), either a vector or a single value. If not available, see \code{\link[=getWoodDensity]{getWoodDensity()}}.}

\item{H}{(optional) Tree height (H in m), either a vector or a single value. If not available, see \code{\link[=retrieveH]{retrieveH()}}
and \code{\link[=modelHD]{modelHD()}}. Compulsory if the coordinates \code{coord} are not given.}

\item{coord}{(optional) Coordinates of the site(s), either a vector giving a single site
(e.g. c(longitude, latitude)) or a matrix/dataframe with two columns (e.g. cbind(longitude, latitude)).
The coordinates are used to account for variation in height-diameter relationship thanks to an environmental
proxy (parameter E in Chave et al. 2014). Compulsory if tree heights \code{H} are not given.}

\item{Dlim}{(optional) Minimum diameter (in cm) for which aboveground biomass should be calculated
(all diameter below \code{Dlim} will have a 0 value in the output).}

\item{useCache}{logical. Whether or not use a cache to avoid downloading multiple time the same files.
Strongly recommended to reduce computing time (but FALSE by default due to CRAN policy).}
}
\value{
The function returns the AGB in Mg (or ton) as a single value or a vector.
}
\description{
This function uses Chave et al. 2014's pantropical models to estimate the above-ground biomass of tropical trees.
}
\details{
This function uses two different ways of computing the above-ground biomass of a tree:
\enumerate{
\item If tree height data are available, the AGB is computed thanks to the following equation (Eq. 4 in Chave et al., 2014):
\deqn{AGB = 0.0673 * (WD * H * D^2)^0.976}
\item If no tree height data is available, the AGB is computed thanks to the site coordinates with the following equation, slightly modified from Eq. 7 in Chave et al., 2014 (see Réjou-Méchain et al. 2017):
\deqn{AGB = exp(-2.024- 0.896*E + 0.920*log(WD) + 2.795*log(D) - 0.0461*(log(D)^2))} where \code{E} is a measure of environmental stress estimated from the site coordinates (\code{coord}).
}
}
\examples{
# Create variables
D <- 10:99
WD <- runif(length(D), min = 0.1, max = 1)
H <- D^(2 / 3)

# If you have height data
AGB <- computeAGB(D, WD, H)

# If you do not have height data and a single site
lat <- 4.08
long <- -52.68
coord <- c(long, lat)
\donttest{
AGB <- computeAGB(D, WD, coord = coord)
}

# If you do not have height data and several sites (here three)
lat <- c(rep(4.08, 30), rep(3.98, 30), rep(4.12, 30))
long <- c(rep(-52.68, 30), rep(-53.12, 30), rep(-53.29, 30))
coord <- cbind(long, lat)
\donttest{
AGB <- computeAGB(D, WD, coord = coord)
}

}
\references{
Chave et al. (2014) \emph{Improved allometric models to estimate the aboveground biomass of tropical trees},
Global Change Biology, 20 (10), 3177-3190
}
\seealso{
\code{\link[=computeE]{computeE()}}
}
\author{
Maxime REJOU-MECHAIN, Ariane TANGUY, Arthur PERE
}
\keyword{AGB}
\keyword{above-ground}
\keyword{allometry}
\keyword{biomass}
\keyword{carbon}
\keyword{forest}
