% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plotRateThroughTime.R
\name{plotRateThroughTime}
\alias{plotRateThroughTime}
\title{Plot rates through time}
\usage{
plotRateThroughTime(ephy, useMedian = TRUE, intervals = seq(from = 0, to =
  1, by = 0.01), ratetype = "auto", nBins = 100, smooth = FALSE,
  smoothParam = 0.2, opacity = 0.01, intervalCol = "blue",
  avgCol = "red", start.time = NULL, end.time = NULL, node = NULL,
  nodetype = "include", plot = TRUE, cex.axis = 1, cex.lab = 1.3,
  lwd = 3, xline = 3.5, yline = 3.5, mar = c(6, 6, 1, 1),
  xticks = NULL, yticks = NULL, xlim = "auto", ylim = "auto",
  add = FALSE, axis.labels = TRUE)
}
\arguments{
\item{ephy}{Object of class \code{bammdata} or \code{bamm-ratematrix}.}

\item{useMedian}{A logical: will plot median if \code{TRUE}, mean if
\code{FALSE}.}

\item{intervals}{If \code{NULL}, no intervals will be plotted, otherwise a
vector of quantiles must be supplied (these will define shaded
polygons).}

\item{ratetype}{If 'auto', defaults to speciation (for diversification) or
beta (for traits). Can alternatively specify 'netdiv' or 'extinction'.}

\item{nBins}{Number of time slices used to generate rates through time.}

\item{smooth}{A logical: whether or not to apply loess smoothing.}

\item{smoothParam}{Loess smoothing parameter, ignored if
\code{smooth = FALSE}.}

\item{opacity}{Opacity of color for interval polygons.}

\item{intervalCol}{Color for interval polygons.}

\item{avgCol}{Color for mean/median line (line will not be plotted if
\code{avgCol = NULL}).}

\item{start.time}{Start time (in units before present). If \code{NULL},
starts at root.}

\item{end.time}{End time (in units before present). If \code{NULL}, ends
at present.}

\item{node}{If supplied, the clade descended from this node will be used
or ignored, depending on \code{nodetype}.}

\item{nodetype}{If 'include', rates will be plotted only for the clade
descended from \code{node}. If 'exclude', the clade descended from
\code{node} will be left out of the calculation of rates.}

\item{plot}{A logical: if \code{TRUE}, a plot will be returned, if
\code{FALSE}, the data for the plot will be returned.}

\item{cex.axis}{Size of axis tick labels.}

\item{cex.lab}{Size of axis labels.}

\item{lwd}{Line width of the average rate.}

\item{xline}{Margin line for placing x-axis label.}

\item{yline}{Margin line for placing y-axis label.}

\item{mar}{Passed to \code{par()} to set plot margins.}

\item{xticks}{Number of ticks on the x-axis, automatically inferred if
\code{NULL}.}

\item{yticks}{Number of ticks on the y-axis, automatically inferred if
\code{NULL}.}

\item{xlim}{Vector of length 2 with min and max times for x axis. X axis
is time since present, so if plotting till the present,
\code{xlim[2]==0}. Can also be 'auto'.}

\item{ylim}{Vector of length 2 with min and max rates for y axis. Can also
be 'auto'.}

\item{add}{A logical: should rates be added to an existing plot.}

\item{axis.labels}{A logical: if \code{TRUE}, axis labels will be plotted.}
}
\value{
If \code{plot = FALSE}, then a list is returned with the following
    components:
    \itemize{
        \item{poly} {A list of matrices, where each matrix contains the
            coordinates that define each overlapping confidence interval
            polygon.}
        \item{avg} {A vector of y-coordinates for mean or median rates
            used to plot the average rates line.}
        \item{times} {A vector of time values, used as x-coordinates in
            this plot function.}
    }
}
\description{
Generates a plot of diversification or phenotypic rate through
    time with confidence intervals.
}
\details{
If the input \code{ephy} object has been generated by
    \code{\link{getEventData}} and is of class \code{bammdata}, then
    \code{start.time}, \code{end.time}, \code{node}, and \code{nodetype}
    can be specified. If the input \code{ephy} object has been generated
    by \code{\link{getRateThroughTimeMatrix}} and is of class
    \code{bamm-ratematrix}, then those arguments cannot be specified
    because they are needed to generate the rate matrix, which in this
    case has already happened. 

    The user has complete control of the plotting of the confidence
    intervals. Confidence intervals will not be plotted at all if
    \code{intervals=NULL}. If a single confidence interval polygon is
    desired, rather than overlapping polygons, then \code{intervals} can
    specify the confidence interval bounds, and \code{opacity} should be
    set to 1 (see examples).

    If working with a large dataset, we recommend first creating a
    \code{bamm-ratematrix} object with
    \code{\link{getRateThroughTimeMatrix}} and then using that object as
    input for \code{plotRateThroughTime}. This way, the computation of
    rates has already happened and will not slow the plotting function
    down, making it easier to adjust plotting parameters.
}
\examples{
\dontrun{
data(events.whales)
data(whales)
ephy <- getEventData(whales,events.whales)

# Simple plot of rates through time with default settings
plotRateThroughTime(ephy)

# Plot two processes separately with 90\% CI and loess smoothing
plotRateThroughTime(ephy, intervals=seq(from=0,0.9,by=0.01), smooth=TRUE,
                    node=141, nodetype='exclude', ylim=c(0,1.2))

plotRateThroughTime(ephy, intervals=seq(from=0,0.9,by=0.01), smooth=TRUE, 
                    node=141, nodetype='include', add=TRUE,
                    intervalCol='orange')

legend('topleft', legend=c('Dolphins','Whales'), col='red',
    fill=c('orange','blue'), border=FALSE, lty=1, lwd=2, merge=TRUE,
           seg.len=0.6)

# Same plot, but from bamm-ratematrix objects
rmat1 <- getRateThroughTimeMatrix(ephy, node=141, nodetype='exclude')
rmat2 <- getRateThroughTimeMatrix(ephy, node=141, nodetype='include')

plotRateThroughTime(rmat1, intervals=seq(from=0,0.9,by=0.01), 
    smooth=TRUE, ylim=c(0,1.2))

plotRateThroughTime(rmat2, intervals=seq(from=0,0.9,by=0.01), 
    smooth=TRUE, add=TRUE, intervalCol='orange')

# To plot the mean rate without the confidence envelope
plotRateThroughTime(ephy,useMedian=FALSE, intervals=NULL)

# To plot the mean rate, with a single 95\% confidence envelope, grayscale
plotRateThroughTime(ephy,useMedian=FALSE, intervals=c(0.05,0.95),
    intervalCol='gray70', avgCol='black', opacity=1)

# To not plot, but instead return the plotting data generated in this
# function, we can make plot=FALSE
plotRateThroughTime(ephy, plot=FALSE)}
}
\author{
Pascal Title
}
\seealso{
See \code{\link{getEventData}} and
    \code{\link{getRateThroughTimeMatrix}} to generate input data.
}

