# Tests for art
#
# Author: mjskay
###############################################################################

library(testthat)
library(ARTool)

context("art")

test_that("art does not allow factors as responses", {
    expect_error(art(y ~ a, data=data.frame(y=factor(c(1,2,3)), a=factor(c(1,2,3)))), 'Reponse term must be numeric, ordered factor, or logical \\(it was factor\\)')
})

test_that("art allows numeric responses", {
    m = art(y ~ a, data=data.frame(y=c(rep(1, 10), rep(2, 10)), a=factor(rep(1:2, 10))))

    expect_equal(anova(m)$F, 0)
    expect_equal(nrow(anova(m, response="aligned")), 0)
})

test_that("art allows logical responses", {
    m = art(y ~ a, data=data.frame(y=c(rep(TRUE, 10), rep(FALSE, 10)), a=factor(rep(1:2, 10))))

    expect_equal(anova(m)$F, 0)
    expect_equal(nrow(anova(m, response="aligned")), 0)
})

test_that("art allows ordinal responses", {
    m = art(y ~ a, data=data.frame(y=ordered(c(rep(1, 10), rep(2, 10))), a=factor(rep(1:2, 10))))

    expect_equal(anova(m)$F, 0)
    expect_equal(nrow(anova(m, response="aligned")), 0)
})

test_that("art does not allow fixed effects that are numeric", {
    expect_error(art(y ~ a, data=data.frame(y=c(1,2,3), a=c(1,2,3))), 'All fixed effect terms must be factors or logical \\(e.g. not numeric\\).')
})

test_that("art allows logical fixed effect terms", {
    m = art(y ~ a, data=data.frame(y=c(rep(1, 10), rep(2, 10)), a=rep(c(FALSE,TRUE), 10)))

    expect_equal(anova(m)$F, 0)
    expect_equal(nrow(anova(m, response="aligned")), 0)
})

test_that("art does not allow incomplete cases in fixed effects", {
    expect_error(art(y ~ a, data=data.frame(y=c(NA,2,3), a=factor(c(1,2,3)))), "Aligned Rank Transform cannot be performed when fixed effects have missing data \\(NAs\\).")
    expect_error(art(y ~ a, data=data.frame(y=c(1,2,3), a=factor(c(1,NA,3)))), "Aligned Rank Transform cannot be performed when fixed effects have missing data \\(NAs\\).")
})

test_that("art allows models with only one fixed effect", {
    df = data.frame(y=1:20, a=factor(rep(c(1,2),10)))
    m = art(y ~ a, data=df)

    #neither of the following anovas should throw an error
    expect_equal(anova(m)$F, 0.13636363636363636)
    expect_equal(nrow(anova(m, response="aligned")), 0)	#only one fixed effect => aligned anova table has no rows
})

test_that("art allows models with missing data in the grouping terms", {
    df = data.frame(y=1:20, a=factor(rep(c(1,2),10)), b=factor(rep(c(1,2,3,NA,5),4)))
    m = art(y ~ a + (a|b), data=df)

    #neither of the following anovas should throw an error
    expect_equal(anova(m)$F, 0.6862745)
    expect_equal(nrow(anova(m, response="aligned")), 0)
})

test_that("art correctly interprets formulas with expressions on the right-hand side", {
    df = data.frame(y=rep(c(1,2,0,3),5), a=c(1,2), fa=factor(c(1,2)))
    m1 = art(y ~ factor(a), data=df)
    m2 = art(y ~ fa, data=df)

    #neither of the following anovas should throw an error
    expect_equal(m1$aligned.ranks$`factor(a)`, m2$aligned.ranks$fa)
})

test_that("art correctly interprets formulas with expressions on the left-hand side", {
    df = data.frame(y=rep(c("1","2","0","3"),5), ny=rep(c(1,2,0,3),5), a=factor(c(1,2)), stringsAsFactors=FALSE)
    m1 = art(as.numeric(y) ~ a, data=df)
    m2 = art(ny ~ a, data=df)

    #neither of the following anovas should throw an error
    expect_equal(m1$aligned.ranks$`factor(a)`, m2$aligned.ranks$fa)
})

test_that("art does not allow Error terms that aren't factors by default", {
    data(Higgins1990Table5, package="ARTool")
    Higgins1990Table5$nt = as.numeric(Higgins1990Table5$Tray)

    expect_error(art(DryMatter ~ Moisture*Fertilizer + Error(nt), data=Higgins1990Table5), "The following Error terms are not factors")
    art(DryMatter ~ Moisture*Fertilizer + Error(Tray), data=Higgins1990Table5)
    art(DryMatter ~ Moisture*Fertilizer + Error(factor(nt)), data=Higgins1990Table5)
})

test_that("art of Higgins1990Table5 matches results of the original ARTool", {
    data(Higgins1990Table5, Higgins1990Table5.art, package="ARTool")

    #run art on original data
    m = art(DryMatter ~ Moisture*Fertilizer + (1|Tray), data=Higgins1990Table5)

    #verify column sums on aligned columns and F scores on aligned columns not of interest are all 0
    expect_equal(colSums(m$aligned), rep(0, ncol(m$aligned)), check.names=FALSE)
    aligned.anova = suppressWarnings(anova(m, response="aligned"))
    expect_equal(round(aligned.anova$F, digits=25), rep(0, nrow(aligned.anova)), check.names=FALSE)

    #verify that aligned responses were all calculated correctly
    expect_equal(m$aligned$Moisture, Higgins1990Table5.art$aligned.DryMatter..for.Moisture)
    expect_equal(m$aligned$Fertilizer, Higgins1990Table5.art$aligned.DryMatter..for.Fertilizer)
    expect_equal(m$aligned$`Moisture:Fertilizer`, Higgins1990Table5.art$aligned.DryMatter..for.Moisture.Fertilizer)

    #verify that ART responses were all calculated correctly
    expect_equal(m$aligned.ranks$Moisture, Higgins1990Table5.art$ART.DryMatter..for.Moisture)
    expect_equal(m$aligned.ranks$Fertilizer, Higgins1990Table5.art$ART.DryMatter..for.Fertilizer)
    expect_equal(m$aligned.ranks$`Moisture:Fertilizer`, Higgins1990Table5.art$ART.DryMatter..for.Moisture.Fertilizer)
})

test_that("art of Higgins1990Table1 matches results of the original ARTool", {
    data(Higgins1990Table1, Higgins1990Table1.art, package="ARTool")

    #run art on original data
    m = art(Response ~ Row*Column, data=Higgins1990Table1)

    #verify column sums on aligned columns and F scores on aligned columns not of interest are all 0
    expect_equal(colSums(m$aligned), rep(0, ncol(m$aligned)), check.names=FALSE)
    aligned.anova = suppressWarnings(anova(m, response="aligned"))
    expect_equal(round(aligned.anova$F, digits=25), rep(0, nrow(aligned.anova)), check.names=FALSE)

    #verify that aligned responses were all calculated correctly
    expect_equal(m$aligned$Row, Higgins1990Table1.art$aligned.Response..for.Row)
    expect_equal(m$aligned$Column, Higgins1990Table1.art$aligned.Response..for.Column)
    expect_equal(m$aligned$`Row:Column`, Higgins1990Table1.art$aligned.Response..for.Row.Column)

    #verify that ART responses were all calculated correctly
    expect_equal(m$aligned.ranks$Row, Higgins1990Table1.art$ART.Response..for.Row)
    expect_equal(m$aligned.ranks$Column, Higgins1990Table1.art$ART.Response..for.Column)
    expect_equal(m$aligned.ranks$`Row:Column`, Higgins1990Table1.art$ART.Response..for.Row.Column)
})

test_that("art of HigginsABC matches results of the original ARTool", {
    data(HigginsABC, HigginsABC.art, package="ARTool")

    #run art on original data
    m = art(Y ~ A*B*C + Error(Subject), data=HigginsABC)

    #verify column sums on aligned columns and F scores on aligned columns not of interest are all 0
    expect_equal(colSums(m$aligned), rep(0, ncol(m$aligned)), check.names=FALSE)
    aligned.anova = anova(m, response="aligned")
    expect_equal(round(aligned.anova$F, digits=25), rep(0, nrow(aligned.anova)), check.names=FALSE)

    #verify that aligned responses were all calculated correctly
    expect_equal(m$aligned$A, HigginsABC.art$aligned.Y..for.A)
    expect_equal(m$aligned$B, HigginsABC.art$aligned.Y..for.B)
    expect_equal(m$aligned$C, HigginsABC.art$aligned.Y..for.C)
    expect_equal(m$aligned$`A:B`, HigginsABC.art$aligned.Y..for.A.B)
    expect_equal(m$aligned$`A:C`, HigginsABC.art$aligned.Y..for.A.C)
    expect_equal(m$aligned$`B:C`, HigginsABC.art$aligned.Y..for.B.C)
    expect_equal(m$aligned$`A:B:C`, HigginsABC.art$aligned.Y..for.A.B.C)

    #verify that ART responses were all calculated correctly
    expect_equal(m$aligned.ranks$A, HigginsABC.art$ART.Y..for.A)
    expect_equal(m$aligned.ranks$B, HigginsABC.art$ART.Y..for.B)
    expect_equal(m$aligned.ranks$C, HigginsABC.art$ART.Y..for.C)
    expect_equal(m$aligned.ranks$`A:B`, HigginsABC.art$ART.Y..for.A.B)
    expect_equal(m$aligned.ranks$`A:C`, HigginsABC.art$ART.Y..for.A.C)
    expect_equal(m$aligned.ranks$`B:C`, HigginsABC.art$ART.Y..for.B.C)
    expect_equal(m$aligned.ranks$`A:B:C`, HigginsABC.art$ART.Y..for.A.B.C)
})

test_that("art works correctly on 4+-order interactions", {
    df_5way = dplyr::tribble(
        ~ S , ~ A , ~ B , ~ C , ~ D , ~ E , ~ R,
        "s1", "a1", "b1", "c1", "d1", "e1", 5,
        "s1", "a1", "b1", "c1", "d1", "e2", 4,
        "s1", "a1", "b1", "c1", "d2", "e1", 7,
        "s1", "a1", "b1", "c1", "d2", "e2", 6,
        "s1", "a1", "b1", "c2", "d1", "e1", 3,
        "s1", "a1", "b1", "c2", "d1", "e2", 7,
        "s1", "a1", "b1", "c2", "d2", "e1", 1,
        "s1", "a1", "b1", "c2", "d2", "e2", 0,
        "s1", "a1", "b2", "c1", "d1", "e1", 1,
        "s1", "a1", "b2", "c1", "d1", "e2", 4,
        "s1", "a1", "b2", "c1", "d2", "e1", 5,
        "s1", "a1", "b2", "c1", "d2", "e2", 4,
        "s1", "a1", "b2", "c2", "d1", "e1", 8,
        "s1", "a1", "b2", "c2", "d1", "e2", 0,
        "s1", "a1", "b2", "c2", "d2", "e1", 5,
        "s1", "a1", "b2", "c2", "d2", "e2", 6,
        "s1", "a2", "b1", "c1", "d1", "e1", 5,
        "s1", "a2", "b1", "c1", "d1", "e2", 9,
        "s1", "a2", "b1", "c1", "d2", "e1", 0,
        "s1", "a2", "b1", "c1", "d2", "e2", 7,
        "s1", "a2", "b1", "c2", "d1", "e1", 7,
        "s1", "a2", "b1", "c2", "d1", "e2", 3,
        "s1", "a2", "b1", "c2", "d2", "e1", 9,
        "s1", "a2", "b1", "c2", "d2", "e2", 8,
        "s1", "a2", "b2", "c1", "d1", "e1", 4,
        "s1", "a2", "b2", "c1", "d1", "e2", 5,
        "s1", "a2", "b2", "c1", "d2", "e1", 2,
        "s1", "a2", "b2", "c1", "d2", "e2", 8,
        "s1", "a2", "b2", "c2", "d1", "e1", 7,
        "s1", "a2", "b2", "c2", "d1", "e2", 8,
        "s1", "a2", "b2", "c2", "d2", "e1", 6,
        "s1", "a2", "b2", "c2", "d2", "e2", 1,
        "s2", "a1", "b1", "c1", "d1", "e1", 5,
        "s2", "a1", "b1", "c1", "d1", "e2", 3,
        "s2", "a1", "b1", "c1", "d2", "e1", 5,
        "s2", "a1", "b1", "c1", "d2", "e2", 6,
        "s2", "a1", "b1", "c2", "d1", "e1", 7,
        "s2", "a1", "b1", "c2", "d1", "e2", 2,
        "s2", "a1", "b1", "c2", "d2", "e1", 4,
        "s2", "a1", "b1", "c2", "d2", "e2", 5,
        "s2", "a1", "b2", "c1", "d1", "e1", 8,
        "s2", "a1", "b2", "c1", "d1", "e2", 7,
        "s2", "a1", "b2", "c1", "d2", "e1", 5,
        "s2", "a1", "b2", "c1", "d2", "e2", 6,
        "s2", "a1", "b2", "c2", "d1", "e1", 2,
        "s2", "a1", "b2", "c2", "d1", "e2", 4,
        "s2", "a1", "b2", "c2", "d2", "e1", 1,
        "s2", "a1", "b2", "c2", "d2", "e2", 4,
        "s2", "a2", "b1", "c1", "d1", "e1", 8,
        "s2", "a2", "b1", "c1", "d1", "e2", 4,
        "s2", "a2", "b1", "c1", "d2", "e1", 9,
        "s2", "a2", "b1", "c1", "d2", "e2", 6,
        "s2", "a2", "b1", "c2", "d1", "e1", 4,
        "s2", "a2", "b1", "c2", "d1", "e2", 7,
        "s2", "a2", "b1", "c2", "d2", "e1", 9,
        "s2", "a2", "b1", "c2", "d2", "e2", 0,
        "s2", "a2", "b2", "c1", "d1", "e1", 9,
        "s2", "a2", "b2", "c1", "d1", "e2", 5,
        "s2", "a2", "b2", "c1", "d2", "e1", 3,
        "s2", "a2", "b2", "c1", "d2", "e2", 5,
        "s2", "a2", "b2", "c2", "d1", "e1", 4,
        "s2", "a2", "b2", "c2", "d1", "e2", 7,
        "s2", "a2", "b2", "c2", "d2", "e1", 2,
        "s2", "a2", "b2", "c2", "d2", "e2", 3
    ) %>%
        dplyr::mutate_if(is.character, as.factor)

    aligned_results =
        data.frame(check.names = FALSE,
            A = c(-0.53125, -0.03125, 0.46875, -0.53125, -2.53125,  1.96875, -2.03125, -3.03125, -4.03125, -2.03125, -0.53125, -1.53125,  2.46875, -2.53125, 1.46875, 0.46875, -0.96875, 3.03125, -3.96875,  1.03125, 2.03125, -1.46875, 0.53125, 4.53125, -1.96875, 0.53125,  0.03125, 2.03125, 2.03125, 1.03125, 2.53125, -0.46875, -0.53125,  -1.03125, -1.53125, -0.53125, 1.46875, -3.03125, 0.96875, 1.96875,  2.96875, 0.96875, -0.53125, 0.46875, -3.53125, 1.46875, -2.53125,  -1.53125, 2.03125, -1.96875, 5.03125, 0.03125, -0.96875, 2.53125,  0.53125, -3.46875, 3.03125, 0.53125, 1.03125, -0.96875, -0.96875,  0.03125, -1.46875, 1.53125),
            B = c(0.25, 0.75, 1.25, 0.25, -1.75,  2.75, -1.25, -2.25, -3.75, -1.75, -0.25, -1.25, 2.75, -2.25,  1.75, 0.75, -1.25, 2.75, -4.25, 0.75, 1.75, -1.75, 0.25, 4.25,  -2.75, -0.25, -0.75, 1.25, 1.25, 0.25, 1.75, -1.25, 0.25, -0.25,  -0.75, 0.25, 2.25, -2.25, 1.75, 2.75, 3.25, 1.25, -0.25, 0.75,  -3.25, 1.75, -2.25, -1.25, 1.75, -2.25, 4.75, -0.25, -1.25, 2.25,  0.25, -3.75, 2.25, -0.25, 0.25, -1.75, -1.75, -0.75, -2.25, 0.75 ),
            C = c(0.40625, 0.90625, 1.40625, 0.40625, -2.40625, 2.09375,  -1.90625, -2.90625, -3.09375, -1.09375, 0.40625, -0.59375, 2.59375,  -2.40625, 1.59375, 0.59375, -1.09375, 2.90625, -4.09375, 0.90625,  1.09375, -2.40625, -0.40625, 3.59375, -2.09375, 0.40625, -0.09375,  1.90625, 1.09375, 0.09375, 1.59375, -1.40625, 0.40625, -0.09375,  -0.59375, 0.40625, 1.59375, -2.90625, 1.09375, 2.09375, 3.90625,  1.90625, 0.40625, 1.40625, -3.40625, 1.59375, -2.40625, -1.40625,  1.90625, -2.09375, 4.90625, -0.09375, -1.90625, 1.59375, -0.40625,  -4.40625, 2.90625, 0.40625, 0.90625, -1.09375, -1.90625, -0.90625,  -2.40625, 0.59375),
            D = c(0.28125, 0.78125, 0.71875, -0.28125,  -1.71875, 2.78125, -1.78125, -2.78125, -3.21875, -1.21875, -0.28125,  -1.28125, 3.28125, -1.71875, 1.71875, 0.71875, -1.21875, 2.78125,  -4.78125, 0.21875, 1.78125, -1.71875, -0.28125, 3.71875, -2.21875,  0.28125, -0.78125, 1.21875, 1.78125, 0.78125, 1.71875, -1.28125,  0.28125, -0.21875, -1.28125, -0.28125, 2.28125, -2.21875, 1.21875,  2.21875, 3.78125, 1.78125, -0.28125, 0.71875, -2.71875, 2.28125,  -2.28125, -1.28125, 1.78125, -2.21875, 4.21875, -0.78125, -1.21875,  2.28125, -0.28125, -4.28125, 2.78125, 0.28125, 0.21875, -1.78125,  -1.21875, -0.21875, -2.28125, 0.71875),
            E = c(0.09375, 0.40625,  1.09375, -0.09375, -1.90625, 2.40625, -1.40625, -2.59375, -3.40625,  -1.59375, 0.09375, -1.09375, 3.09375, -2.09375, 2.09375, 0.90625,  -1.40625, 2.40625, -4.40625, 0.40625, 1.59375, -2.09375, 0.09375,  3.90625, -2.40625, -0.09375, -0.40625, 1.40625, 1.59375, 0.40625,  2.09375, -1.09375, 0.09375, -0.59375, -0.90625, -0.09375, 2.09375,  -2.59375, 1.59375, 2.40625, 3.59375, 1.40625, 0.09375, 0.90625,  -2.90625, 1.90625, -1.90625, -1.09375, 1.59375, -2.59375, 4.59375,  -0.59375, -1.40625, 1.90625, 0.09375, -4.09375, 2.59375, -0.09375,  0.59375, -1.59375, -1.40625, -0.59375, -1.90625, 0.90625),
            "A:B" = c(-0.25,  0.25, 0.75, -0.25, -2.25, 2.25, -1.75, -2.75, -3.25, -1.25, 0.25,  -0.75, 3.25, -1.75, 2.25, 1.25, -1.25, 2.75, -4.25, 0.75, 1.75,  -1.75, 0.25, 4.25, -2.75, -0.25, -0.75, 1.25, 1.25, 0.25, 1.75,  -1.25, -0.25, -0.75, -1.25, -0.25, 1.75, -2.75, 1.25, 2.25, 3.75,  1.75, 0.25, 1.25, -2.75, 2.25, -1.75, -0.75, 1.75, -2.25, 4.75,  -0.25, -1.25, 2.25, 0.25, -3.75, 2.25, -0.25, 0.25, -1.75, -1.75,  -0.75, -2.25, 0.75),
            "A:C" = c(0.28125, 0.78125, 1.28125, 0.28125,  -2.28125, 2.21875, -1.78125, -2.78125, -3.21875, -1.21875, 0.28125,  -0.71875, 2.71875, -2.28125, 1.71875, 0.71875, -1.78125, 2.21875,  -4.78125, 0.21875, 1.78125, -1.71875, 0.28125, 4.28125, -2.78125,  -0.28125, -0.78125, 1.21875, 1.78125, 0.78125, 2.28125, -0.71875,  0.28125, -0.21875, -0.71875, 0.28125, 1.71875, -2.78125, 1.21875,  2.21875, 3.78125, 1.78125, 0.28125, 1.28125, -3.28125, 1.71875,  -2.28125, -1.28125, 1.21875, -2.78125, 4.21875, -0.78125, -1.21875,  2.28125, 0.28125, -3.71875, 2.21875, -0.28125, 0.21875, -1.78125,  -1.21875, -0.21875, -1.71875, 1.28125),
            "B:C" = c(0, 0.5, 1,  0, -2, 2.5, -1.5, -2.5, -3.5, -1.5, 0, -1, 3, -2, 2, 1, -1.5,  2.5, -4.5, 0.5, 1.5, -2, 0, 4, -2.5, 0, -0.5, 1.5, 1.5, 0.5,  2, -1, 0, -0.5, -1, 0, 2, -2.5, 1.5, 2.5, 3.5, 1.5, 0, 1, -3,  2, -2, -1, 1.5, -2.5, 4.5, -0.5, -1.5, 2, 0, -4, 2.5, 0, 0.5,  -1.5, -1.5, -0.5, -2, 1),
            "A:D" = c(-0.28125, 0.21875, 1.28125,  0.28125, -2.28125, 2.21875, -1.21875, -2.21875, -3.78125, -1.78125,  0.28125, -0.71875, 2.71875, -2.28125, 2.28125, 1.28125, -1.21875,  2.78125, -4.78125, 0.21875, 1.78125, -1.71875, -0.28125, 3.71875,  -2.21875, 0.28125, -0.78125, 1.21875, 1.78125, 0.78125, 1.71875,  -1.28125, -0.28125, -0.78125, -0.71875, 0.28125, 1.71875, -2.78125,  1.78125, 2.78125, 3.21875, 1.21875, 0.28125, 1.28125, -3.28125,  1.71875, -1.71875, -0.71875, 1.78125, -2.21875, 4.21875, -0.78125,  -1.21875, 2.28125, -0.28125, -4.28125, 2.78125, 0.28125, 0.21875,  -1.78125, -1.21875, -0.21875, -2.28125, 0.71875),
            "B:D" = c(-0.25,  0.25, 1.25, 0.25, -2.25, 2.25, -1.25, -2.25, -3.25, -1.25, -0.25,  -1.25, 3.25, -1.75, 1.75, 0.75, -1.75, 2.25, -4.25, 0.75, 1.25,  -2.25, 0.25, 4.25, -2.25, 0.25, -0.75, 1.25, 1.75, 0.75, 1.75,  -1.25, -0.25, -0.75, -0.75, 0.25, 1.75, -2.75, 1.75, 2.75, 3.75,  1.75, -0.25, 0.75, -2.75, 2.25, -2.25, -1.25, 1.25, -2.75, 4.75,  -0.25, -1.75, 1.75, 0.25, -3.75, 2.75, 0.25, 0.25, -1.75, -1.25,  -0.25, -2.25, 0.75),
            "C:D" = c(-0.21875, 0.28125, 1.21875, 0.21875,  -1.78125, 2.71875, -1.71875, -2.71875, -3.71875, -1.71875, 0.21875,  -0.78125, 3.21875, -1.78125, 1.78125, 0.78125, -1.71875, 2.28125,  -4.28125, 0.71875, 1.71875, -1.78125, -0.21875, 3.78125, -2.71875,  -0.21875, -0.28125, 1.71875, 1.71875, 0.71875, 1.78125, -1.21875,  -0.21875, -0.71875, -0.78125, 0.21875, 2.21875, -2.28125, 1.28125,  2.28125, 3.28125, 1.28125, 0.21875, 1.21875, -2.78125, 2.21875,  -2.21875, -1.21875, 1.28125, -2.71875, 4.71875, -0.28125, -1.28125,  2.21875, -0.21875, -4.21875, 2.28125, -0.21875, 0.71875, -1.28125,  -1.28125, -0.28125, -2.21875, 0.78125),
            "A:E" = c(0.03125, 0.46875,  1.03125, -0.03125, -1.96875, 2.46875, -1.46875, -2.53125, -3.46875,  -1.53125, 0.03125, -1.03125, 3.03125, -2.03125, 2.03125, 0.96875,  -1.53125, 2.53125, -4.53125, 0.53125, 1.46875, -1.96875, -0.03125,  4.03125, -2.53125, 0.03125, -0.53125, 1.53125, 1.46875, 0.53125,  1.96875, -0.96875, 0.03125, -0.53125, -0.96875, -0.03125, 2.03125,  -2.53125, 1.53125, 2.46875, 3.53125, 1.46875, 0.03125, 0.96875,  -2.96875, 1.96875, -1.96875, -1.03125, 1.46875, -2.46875, 4.46875,  -0.46875, -1.53125, 2.03125, -0.03125, -3.96875, 2.46875, 0.03125,  0.46875, -1.46875, -1.53125, -0.46875, -2.03125, 1.03125),
            "B:E" = c(0.25,  0.25, 1.25, -0.25, -1.75, 2.25, -1.25, -2.75, -3.75, -1.25, -0.25,  -0.75, 2.75, -1.75, 1.75, 1.25, -1.25, 2.25, -4.25, 0.25, 1.75,  -2.25, 0.25, 3.75, -2.75, 0.25, -0.75, 1.75, 1.25, 0.75, 1.75,  -0.75, 0.25, -0.75, -0.75, -0.25, 2.25, -2.75, 1.75, 2.25, 3.25,  1.75, -0.25, 1.25, -3.25, 2.25, -2.25, -0.75, 1.75, -2.75, 4.75,  -0.75, -1.25, 1.75, 0.25, -4.25, 2.25, 0.25, 0.25, -1.25, -1.75,  -0.25, -2.25, 1.25),
            "C:E" = c(-0.34375, 0.84375, 0.65625, 0.34375,  -1.65625, 2.15625, -1.15625, -2.84375, -3.84375, -1.15625, -0.34375,  -0.65625, 3.34375, -2.34375, 2.34375, 0.65625, -1.84375, 2.84375,  -4.84375, 0.84375, 1.84375, -2.34375, 0.34375, 3.65625, -2.84375,  0.34375, -0.84375, 1.84375, 1.84375, 0.15625, 2.34375, -1.34375,  -0.34375, -0.15625, -1.34375, 0.34375, 2.34375, -2.84375, 1.84375,  2.15625, 3.15625, 1.84375, -0.34375, 1.34375, -2.65625, 1.65625,  -1.65625, -1.34375, 1.15625, -2.15625, 4.15625, -0.15625, -1.15625,  1.65625, 0.34375, -4.34375, 2.15625, 0.34375, 0.15625, -1.15625,  -1.15625, -0.84375, -1.65625, 0.65625),
            "D:E" = c(0.15625, 0.34375,  0.84375, 0.15625, -1.84375, 2.34375, -1.65625, -2.34375, -3.34375,  -1.65625, -0.15625, -0.84375, 3.15625, -2.15625, 1.84375, 1.15625,  -1.34375, 2.34375, -4.65625, 0.65625, 1.65625, -2.15625, -0.15625,  4.15625, -2.34375, -0.15625, -0.65625, 1.65625, 1.65625, 0.34375,  1.84375, -0.84375, 0.15625, -0.65625, -1.15625, 0.15625, 2.15625,  -2.65625, 1.34375, 2.65625, 3.65625, 1.34375, -0.15625, 1.15625,  -2.84375, 1.84375, -2.15625, -0.84375, 1.65625, -2.65625, 4.34375,  -0.34375, -1.34375, 1.84375, -0.15625, -3.84375, 2.65625, -0.15625,  0.34375, -1.34375, -1.34375, -0.65625, -2.15625, 1.15625),
            "A:B:C" = c(0.0625,  0.5625, 1.0625, 0.0625, -2.0625, 2.4375, -1.5625, -2.5625, -3.5625,  -1.5625, -0.0625, -1.0625, 3.0625, -1.9375, 2.0625, 1.0625, -1.5625,  2.4375, -4.5625, 0.4375, 1.5625, -1.9375, 0.0625, 4.0625, -2.4375,  0.0625, -0.4375, 1.5625, 1.4375, 0.4375, 1.9375, -1.0625, 0.0625,  -0.4375, -0.9375, 0.0625, 1.9375, -2.5625, 1.4375, 2.4375, 3.4375,  1.4375, -0.0625, 0.9375, -2.9375, 2.0625, -1.9375, -0.9375, 1.4375,  -2.5625, 4.4375, -0.5625, -1.4375, 2.0625, 0.0625, -3.9375, 2.5625,  0.0625, 0.5625, -1.4375, -1.5625, -0.5625, -2.0625, 0.9375),
            "A:B:D" = c(0.375, 0.875, 0.625, -0.375, -1.625, 2.875, -1.875,      -2.875, -3.875, -1.875, 0.375, -0.625, 2.625, -2.375, 2.375,      1.375, -1.875, 2.125, -4.125, 0.875, 1.125, -2.375, 0.375,      4.375, -2.125, 0.375, -0.875, 1.125, 1.875, 0.875, 1.625,      -1.375, 0.375, -0.125, -1.375, -0.375, 2.375, -2.125, 1.125,      2.125, 3.125, 1.125, 0.375, 1.375, -3.375, 1.625, -1.625,      -0.625, 1.125, -2.875, 4.875, -0.125, -1.875, 1.625, 0.375,      -3.625, 2.875, 0.375, 0.125, -1.875, -1.125, -0.125, -2.375,      0.625),
            "A:C:D" = c(-0.21875, 0.28125, 1.21875, 0.21875,      -1.78125, 2.71875, -1.71875, -2.71875, -3.71875, -1.71875,      0.21875, -0.78125, 3.21875, -1.78125, 1.78125, 0.78125, -1.28125,      2.71875, -4.71875, 0.28125, 1.28125, -2.21875, 0.21875, 4.21875,      -2.28125, 0.21875, -0.71875, 1.28125, 1.28125, 0.28125, 2.21875,      -0.78125, -0.21875, -0.71875, -0.78125, 0.21875, 2.21875,      -2.28125, 1.28125, 2.28125, 3.28125, 1.28125, 0.21875, 1.21875,      -2.78125, 2.21875, -2.21875, -1.21875, 1.71875, -2.28125,      4.28125, -0.71875, -1.71875, 1.78125, 0.21875, -3.78125,      2.71875, 0.21875, 0.28125, -1.71875, -1.71875, -0.71875,      -1.78125, 1.21875),
            "B:C:D" = c(0, 0.5, 1, 0, -2, 2.5, -1.5,      -2.5, -3.5, -1.5, 0, -1, 3, -2, 2, 1, -1.5, 2.5, -4.5, 0.5,      1.5, -2, 0, 4, -2.5, 0, -0.5, 1.5, 1.5, 0.5, 2, -1, 0, -0.5,      -1, 0, 2, -2.5, 1.5, 2.5, 3.5, 1.5, 0, 1, -3, 2, -2, -1,      1.5, -2.5, 4.5, -0.5, -1.5, 2, 0, -4, 2.5, 0, 0.5, -1.5,      -1.5, -0.5, -2, 1),
            "A:B:E" = c(-0.125, 0.625, 0.875, 0.125,      -2.125, 2.625, -1.625, -2.375, -3.375, -1.625, 0.125, -1.125,      3.125, -2.125, 2.125, 0.875, -1.375, 2.375, -4.375, 0.375,      1.625, -2.125, 0.125, 3.875, -2.625, 0.125, -0.625, 1.625,      1.375, 0.625, 1.875, -0.875, -0.125, -0.375, -1.125, 0.125,      1.875, -2.375, 1.375, 2.625, 3.625, 1.375, 0.125, 0.875,      -2.875, 1.875, -1.875, -1.125, 1.625, -2.625, 4.625, -0.625,      -1.375, 1.875, 0.125, -4.125, 2.375, 0.125, 0.375, -1.375,      -1.625, -0.375, -2.125, 1.125),
            "A:C:E" = c(0.28125, 0.21875,      1.28125, -0.28125, -2.28125, 2.78125, -1.78125, -2.21875,      -3.21875, -1.78125, 0.28125, -1.28125, 2.71875, -1.71875,      1.71875, 1.28125, -1.78125, 2.78125, -4.78125, 0.78125, 1.78125,      -2.28125, 0.28125, 3.71875, -2.78125, 0.28125, -0.78125,      1.78125, 1.78125, 0.21875, 2.28125, -1.28125, 0.28125, -0.78125,      -0.71875, -0.28125, 1.71875, -2.21875, 1.21875, 2.78125,      3.78125, 1.21875, 0.28125, 0.71875, -3.28125, 2.28125, -2.28125,      -0.71875, 1.21875, -2.21875, 4.21875, -0.21875, -1.21875,      1.71875, 0.28125, -4.28125, 2.21875, 0.28125, 0.21875, -1.21875,      -1.21875, -0.78125, -1.71875, 0.71875),
            "B:C:E" = c(-0.0625,      0.5625, 0.9375, 0.0625, -1.9375, 2.4375, -1.4375, -2.5625,      -3.4375, -1.5625, 0.0625, -1.0625, 2.9375, -1.9375, 1.9375,      1.0625, -1.5625, 2.5625, -4.5625, 0.5625, 1.5625, -2.0625,      0.0625, 3.9375, -2.4375, -0.0625, -0.4375, 1.4375, 1.4375,      0.5625, 1.9375, -0.9375, -0.0625, -0.4375, -1.0625, 0.0625,      2.0625, -2.5625, 1.5625, 2.4375, 3.5625, 1.4375, 0.0625,      0.9375, -3.0625, 2.0625, -2.0625, -0.9375, 1.4375, -2.4375,      4.4375, -0.4375, -1.4375, 1.9375, 0.0625, -4.0625, 2.5625,      -0.0625, 0.5625, -1.5625, -1.5625, -0.4375, -2.0625, 1.0625     ),
            "A:D:E" = c(0.21875, 0.28125, 0.78125, 0.21875, -1.78125,      2.28125, -1.71875, -2.28125, -3.28125, -1.71875, -0.21875,      -0.78125, 3.21875, -2.21875, 1.78125, 1.21875, -1.71875,      2.71875, -4.28125, 0.28125, 1.28125, -1.78125, 0.21875, 3.78125,      -2.71875, 0.21875, -0.28125, 1.28125, 1.28125, 0.71875, 2.21875,      -1.21875, 0.21875, -0.71875, -1.21875, 0.21875, 2.21875,      -2.71875, 1.28125, 2.71875, 3.71875, 1.28125, -0.21875, 1.21875,      -2.78125, 1.78125, -2.21875, -0.78125, 1.28125, -2.28125,      4.71875, -0.71875, -1.71875, 2.21875, 0.21875, -4.21875,      2.28125, 0.21875, 0.71875, -1.71875, -1.71875, -0.28125,      -1.78125, 0.78125),
            "B:D:E" = c(-0.1875, 0.6875, 1.1875,      -0.1875, -2.1875, 2.6875, -1.3125, -2.6875, -3.3125, -1.6875,      -0.1875, -0.8125, 3.1875, -2.1875, 1.8125, 1.1875, -1.6875,      2.6875, -4.3125, 0.3125, 1.3125, -1.8125, 0.1875, 3.8125,      -2.3125, -0.1875, -0.6875, 1.6875, 1.6875, 0.3125, 1.8125,      -0.8125, -0.1875, -0.3125, -0.8125, -0.1875, 1.8125, -2.3125,      1.6875, 2.3125, 3.6875, 1.3125, -0.1875, 1.1875, -2.8125,      1.8125, -2.1875, -0.8125, 1.3125, -2.3125, 4.6875, -0.6875,      -1.6875, 2.1875, 0.1875, -4.1875, 2.6875, -0.1875, 0.3125,      -1.3125, -1.3125, -0.6875, -2.1875, 1.1875),
            "C:D:E" = c(0.34375,      0.15625, 0.65625, 0.34375, -2.34375, 2.84375, -1.15625, -2.84375,      -3.15625, -1.84375, -0.34375, -0.65625, 2.65625, -1.65625,      2.34375, 0.65625, -1.15625, 2.15625, -4.84375, 0.84375, 1.15625,      -1.65625, 0.34375, 3.65625, -2.15625, -0.34375, -0.84375,      1.84375, 1.15625, 0.84375, 2.34375, -1.34375, 0.34375, -0.84375,      -1.34375, 0.34375, 1.65625, -2.15625, 1.84375, 2.15625, 3.84375,      1.15625, -0.34375, 1.34375, -3.34375, 2.34375, -1.65625,      -1.34375, 1.84375, -2.84375, 4.15625, -0.15625, -1.84375,      2.34375, 0.34375, -4.34375, 2.84375, -0.34375, 0.15625, -1.15625,      -1.84375, -0.15625, -1.65625, 0.65625),
            "A:B:C:D" = c(-0.5625,      -0.0625, 1.5625, 0.5625, -1.4375, 3.0625, -2.0625, -3.0625,      -2.9375, -0.9375, -0.5625, -1.5625, 2.4375, -2.5625, 2.5625,      1.5625, -0.9375, 3.0625, -5.0625, -0.0625, 0.9375, -2.5625,      0.5625, 4.5625, -3.0625, -0.5625, 0.0625, 2.0625, 2.0625,      1.0625, 1.4375, -1.5625, -0.5625, -1.0625, -0.4375, 0.5625,      2.5625, -1.9375, 0.9375, 1.9375, 4.0625, 2.0625, -0.5625,      0.4375, -3.5625, 1.4375, -1.4375, -0.4375, 2.0625, -1.9375,      3.9375, -1.0625, -2.0625, 1.4375, 0.5625, -3.4375, 1.9375,      -0.5625, 1.0625, -0.9375, -0.9375, 0.0625, -2.5625, 0.4375     ),
            "A:B:C:E" = c(0.25, 0.25, 1.25, -0.25, -2.25, 2.75, -1.75,      -2.25, -3.75, -1.25, -0.25, -0.75, 3.25, -2.25, 2.25, 0.75,      -1.75, 2.75, -4.75, 0.75, 1.75, -2.25, 0.25, 3.75, -2.25,      -0.25, -0.25, 1.25, 1.25, 0.75, 1.75, -0.75, 0.25, -0.75,      -0.75, -0.25, 1.75, -2.25, 1.25, 2.75, 3.25, 1.75, -0.25,      1.25, -2.75, 1.75, -1.75, -1.25, 1.25, -2.25, 4.25, -0.25,      -1.25, 1.75, 0.25, -4.25, 2.75, -0.25, 0.75, -1.75, -1.75,      -0.25, -2.25, 1.25),
            "A:B:D:E" = c(0.0625, 0.4375, 0.9375,      0.0625, -1.9375, 2.4375, -1.5625, -2.4375, -3.5625, -1.4375,      0.0625, -1.0625, 2.9375, -1.9375, 2.0625, 0.9375, -1.5625,      2.5625, -4.4375, 0.4375, 1.4375, -1.9375, 0.0625, 3.9375,      -2.4375, -0.0625, -0.5625, 1.5625, 1.5625, 0.4375, 1.9375,      -0.9375, 0.0625, -0.5625, -1.0625, 0.0625, 2.0625, -2.5625,      1.4375, 2.5625, 3.4375, 1.5625, 0.0625, 0.9375, -3.0625,      2.0625, -1.9375, -1.0625, 1.4375, -2.4375, 4.5625, -0.5625,      -1.5625, 2.0625, 0.0625, -4.0625, 2.5625, -0.0625, 0.4375,      -1.4375, -1.4375, -0.5625, -2.0625, 1.0625),
            "A:C:D:E" = c(-0.65625,      1.15625, 1.65625, -0.65625, -1.34375, 1.84375, -2.15625,      -1.84375, -4.15625, -0.84375, 0.65625, -1.65625, 3.65625,      -2.65625, 1.34375, 1.65625, -0.84375, 1.84375, -5.15625,      1.15625, 0.84375, -1.34375, 0.65625, 3.34375, -1.84375, -0.65625,      -1.15625, 2.15625, 0.84375, 1.15625, 2.65625, -1.65625, -0.65625,      0.15625, -0.34375, -0.65625, 2.65625, -3.15625, 0.84375,      3.15625, 2.84375, 2.15625, 0.65625, 0.34375, -2.34375, 1.34375,      -2.65625, -0.34375, 2.15625, -3.15625, 3.84375, 0.15625,      -2.15625, 2.65625, 0.65625, -4.65625, 3.15625, -0.65625,      -0.15625, -0.84375, -2.15625, 0.15625, -1.34375, 0.34375),
            "B:C:D:E" = c(0.125, 0.375, 0.875, 0.125, -2.125, 2.625,      -1.375, -2.625, -3.625, -1.375, 0.125, -1.125, 3.125, -2.125,      1.875, 1.125, -1.375, 2.375, -4.625, 0.625, 1.375, -1.875,      0.125, 3.875, -2.625, 0.125, -0.375, 1.375, 1.625, 0.375,      1.875, -0.875, 0.125, -0.625, -1.125, 0.125, 1.875, -2.375,      1.625, 2.375, 3.375, 1.625, 0.125, 0.875, -2.875, 1.875,      -2.125, -0.875, 1.625, -2.625, 4.375, -0.375, -1.625, 2.125,      0.125, -4.125, 2.375, 0.125, 0.625, -1.625, -1.375, -0.625,      -2.125, 1.125),
            "A:B:C:D:E" = c(0.3125, 0.1875, 0.6875, 0.3125,      -2.3125, 2.8125, -1.1875, -2.8125, -3.8125, -1.1875, 0.3125,      -1.3125, 3.3125, -2.3125, 1.6875, 1.3125, -1.8125, 2.8125,      -4.1875, 0.1875, 1.8125, -2.3125, -0.3125, 4.3125, -2.1875,      -0.3125, -0.8125, 1.8125, 1.1875, 0.8125, 2.3125, -1.3125,      0.3125, -0.8125, -1.3125, 0.3125, 1.6875, -2.1875, 1.8125,      2.1875, 3.1875, 1.8125, 0.3125, 0.6875, -2.6875, 1.6875,      -2.3125, -0.6875, 1.1875, -2.1875, 4.8125, -0.8125, -1.1875,      1.6875, -0.3125, -3.6875, 2.8125, -0.3125, 0.1875, -1.1875,      -1.8125, -0.1875, -1.6875, 0.6875)
        )


    #run art on original data
    m1 <- art(R ~ A*B*C*D*E, data = df_5way)

    expect_equal(m1$aligned, aligned_results)
})

