% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clonal_expansion_plot.R
\name{clonal_expansion_plot}
\alias{clonal_expansion_plot}
\title{Visualize T cell clonal expansion with a ball-packing plot.}
\usage{
clonal_expansion_plot(
  seurat_obj,
  tcr_df = "seurat_obj_already_integrated",
  res = 360,
  clone_scale_factor = 0.1,
  rad_scale_factor = 0.95,
  ORDER = TRUE,
  try_place = FALSE,
  verbose = TRUE,
  repulse = FALSE,
  repulsion_threshold = 1,
  repulsion_strength = 1,
  max_repulsion_iter = 10,
  use_default_theme = TRUE,
  show_origin = FALSE,
  retain_axis_scales = FALSE,
  add_size_legend = TRUE,
  legend_sizes = c(1, 5, 50),
  legend_position = "top_left",
  legend_buffer = 1.5,
  legend_color = "#808080",
  legend_spacing = 0.4
)
}
\arguments{
\item{seurat_obj}{Seurat object with at least a UMAP reduction. Can either already have been integrated with a T cell library via `integrate_tcr(seurat_obj, tcr_df)`, in which case the subsequent `tcr_df` argument can be left empty. Else, the object must be a regular seurat object and a T cell library must be inputted in the following `tcr_df` argument}

\item{tcr_df}{If left empty, `seurat_obj` is assumed to be already integrated. Otherwise, should be a `data.frame` of the T cell library generated by 10X genomics' Cell Ranger. The dataframe has to at least have the `barcode` and `raw_clonotype_id` columns.}

\item{res}{The number of points on the generated path per full circle. From plot viewers, if circles seem slightly too pixelated, it is highly recommended to first try to export the plot as an `.svg` before increasing `res`}

\item{clone_scale_factor}{numeric. Decides how much to scale each circle. Usually should be kept at around 0.01 to somewhat maintain UMAP structure for large datasets. However, if the plot that is displayed is ever blank, first try increasing this value.}

\item{rad_scale_factor}{numeric. indicates how much the radii of the clones should decrease to add a slight gap between all of them. Defaults to 1 but 0.85-0.95 values are recommended. Both `rad_scale_factor` and `clone_scale_factor` may need to be repeatedly readjusted}

\item{ORDER}{logical. Decides if the largest clones should be at the cluster centroids}

\item{try_place}{If `TRUE`, always minimizes distance from a newly placed circle to the origin}

\item{verbose}{logical. Decides if visual cues print to the R console of the packing progress}

\item{repulse}{If `TRUE`, will attempt to push overlapping clusters away from each other.}

\item{repulsion_threshold}{numeric. The radius that cluster overlap is acceptable}

\item{repulsion_strength}{numeric. The smaller the value the less the clusters repulse each other}

\item{max_repulsion_iter}{numeric. The number of repulsion iterations, note that increasing this value may occasionally even lead to worse looking plots as clusters may repulse eachother too much}

\item{use_default_theme}{If `TRUE`, the resulting plot will have the same theme as the seurat UMAP. Else, the plot will simply have a blank background}

\item{show_origin}{logical. If `TRUE`, only the centers of each circle will be plotted}

\item{retain_axis_scales}{If `TRUE`, approximately maintains the axis scales of the original UMAP. However, it will only attempt to extend the axes and never shorten.}

\item{add_size_legend}{If `TRUE`, adds a legend to the plot titled `"Clone sizes"` indicating the relative sizes of clones.}

\item{legend_sizes}{numeric vector. Indicates the circle sizes to be displayed on the legend and defaults to `c(1, 5, 10)`.}

\item{legend_position}{character. Can be set to either `"top_left"`, `"top_right"`, `"bottom_left"`, `"bottom_right"` and places the legend roughly in the corresponding position}

\item{legend_buffer}{numeric. Indicates how much to "push" the legend towards the center of the plot from the selected corner. If negative, will push away}

\item{legend_color}{character. Indicates the hex color of the circles displayed on the legend. Defaults to the hex code for gray}

\item{legend_spacing}{numeric. Indicates the horizontal distance between each stacked circle on the size legend. Usually should be kept below 0.75 -ish depending on plot size.}
}
\value{
Returns a ggplot2 object of the ball packing plot. Can be operated on like normal ggplot objects
}
\description{
Integrates a cell ranger T cell library into a Seurat object with a UMAP
reduction. Then gets sizes of unique clones and utilizes a circle-packing
algorithm to pack circles representing individual clones in approximately
the same UMAP coordinates and clusters into a ggplot object.
}
\details{
Check out the web-only user vignette at
`https://qile0317.github.io/APackOfTheClones/articles/web_only/Clonal_expansion_plotting.html`
for a walkthrough on using this function, and additional details.
}
\examples{
library(Seurat)
library(APackOfTheClones)
data("mini_clonotype_data","mini_seurat_obj")

# produce and show the ball-packing plot by integrating the data
ball_pack_plot <- clonal_expansion_plot(
  mini_seurat_obj, mini_clonotype_data, verbose = FALSE
)

#> integrating TCR library into seurat object
#>   |===============================================| 100\%
#> Percent of unique barcodes: 100 \%
#>
#> packing cluster 0
#> [==================================================] 100\%
#> packing cluster 1
#> [==================================================] 100\%
#> Plotting completed successfully

ball_pack_plot

# it's also possible (and preferable) to input an integrated Seurat object
integrated_seurat_object <- integrate_tcr(
  mini_seurat_obj, mini_clonotype_data, verbose = FALSE
)
ball_pack_plot <- clonal_expansion_plot(
  integrated_seurat_object, verbose = FALSE
)
ball_pack_plot

}
\seealso{
\code{\link{integrate_tcr}}
}
