#' @title Sample (with noise) from annuli
#'
#' @description These functions generate uniform samples from annuli with in
#'   2-dimensional space with major radius 1, optionally with noise.
#'
#' @details The sample is generated by an area-preserving parameterization of
#'   the annulus. This parameterization was derived through the method for
#'   sampling 2-manifolds as described by Arvo (2001).

#' @template ref-arvo2001
#' 

#' @name annuli
#' @aliases disks
#' @param n Number of observations.
#' @param r Minor radius of the annulus; `0` yields the disk.
#' @param bins Number of intervals per dimension to stratify by. Default set to
#'   1, which generates a uniform sample.
#' @param sd Standard deviation of (independent multivariate) Gaussian noise.
#' @example inst/examples/ex-annuli.r
NULL

#' @rdname annuli
#' @export
sample_annulus <- function(n, r = .5, bins = 1L, sd = 0) {
  # generate a uniform sample from [0,1]^2
  unit_square <- sample_strat_square(n, bins)
  # area-preserving parameterization of annulus
  res <- apm_annulus(unit_square, r)
  # add noise
  add_noise(res, sd = sd)
}

#' @rdname annuli
#' @export
sample_disk <- function(n, bins = 1L, sd = 0) {
  sample_annulus(n, r = 0, bins = bins, sd = sd)
}

# area-preserving map from [r,1]^2 to the disk
apm_annulus <- function(x, r) {
  cbind(
    x = sqrt(1 - (1 - r^2) * x[, 1]) * cos(2 * pi * x[, 2]),
    y = sqrt(1 - (1 - r^2) * x[, 1]) * sin(2 * pi * x[, 2])
  )
}
