#' Paired measures (T-Test & Regression)
#'
#' @description Generates two paired measures. It provides T-test and a simple linear regression model for generated data.
#'
#' @param n size of each sample.
#' @param mean vector of means.
#' @param sigma vector of standard deviations.
#' @param coefvar an optional vector of coefficients of variation.
#' @param rho Pearson correlation coefficient (optional). If \code{rho} = \code{NULL} a random covariance matrix is generated by \code{genPositiveDefMat()}.
#' @param alternative a character string specifying the alternative hypothesis for T-Test. Must be one of ``two.sided`` (default), ``greater`` or ``less``. Can be specified just the initial letter.
#' @param delta true value of the difference in means.
#' @param conf.level confidence level for interval in T-Test.
#' @param dec number of decimals for observations.
#' @param random a logical a logical indicating whether you want a random covariance/variance matrix.
#'
#' @usage
#' pairedm(n, mean = 0, sigma = 1, coefvar = NULL,
#'         rho = NULL, alternative = c("two.sided", "less", "greater"),
#'         delta = 0, conf.level = 0.95, dec = 2,
#'         random = FALSE)
#'
#' @details If \code{random} = TRUE, \code{rho} is omitted and \code{sigma} is taken as range for variances of the covariance matrix.
#'
#' @return List containing the following components :
#'
#' \itemize{
#'
#'  \item \code{Data}: a data frame containing the samples created.
#'
#'
#'  \item \code{Model}: linear regression model.
#'
#'  \item \code{T.Test}: a t-test for the samples.
#'
#' }
#'
#' @seealso \code{[clusterGeneration::genpositiveDefMat()]}
#' @examples
#'
#' pairedm(10, mean = c(10,2), sigma = c(1.2,0.7), rho = 0.5, alternative = "g")
#' pairedm(15, mean =c(1,2), coefvar = 0.1, random = TRUE)
#'
#' @export

pairedm <- function(n, mean = 0, sigma = 1, coefvar = NULL, rho = NULL, alternative = c("two.sided", "less", "greater"), delta = 0, conf.level = 0.95, dec = 2, random = FALSE){

  if(length(n) > 1)
    stop("n must be an integer")

  if(any(length(mean), length(sigma), length(coefvar)) > 2)
    stop("Some parameter has size more than two")

  mean <- rep(mean, 2)[1:2]
  sigma <- rep(sigma, 2)[1:2]
  coefvar <- rep(coefvar,2)[1:2]

  if(!is.null(coefvar)){
    if (any(mean == 0))
      stop("Parameter coefvar is given. Vector of means cannot have zeros.")
    else
      sigma <- coefvar*abs(mean)
  }

  if(random == TRUE){
    message("Random covariance matrix generated. Vector of standard deviations omitted")
    randm <- clusterGeneration::genPositiveDefMat(2, covMethod = "unifcorrmat", rangeVar = c(min(sigma^2), max(sigma^2)))
    mcov<- randm$Sigma
  }

  else{

    if(is.null(rho)){
     rho <- stats::runif(1,-1,1)
    }

    mcorr <- matrix(c(1, rho, rho, 1), nrow = 2)
    mcov <- mCorrCov(mcorr, sigma)

  }


  gen <- generator(n, mean, sigma = mcov, dec = dec)
  d <- gen$Samples

  #Regression model

  model <- stats::lm(X2 ~ X1, data = d)
  s.model <- summary(model)

  #T-test
  test <- stats::t.test(d$X1, d$X2, alternative = alternative[1], paired = TRUE, mu = delta, conf.level = conf.level)


  return(list(Data = d, Regression = s.model, T.Test = test))
}

