% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sample_index.r
\name{sample_index}
\alias{sample_index}
\title{Sample the biomass with observation error}
\usage{
sample_index(dat_list, outfile = NULL, fleets, years, sds_obs,
  make_plot = FALSE)
}
\arguments{
\item{dat_list}{An SS data list object as read in from
\code{\link[r4ss]{SS_readdat}} in the \pkg{r4ss} package. Make sure
you select option \code{section=2}.}

\item{outfile}{A character string specifying the file name to use
when writing the information to the disk. The string must include
the proper file extension. No file is written using the default value
of \code{NULL}, which leads to increased speed because writing the 
file takes time and computing resources.}

\item{fleets}{*A vector of integers specifying which fleets to include.
The order of the fleets pertains to the input order of other arguments.
An entry of \code{fleets=NULL} leads to zero samples for any fleet.}

\item{years}{*A list the same length as \code{fleets} giving the years as
numeric vectors. If no fleet collected samples, keep the value to
\code{years=NULL}.}

\item{sds_obs}{*A list the same length as \code{fleets}. The list should
contain either single values or numeric vectors of the same length as the
number of years which represent the standard deviation of the observation
error. Single values are repeated for all years.}

\item{make_plot}{A logical switch for whether to make a crude plot showing
the results. Useful for testing and exploring the function.}
}
\value{
A modified \code{.dat} file if \code{!is.null(outfile)}. A list object
containing the modified \code{.dat} file is returned invisibly.
}
\description{
This function creates an index of abundance sampled from the expected
available biomass for given fleets in given years. Let \code{B_y} be the biomass
from the operating model for year y. Then the sampled value is calculated as:
\code{B_y*exp(rnorm(1, 0, sds_obs)-sds_obs^2/2)}. The second term
adjusts the random samples so that their expected value is \code{B_y} (i.e. the
log-normal bias correction).
If used with \code{\link{run_ss3sim}} the case file should be named
\code{index}. A suggested (default) case letter is \code{D} for data.
}
\section{Which arguments to specifiy in case files}{

All function argument descriptions that start with an asterisk (*) will be passed
through the case files to \code{\link{run_ss3sim}}. If one of these arguments
is not specified in a case file, then a value of \code{NULL} will be passed,
which may or may not be an appropriate value. Other arguments will be ignored
if specified.
}

\examples{
\dontrun{
# Find the example data location:
d <- system.file("extdata", package = "ss3sim")
f_in <- file.path(d, "example-om", "ss3_expected_values.dat")
dat_list <- r4ss::SS_readdat(f_in, version = NULL, verbose = FALSE)
# Note the initial expected values for the index data:
dat_list$CPUE # Only has expected values for fleet 2 in every other year from
# 76 to 100, so can only sample from fleet 2 during every other year between
# 76 and 100
sam_yrs <- seq(76, 100, by = 2)
ex1 <- sample_index(dat_list,
                   outfile = NULL,
                   fleets = 2,
                   years = list(sam_yrs),
                   sds_obs=list(seq(.001, .1,
                     length.out = length(sam_yrs))))
ex1$CPUE
# could sample from less years, but not more:
ex2 <- sample_index(dat_list,
                    outfile = NULL,
                    fleets = 2,
                    years = list(sam_yrs[c(-1, -2)]),
                    sds_obs=list(seq(.001, .1,
                      length.out = length(sam_yrs[c(-1, -2)]))))
ex2$CPUE
# Also, sd can be fixed across years:
ex3 <- sample_index(dat_list,
                    outfile = NULL,
                    fleets = 2,
                    years = list(sam_yrs),
                    sds_obs=list(0.01))
ex3$CPUE
# If fleet 1 also had expected values in the index that you wanted to sample:
# ex4 <- sample_index(dat_list,
#                     outfile = NULL,
#                     fleets = c(1,2),
#                     years = list(sam_yrs, sam_yrs),
#                     sds_obs=list(0.01, 0.01))
}
}
\seealso{
Other sampling functions: \code{\link{clean_data}},
  \code{\link{sample_agecomp}},
  \code{\link{sample_calcomp}}, \code{\link{sample_lcomp}},
  \code{\link{sample_mlacomp}},
  \code{\link{sample_wtatage}}
}
\author{
Cole Monnahan, Kotaro Ono
}
\concept{sampling functions}
