% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit.mean.R
\name{fit.mean}
\alias{fit.mean}
\title{Kernel smoothing with additional constraints}
\usage{
fit.mean(X.eval, y.eval, X.constrained = NA, X.fit = NA, H.inv = NA,
  H.mult = 1, method = "u", scale.constraints = TRUE)
}
\arguments{
\item{X.eval}{Matrix of inputs used for fitting}

\item{y.eval}{Vector of outputs used for fitting}

\item{X.constrained}{Matrix of inputs where constraints apply}

\item{X.fit}{Matrix of inputs where curve is fit; defaults to X.constrained}

\item{H.inv}{Inverse of the smoothing matrix (must be positive definite); defaults to rule of thumb}

\item{H.mult}{Scaling factor for rule of thumb smoothing matrix}

\item{method}{Constraints to apply; "u" for unconstrained, "m" for monotonically increasing, and "mc" for monotonically increasing and concave}

\item{scale.constraints}{Boolean, whether to scale constraints by their average value, can help with convergence}
}
\value{
Returns a list with the following elements
\item{y.fit}{Estimated value of the frontier at X.fit}
\item{gradient.fit}{Estimated gradient of the frontier at X.fit}
\item{solution}{Boolean; TRUE if frontier successfully estimated}
\item{X.eval}{Matrix of inputs used for fitting}
\item{X.constrained}{Matrix of inputs where constraints apply}
\item{X.fit}{Matrix of inputs where curve is fit}
\item{H.inv}{Inverse smoothing matrix used in fitting}
\item{method}{Method used to fit frontier}
\item{scaling.factor}{Factor by which constraints are multiplied before quadratic programming}
}
\description{
Fits conditional mean of data with kernel smoothing, imposing monotonicity and/or concavity constraints.
}
\details{
This method uses kernel smoothing to fit the mean of the data
while imposing specified monotonicity and concavity constraints. The procedure is
derived from Racine et al. (2009), which develops kernel smoothing methods with
bounding, monotonicity and concavity constraints. Specifically, the smoothing procedure
involves finding optimal weights for a Nadaraya-Watson estimator of the form 

\deqn{\hat{y} = m(x) = \sum_{i=1}^N p_i A(x, x_i) y_i,}

where \eqn{x} are inputs, \eqn{y} are outputs, \eqn{p} are weights, subscripts
index observations, and 

\deqn{A(x, x_i) = \frac{K(x, x_i)}{\sum_{h=1}^N K(x, x_h)}}

for a kernel \eqn{K}. This method uses a multivariate normal kernel of the form

\deqn{K(x, x_h) = \exp\left(-\frac12 (x - x_h)'H^{-1}(x - x_h)\right),}

where \eqn{H} is a bandwidth matrix. Bandwidth selection is performed via Silverman's
(1986) rule-of-thumb, in the function \code{H.inv.select}.

Optimal weights \eqn{\hat{p}} are selected by solving the quadratic programming problem

\deqn{\min_p \mbox{\ \ }-\mathbf{1}'p + \frac12 p'p.}

Monotonicity constraints of the following form can be imposed at 
specified points:

\deqn{\frac{\partial m(x)}{\partial x^j} = \sum_{h=1}^N p_h \frac{\partial A(x, x_h)}{\partial x^j} y_h \geq 0 \mbox{\ \ \ \ }\forall x, j,}

where superscripts index inputs. Finally concavity constraints of the following form can also be imposed using Afriat's
(1967) conditions:

\deqn{m(x) - m(z) \leq \nabla_x m(z) \cdot (x - z) \mbox{\ \ \ \ }\forall x, z.}

The gradient of the estimated curve at a point \eqn{x} is given by 

\deqn{\nabla_x m(x) = \sum_{i=1}^N \hat{p}_i \nabla_x A(x, x_i) y_i,}

where \eqn{\hat{p}_i} are estimated weights.
}
\examples{
data(USMacro)

USMacro <- USMacro[complete.cases(USMacro),]

#Extract data
X <- as.matrix(USMacro[,c("K", "L")])
y <- USMacro$Y

#Reflect data for fitting
reflected.data <- reflect.data(X, y)
X.eval <- reflected.data$X
y.eval <- reflected.data$y

#Fit frontier
fit.mc <- fit.mean(X.eval, y.eval, 
                   X.constrained = X,
                   X.fit = X,
                   method = "mc")

#Plot input productivities over time
library(ggplot2)
plot.df <- data.frame(Year = rep(USMacro$Year, times = 2),
                      Elasticity = c(fit.mc$gradient.fit[,1] * X[,1] / y,
                                     fit.mc$gradient.fit[,2] * X[,2] / y),
                      Variable = rep(c("Capital", "Labor"), each = nrow(USMacro)))

ggplot(plot.df, aes(Year, Elasticity)) +
  geom_line() +
  facet_grid(Variable ~ ., scales = "free_y")
  
}
\references{
\insertRef{ParmeterRacine}{snfa}
}
