\name{slp.mgcv}
\alias{slp.mgcv}
\alias{smooth.construct.slp.smooth.spec}
\alias{slp.smooth.spec}
\title{Generate a Basis Matrix for Discrete Prolate Spheroidal (Slepian) Sequences}
\description{
  Generate the basis matrix for a particular \code{N, W} Slepian sequence
  family member, with the additional property that the smoother captures/passes constants
  without distortion. Can be quite slow in execution due to the latter property.

  Based on \code{\link[mgcv]{smooth.construct.cr.smooth.spec}} for implementation 
  with \code{\link[mgcv:mgcv-package]{mgcv}}. 
}
\usage{
\method{smooth.construct}{slp.smooth.spec}(object,data,knots)
}
\arguments{
  \item{object}{a smooth specification object, usually generated by a model term \code{s(..., bs = 'slp', ..., xt = list(...))},
       and for this type, \bold{requiring} an additional \code{xt = list()} object containing parameters.
       For examples, see below. }
  \item{data}{a list containing just the data required by this term, with names corresponding to
       \code{object[['term']]}. Typically just a single time index array.}
  \item{knots}{a list containing any knots supplied for basis setup -- should be \code{NULL}, as
       slp basis objects are not generated from knots.}
}
\value{
    An object of class \code{slp.smooth}. In addition to the usual elements of a smooth
    class (see \code{\link[mgcv:smooth.construct]{smooth.construct}}), this object will
    contain:
    \item{C}{a constraint matrix which restricts \code{mgcv} from modifying the (already) 
        orthogonal and mean-passing/capturing basis vectors. }
    \item{K}{the user-specified number of basis vectors, or the computed \code{K} from
        user-supplied \code{W}.}
    \item{W}{the user-specified bandwidth \code{W}, or the computed \code{W} from
        user-supplied \code{K}.}
    \item{fullBasis}{the full-span computed, normalized basis set, before contiguity is
        taken into account. Used by \code{\link[mgcv:Predict.matrix]{predict}} 
        when given an object of type \code{slp.smooth}. }
    \item{contiguous}{a logical variable declaring whether or not the input time array
        was considered to be contiguous by the basis computation procedure.}
    \item{wx}{the ``corrected'' input time array; if \code{contiguous == FALSE} then 
        this will be the same as \code{data[object[['term']]]}. }
}
\details{
  \code{slp} is based on \code{\link{.dpss}}, which generates a family of Discrete
  Prolate Spheroidal (Slepian) Sequences. These vectors are orthonormal, have alternating
  even/odd parity, and form the optimally concentrated basis set for the subspace of 
  \code{R^N} corresponding to the bandwidth \code{W}. Full details are given
  in Slepian (1978).  These basis functions have natural boundary conditions, and lack any form of 
  knot structure. This version is returned for \code{naive = TRUE}.

  The \code{dpss} basis vectors can be adapted to provide the additional
  useful property of capturing or passing constants perfectly. That is, the smoother matrix
  \code{S} formed from the returned rectangular matrix will either reproduce constants
  at near round-off precision, i.e., \code{S \%*\% rep(1, N) = rep(1, N)}, 
  for \code{naive = FALSE} with \code{intercept = TRUE}, or will pass constants, 
  i.e., \code{S \%*\% rep(1, N) = rep(0, N)}, for \code{naive = FALSE} with \code{intercept = FALSE}. 

  The primary use is in modeling formula to directly specify a Slepian time-based smoothing
  term in a model: see the examples.

  For large \code{N} this routine can be \bold{very} slow. If you are computing models with
  large \code{N}, we highly recommend pre-computing the basis object, then using it
  in your models without recomputation. The third example below demonstrates this approach.
}
\seealso{
  \code{\link[mgcv:smooth.construct]{smooth.construct}} 
}
\references{
  Wood S.N. (2006) Generalized Additive Models: An Introduction with 
  R. Chapman and Hall/CRC Press.

  Hastie T.J. & Tibshirani, R.J. (1990) Generalized Additive Models.
  Chapman and Hall.

  Thomson, D.J (1982)
  Spectrum estimation and harmonic analysis. \emph{Proceedings of the IEEE}.
  Volume \bold{70}, number 9, pp. 1055-1096.

  Slepian, David (1978)
  Prolate Spheroidal Wave Functions, Fourier Analysis, and Uncertainty V: the Discrete Case.
  \emph{Bell System Technical Journal}. Volume \bold{57}, pp. 1371-1429.
}
\examples{
    # Examples using pkg:mgcv
    library("mgcv")
    library("slp")

    N <- 730
    W <- 8 / N
    K <- 16         # will actually use 15 df as intercept = FALSE
    x <- rnorm(n = N, sd = 1)
    y <- x + rnorm(n = N, sd = 2) + 5.0
    t <- seq(1, N)    

    # note: all three examples share identical results

    # example with in-call computation, using K (df)
    fit1 <- gam(y ~ x + s(t, bs = 'slp', xt = list(K = K)), family = gaussian) 

    # example with in-call computation, using W
    fit2 <- gam(y ~ x + s(t, bs = 'slp', xt = list(W = W)), family = gaussian)

}
\keyword{models}
\keyword{smooth}
\keyword{regression}
\keyword{Slepian}
\keyword{mgcv}
