% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/powerShapeNA.R
\name{powerShapeNA}
\alias{powerShapeNA}
\alias{classicShapeNA}
\alias{tylerShapeNA}
\title{M-estimators of the Shape from the Power Family when Data is Missing}
\usage{
powerShapeNA(x, alpha, center = NULL, normalization = c("det", "trace", "one"),
         maxiter = 1e4, eps = 1e-6)

tylerShapeNA(x, center = NULL, normalization = c("det", "trace", "one"),
          maxiter = 1e4, eps = 1e-6)

classicShapeNA(x, center = NULL, normalization = c("det", "trace", "one"),
         maxiter = 1e4, eps = 1e-6)
}
\arguments{
\item{x}{A data matrix or data.frame with missing data and \code{p} > 2 columns.}

\item{alpha}{Tail index, a numeric value in the interval \code{[0, 1]}.
Determines the power function. For more information see 'Details'.}

\item{center}{An optional vector of the data's center, if \code{NULL} the center
will be estimated simultaneously with the shape.}

\item{normalization}{A string determining how the shape matrix is standardized.
The possible values are
\itemize{
\item{\code{'det'}}{such that the returned shape estimate has determinant 1.}
\item{\code{'trace'}}{such that the returned shape estimate has trace \code{ncol(x)}.}
\item{\code{'one'}}{such that the returned shape estimate's top left entry
(\code{S[1, 1]}) is 1.}
}}

\item{maxiter}{A positive integer, restricting the maximum number of iterations.}

\item{eps}{A numeric, specifying tolerance level of when the iteration stops.}
}
\value{
A list with class 'shapeNA' containing the following components:
\describe{
\item{S}{The estimated shape matrix.}
\item{scale}{The scale with which the shape matrix may be scaled to obtain
a scatter estimate. If \code{alpha} = 1, then this value will be \code{NA}, as
Tyler's shape matrix has no natural scale.}
\item{mu}{The location parameter, either provided by the user or estimated.}
\item{alpha}{The tail index with which the Power M-estimator has been called.}
\item{naBlocks}{An \code{naBlocks} object, with information about the missingness
of the data.}
\item{iterations}{Number of computed iterations before convergence.}
\item{call}{The matched call.}
}
}
\description{
Power M-estimators of shape and location were recently suggested in
Frahm et al. (2020). They have a tuning parameter \code{alpha} taking values in
\code{[0,1]}. The extreme case \code{alpha} = 1 corresponds to Tyler's shape
matrix and \code{alpha} = 0 to the classical covariance matrix. These special
cases have their own, more efficient functions \code{\link{tylerShapeNA}} and
\code{\link{classicShapeNA}}, respectively.
If the true location is known, it should be supplied as \code{center}, otherwise
it is estimated simultaneously with the shape.
}
\details{
These functions assume that the data were generated from an
elliptical distribution, for Tyler's estimate this can be relaxed to
generalized elliptical distributions
The missingness mechanism should be MCAR or, under
stricter distributional assumptions, MAR. See the references for details.

For multivariate normally distributed data, \code{\link{classicShapeNA}} is the maximum
likelihood estimator of the location and scale. It is a special case of the
power M-estimator with tail index \code{alpha} = 0, which returns the
empirical covariance matrix and the empirical mean vector.

The function \code{\link{tylerShapeNA}} maximizes the likelihood function after projecting
the observed data of each individual onto the unit hypersphere, in which case
we obtain an angular central Gaussian distribution. It is a special case of
the power M-estimator with tail index \code{alpha} = 1, which returns Tyler's
M-estimator of scatter and an affine equivariant multivariate median
according to Hettmansperger and Randles (2002).

The function \code{\link{powerShapeNA}} requires an additional parameter, the so-called
tail index \code{alpha}. For heavy tailed data, the index should be chosen closer
to 1, whereas for light tailed data the index should be chosen closer to 0.
}
\examples{
    ## Generate a data set with missing values
    x <- mvtnorm::rmvt(100, toeplitz(seq(1, 0.1, length.out = 3)), df = 5)
    y <- mice::ampute(x, mech = 'MCAR')$amp

    ## Compute some M-estimators
    res0 <- classicShapeNA(y, center = c(0, 0, 0))
    res1 <- powerShapeNA(y, alpha = 0.67, normalization = 'one')
    res2 <- tylerShapeNA(y, normalization = 'trace')

    ## Get location estimates
    res1$mu
    res2$mu
    ## Get shape estimates
    res0$S
    res1$S
    res2$S

    ## Print summary
    summary(res0)
    ## Inspect missingness pattern
    plot(res0$naBlocks)
    barplot(res0)
}
\references{
Frahm, G., & Jaekel, U. (2010). A generalization of Tyler's M-estimators to the case of incomplete data. Computational Statistics & Data Analysis, 54, 374-393. \doi{10.1016/j.csda.2009.08.019}.

Frahm, G., Nordhausen, K., & Oja, H. (2020). M-estimation with incomplete and dependent multivariate data. Journal of Multivariate Analysis, 176, 104569. \doi{10.1016/j.jmva.2019.104569}.

Hettmansperger, T. P., & Randles, R. H. (2002). A practical affine equivariant multivariate median. Biometrika, 89(4), 851-860. \doi{10.1093/biomet/89.4.851}
}
\seealso{
\link{powerShape}, \link{tylerShape} and \link{classicShape} for the
corresponding functions for data without missing values.
}
