% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/STBP.R
\name{stbp_composite}
\alias{stbp_composite}
\title{Sequential test of Bayesian posterior probabilities for composite hypotheses}
\usage{
stbp_composite(
  data,
  greater_than = TRUE,
  hypothesis,
  density_func,
  overdispersion = NA,
  prior = 0.5,
  lower_bnd = 0,
  upper_bnd = Inf,
  lower_criterion,
  upper_criterion
)
}
\arguments{
\item{data}{For count data, either a vector (for purely sequential designs)
or a matrix (group sequential designs) with sequential (non-negative) count
data, with sampling bouts collected over time in columns and samples within
bouts in rows. NAs are allowed in case sample size within bouts is unbalanced.
For binomial data, a list of matrices with integer non-negative values of
observations in col 1 and number of samples in col 2, so that each matrix
within the list corresponds to a sampling bout. NAs are \emph{not} allowed for
binomial data.}

\item{greater_than}{logical; if TRUE (default), the tested hypothesis is of
the form \eqn{H:\mu > \psi} otherwise, \eqn{H:\mu < \psi}.}

\item{hypothesis}{Either a single non-negative value or a vector of
non-negative values with the hypothesized population densities, \eqn{\psi}.
If a vector, it should contain at least as many values as \code{ncol(data)}
for count data or as \code{length(data)} for binomial data.}

\item{density_func}{Kernel probability density function for the data. See
details.}

\item{overdispersion}{A character string (if a function) or a number
specifying the overdispersion parameter. Only required when using
\code{"negative binomial"} or \code{"beta-binomial"} as kernel densities.
See details.}

\item{prior}{Single number with initial prior. Must be on the interval
\eqn{[0,1]}. If no prior information is available 0.5 (default) is recommended.}

\item{lower_bnd}{Single number indicating the lower bound of the parameter
space for \eqn{\mu}. Most cases is \eqn{0} (default).}

\item{upper_bnd}{Single number indicating the upper bound of the parameter
space for \eqn{\mu}. For count data, is often \code{Inf} (default), but it
must be \eqn{\leq 1} for binomial data.}

\item{lower_criterion}{Criterion to decide against the tested hypothesis.
This is the maximum credibility to the hypothesis to stop sampling and
decide against.}

\item{upper_criterion}{Criterion to decide in favor of the tested hypothesis.
This is the minimum credibility to the hypothesis to stop sampling and
decide in favor.}
}
\value{
An object of class \code{"STBP"}.
}
\description{
Runs a Sequential test of Bayesian Posterior Probabilities for hypotheses
about population densities of the form \eqn{H:\mu > \psi} or \eqn{H:\mu < \psi}.
Data is treated in a sequential framework.
}
\details{
The \code{density_func} argument should be specified as character string.
Acceptable options are \code{"poisson"}, \code{"negative binomial"},
\code{"binomial"} and \code{"beta-binomial"}. The overdispersion
parameter for \code{"negative binomial"} and \code{"beta-binomial"} can be
either a constant or a function of the mean.

If a function, it should be specified as a character string with the name of
an existing function. For options of empirical functions to describe
overdispersion as a function of the mean see Binns et al. (2000). The most
common approach for the negative binomial family is Taylor's Power Law, which
describes the variance as a function of the mean with two parameters, \eqn{a}
and \eqn{b}. Overdispersion, \eqn{k}, can then be specified as: \deqn{k = \frac{\mu^2}{a \mu^b - \mu}}
}
\examples{
# Testing the hypothesis of a population size being greater than 5 individuals
# per sampling unit (H: mu > 5). The sequential sampling is made of 5 sampling
# bouts made of one sample each.

set.seed(101)
counts3 <- rpois(5, lambda = 3)

test1F <- stbp_composite(data = counts3,
                          greater_than = TRUE,
                          hypothesis = 5,
                          density_func = "poisson",
                          prior = 0.5,
                          lower_bnd = 0,
                          upper_bnd = Inf,
                          lower_criterion = 0.001,
                          upper_criterion = 0.999)
test1F

# returns "reject H".

# Testing the hypothesis of a sampled population being greater than trajectory H
H <- c(2, 5, 10, 20, 40, 40, 20, 10, 5, 2)

# Generating sequential samples (n = 3) from a population that is 1 below H
# (H - 1)

countP <- matrix(NA, 3, 10)
set.seed(101)
for(i in 1:10){
  countP[, i] <- rpois(3, lambda = (H[i] - 1))
}

# Running STBP on the sample

test2F <- stbp_composite(data = countP,
                          greater_than = TRUE,
                          hypothesis = H,
                          density_func = "poisson",
                          prior = 0.5,
                          lower_bnd = 0,
                          upper_bnd = Inf,
                          lower_criterion = 0.001,
                          upper_criterion = 0.999)
test2F

# returns "reject H".

# Testing the hypothesis of a proportion of infested units being greater than
# 20\% per sampling unit (H: mu > 0.2). The sequential sampling is made of 7
# sampling bouts each with 5 clusters of 10 samples from which binomial
# observations are recorded.

set.seed(101)

# binomial data generated with mu (prob) 0.05 over the hypothesized
# value (0.2)

counts4 <- list()
for(i in 1: 7) {
  counts4[[i]] <- matrix(c(rbinom(5, size = 10, prob = 0.25), rep(10, 5)),
                        5, 2)
}

# Run the test. Notice that upper_bnd = 1!

test3F <- stbp_composite(data = counts4,
                          greater_than = TRUE,
                          hypothesis = 0.2,
                          density_func = "binomial",
                          prior = 0.5,
                          lower_bnd = 0,
                          upper_bnd = 1,
                          lower_criterion = 0.001,
                          upper_criterion = 0.999)

test3F # returns accept H after 3 sampling bouts

# Assuming a negative binomial count variable whose overdispersion parameter,
# k, varies as a function of the mean, and that the variance-mean relationship
# is well described with Taylor's Power Law, a function to obtain k can be:

estimate_k <- function(mean) {
                        a = 1.830012
                        b = 1.218041 # a and b are Taylor's Power Law parameters
                        (mean^2) / ((a * mean^(b)) - mean)
                        }

# Generate some counts to create an STBP object with the model specifications

counts3 <- rnbinom(20, mu = 5, size = estimate_k(5))

# Run the test to create the STBP object

test1F <- stbp_composite(data = counts3,
                          greater_than = TRUE,
                          hypothesis = 9,
                          density_func = "negative binomial",
                          overdispersion = "estimate_k",
                          prior = 0.5,
                          lower_bnd = 0,
                          upper_bnd = Inf,
                          lower_criterion = 0.01,
                          upper_criterion = 0.99)

test1F

## End (Not run)
}
\references{
Binns, M.R., Nyrop, J.P. & Werf, W.v.d. (2000) \emph{Sampling and
monitoring in crop protection: the theoretical basis for developing practical
decision guides}. CABI Pub., Wallingford, Oxon, UK; New York, N.Y.

Rincon, D.F., McCabe, I. & Crowder, D.W. (2025) Sequential testing of
complementary hypotheses about population density. \emph{Methods in Ecology
and Evolution}. <https://doi.org/10.1111/2041-210X.70053>
}
