################################################################################
# ----------------------------- CHECK FUNCTIONS ------------------------------ #
################################################################################

#' Input check
#' 
#' Checks that the input is a numeric vector, and throws an error if it is not. 
#' Converts the input to a plain numeric vector.
#' 
#' @param x     Input to check.
#' @param name  String. Name of the variable for error messages (default: de-
#'              parsed expression of \code{x}).
#' 
#' @return \code{x} as numeric vector.
#' 
#' @keywords internal
#' @noRd
check_numeric_vector <- function(x, name = deparse(substitute(x))) {
  if (!is.numeric(x) || 
      (length(dim(x)) > 2) || 
      (length(dim(x)) == 2 && !any(dim(x) == 1))) {
    stop(sprintf("'%s' must be a numeric vector.", name))
  }
  as.numeric(x)
}

#' Input check
#' 
#' Checks that the input is an integer, and throws an error if it is not.
#' 
#' @param x     Input to check.
#' @param name  String. Name of the variable for error messages (default: de-
#'              parsed expression of \code{x}).
#' 
#' @return TRUE
#' 
#' @keywords internal
#' @noRd
check_integer_scalar <- function(x, name = deparse(substitute(x))) {
  if (!is.numeric(x) || length(x) != 1 || x != as.integer(x) || x<1) {
    stop(sprintf("'%s' must be a natural number.", name))
  }
  invisible(TRUE)
}


#' Input check
#' 
#' Checks that the input is a number between 0 and 1, and throws an error if it 
#' is not.
#' 
#' @param x     Input to check.
#' @param name  String. Name of the variable for error messages (default: de-
#'              parsed expression of \code{x}).
#' 
#' @return TRUE
#' 
#' @keywords internal
#' @noRd
check_significance_level <- function(x, name = deparse(substitute(x))) {
  if (!is.numeric(x) || length(x) != 1 || x < 0 || x > 1) {
    stop(sprintf("'%s' must be a number between 0 and 1.", name))
  }
  invisible(TRUE)
}

#' Input check
#' 
#' Checks that the input is a valid penalisation order, and throws an error if 
#' it is not.
#' 
#' @param x     Input to check.
#' @param name  String. Name of the variable for error messages (default: de-
#'              parsed expression of \code{x}).
#' 
#' @return TRUE
#' 
#' @keywords internal
#' @noRd
check_penalisation_order <- function(x, name = deparse(substitute(x))) {
  check_integer_scalar(x, name)
  if (!(x %in% c(1,2,3,4,5,6))) {
    stop(sprintf("'%s' can only attain the values 1,2,3,4,5,6.", name))
  }
  invisible(TRUE)
}

#' Input check
#' 
#' Checks that the input is a valid method, and throws an error if it is not. If
#' the method is set to "\code{GCV-oracle}" or "\code{ML-oracle}", checks if the
#' true regression function is given.
#' 
#' @param x     Input to check.
#' @param namex String. Name of the variable for error messages (default: de-
#'              parsed expression of \code{x}).
#' @param f     Input to check. True regress function.
#' @param namef String. Name of the variable for error messages (default: de-
#'              parsed expression of \code{f}).
#' 
#' @return TRUE
#' 
#' @keywords internal
#' @noRd
check_method <- function(x, namex = deparse(substitute(x)), 
                         f, namef = deparse(substitute(f))) {
  if (! x %in% c("GCV","GCV-oracle","ML","ML-oracle")) {
    stop(sprintf("'%s' must be one of: GCV, GCV-oracle, ML, ML-oracle.", namex))
  }
  if (x %in% c("GCV-oracle","ML-oracle")) {
    check_numeric_vector(f, namef)
  }
  invisible(TRUE)
}