#' Detect malformed Quarto website projects generated by saros.base
#'
#' This function checks for common issues in a Quarto website project directory.
#' Currently, it finds:
#'   - Subfolders lacking index.qmd
#'   - .qmd files without a title in their YAML front matter
#' Future versions will add more checks.
#'
#' @param root_dir Path to the root directory of the Quarto website project.
#' @return A data.frame with columns: type, path, details
#' @export
#'
#' @examples
#' detect_malformed_quarto_project(tempdir())
detect_malformed_quarto_project <- function(root_dir) {
  if (!dir.exists(root_dir)) {
    cli::cli_abort("Directory '{.path {root_dir}}' does not exist.")
  }
  issues <- list()

  # Find subfolders lacking index.qmd
  subfolders <- list.dirs(root_dir, recursive = TRUE, full.names = TRUE)
  for (folder in subfolders) {
    index_path <- file.path(folder, "index.qmd")
    if (!file.exists(index_path)) {
      issues[[length(issues) + 1]] <- list(
        type = "missing_index.qmd",
        path = folder,
        details = "No index.qmd in subfolder"
      )
    }
  }

  # Find .qmd files without a title in YAML front matter
  qmd_files <- list.files(root_dir, pattern = "\\.qmd$", recursive = TRUE, full.names = TRUE)
  qmd_files <- qmd_files[!grepl("/_.+\\.qmd", x = qmd_files)]
  for (qmd in qmd_files) {
    lines <- readLines(qmd, warn = FALSE)
    # Look for YAML front matter
    # Robust YAML block extraction: find first and second --- markers
    yaml_start <- which(lines == "---")[1]
    yaml_end <- which(lines == "---" & seq_along(lines) > yaml_start)[1]
    if (!is.na(yaml_start) && !is.na(yaml_end) && yaml_end > yaml_start + 1) {
      yaml_lines <- lines[(yaml_start + 1):(yaml_end - 1)]
      yaml_text <- paste(yaml_lines, collapse = "\n")
      yaml_obj <- tryCatch(yaml::yaml.load(yaml_text), error = function(e) NULL)
      has_title <- !is.null(yaml_obj) && !is.null(yaml_obj$title)
      if (!has_title) {
        issues[[length(issues) + 1]] <- list(
          type = "missing_title",
          path = qmd,
          details = "No title in YAML front matter"
        )
      }
    } else if (!is.na(yaml_start) && is.na(yaml_end)) {
      # No closing --- for YAML block
      issues[[length(issues) + 1]] <- list(
        type = "missing_title",
        path = qmd,
        details = "No closing --- for YAML front matter"
      )
    } else {
      # No YAML front matter at all
      issues[[length(issues) + 1]] <- list(
        type = "missing_title",
        path = qmd,
        details = "No YAML front matter"
      )
    }
  }

  # Return as data.frame
  if (length(issues) == 0) {
    return(data.frame(type = character(), path = character(), details = character(), stringsAsFactors = FALSE))
  }
  do.call(rbind, lapply(issues, as.data.frame, stringsAsFactors = FALSE))
}
