\name{fitsaemodel}
\alias{fitsaemodel}
\alias{print.fit_model_b}
\alias{print.summary_fit_model_b}
\alias{summary.fit_model_b}
\alias{coef.fit_model_b}
\alias{convergence}
\title{Fitting SAE Models}
\description{
    \code{fitsaemodel} fits SAE models that have been specified by
    \code{\link[=saemodel]{saemodel()}} (or synthetic data generated by
    \code{\link[=makedata]{makedata()}}) for various (robust) estimation
    methods.
}
\usage{
fitsaemodel(method, model, ...)
convergence(object)

\method{print}{fit_model_b}(x, digits = max(3L, getOption("digits") - 3L),
    ...)
\method{print}{summary_fit_model_b}(x, digits = max(3L, getOption("digits")
    - 3L), ...)
\method{summary}{fit_model_b}(object, ...)
\method{coef}{fit_model_b}(object, type = "both", ...)
}
\arguments{
    \item{method}{\code{[character]} estimation method; \code{method = "ml"}
        for (non-robust) maximum likelihood or \code{method = "huberm"}
        for Huber-type M-estimation.}
    \item{model}{an object of class \code{"saemodel"}, i.e., a SAE model;
        see \code{\link[=saemodel]{saemodel()}}.}
    \item{digits}{\code{[integer]} number of digits printed by the
        \code{print()} and \code{summary()} methods.}
    \item{object}{an object of class \code{"fit_model_b"}.}
    \item{x}{an object of class \code{"fit_model_b"}.}
    \item{type}{\code{[character]} name of the effects to be extracted
        by the \code{coef} method; it can take one of the
        following possibilities: \code{"both"} (extracts fixed and random
        effects; default), \code{"ranef"} (only random effects), or
        \code{"fixef"} (only fixed effects).}
    \item{\dots}{additional arguments passed on to
        \code{\link[=fitsaemodel.control]{fitsaemodel.control()}}.}
}
\details{
    Function \code{fitsaemodel()} computes the following estimators:
    \itemize{
        \item maximum likelihood (ML): \code{method = "ml"},
        \item Huber-type M-estimation: \code{method = "huberm"}; this
            method is called RML II by Richardson and Welsh (1995); see
            Schoch (2012)
    }
    \subsection{Maximum likelihood}{
        The ML is \bold{not} robust against outliers.
    }
    \subsection{Huber-type M-estimation}{
        The call for the Huber-type M-estimator (with Huber psi-function) is:
        \code{fitsaemodel(method = "huberm", model, k)}, where \code{k} is
        the robustness tuning constant of the Huber psi-function,
        \eqn{k \in (0, \infty]}{k in (0, Inf]}.

        By default, the computation of the M-estimator is initialized by a
        robust estimate that derives from a fixed-effects model (centered by
        the median instead of the mean); see Schoch (2012) for the details.

        If the data are supposed to be heavily contaminated (\bold{or} if the
        \bold{default algorithm did not converge}), one may try to initialize
        the algorithm underlying \code{fitsaemodel()} by a high breakdown-point
        estimate. The package offers two initialization methods:
        \bold{NOTE:} the \pkg{robustbase} package (Maechler et al., 2022)
        must be installed to use this functionality.
        \itemize{
            \item \code{init = "lts"}: least trimmed squares (LTS)
                regression from \pkg{robustbase}; see
                \link[robustbase:ltsReg]{ltsReg()} and Rousseeuw and Van
                Driessen (2006),
            \item \code{init = "s"}: regression S-estimator from
                \pkg{robustbase}; see \link[robustbase:lmrob]{lmrob()} and
                Maronna et al. (2019).
        }
        For small and medium size datasets, both methods are equivalent in
        terms of computation time. For large data, the S-estimator is
        considerably faster.
    }
    \subsection{Implementation}{
        The methods are computed by (nested) iteratively re-weighted least
        squares and a derivative of Richard Brent's \code{zeroin} algorithm;
        see Brent (2013, Chapter 4). The functions depend on the subroutines in
        BLAS (Blackford et al., 2002) and LAPACK (Anderson et al., 2000); see
        Schoch (2012).
    }
}
\value{
    An instance of the class \code{"fitmodel"}
}
\references{
Anderson, E., Bai, Z., Bischof, C., Blackford, L. S., Demmel, J., Dongarra,
    J., et al. (2000). \emph{LAPACK users' guide} (3rd ed.). Philadelphia:
    Society for Industrial and Applied Mathematics (SIAM).

Blackford, L.S., Petitet, A., Pozo, R., Remington, K., Whaley, R.C.,
    Demmel, J., et al. (2002).
    An updated set of basic linear algebra subprograms (BLAS).
    \emph{ACM Transactions on Mathematical Software} \bold{28}, 135--151.
    \doi{https://doi.org/10.1145/567806.567807}

Brent, R.P. (2013). \emph{Algorithms for minimization without derivatives}.
    Mineola (NY): Dover Publications Inc. (This publication is an unabridged
    republication of the work originally published by Prentice-Hall Inc.,
    Englewood Cliffs, NJ, in 1973).

Maechler, M., Rousseeuw, P., Croux, C., Todorov, V., Ruckstuhl, A.,
    Salibian-Barrera, M., Verbeke, T., Koller, M., Conceicao, E.L.T. and
    M. Anna di Palma (2022). robustbase: Basic Robust Statistics R package
    version 0.95-0. \url{https://CRAN.R-project.org/package=robustbase}

Maronna, R.A., Martin, D., V.J. Yohai and M. Salibian-Barrera (2019):
    \emph{Robust statistics: Theory and methods}. Chichester: John Wiley
    and Sons, 2nd ed.

Richardson, A.M. and A.H. Welsh (1995).
    Robust restricted maximum likelihood in mixed linear model.
    \emph{Biometrics} \bold{51}, 1429--1439.
    \doi{https://doi.org/10.2307/2533273}

Rousseeuw, P. J. and K. Van Driessen (2006).
    Computing LTS regression for large data sets.
    \emph{Data Mining and Knowledge Discovery} \bold{12}, 29--45.
    \doi{https://doi.org/10.1007/s10618-005-0024-4}

Schoch, T. (2012). Robust Unit-Level Small Area Estimation: A Fast Algorithm
    for Large Datasets. \emph{Austrian Journal of Statistics} \bold{41},
    243--265. \doi{https://doi.org/10.17713/ajs.v41i4.1548}
}
\seealso{
    \code{\link[=fitsaemodel.control]{fitsaemodel.control()}}
}

\examples{
# use the landsat data
head(landsat)

# define the saemodel using the landsat data
model <- saemodel(formula = HACorn ~ PixelsCorn + PixelsSoybeans,
    area = ~CountyName,
    data = subset(landsat, subset = (outlier == FALSE)))

# summary of the model
summary(model)

# maximum likelihood estimates
fitsaemodel("ml", model)

# Huber M-estimate with robustness tuning constant k = 2
m <- fitsaemodel("huberm", model, k = 2)
m

# summary of the fitted model/ estimates
summary(m)

# obtain more information about convergence
convergence(m)

# extract the fixed effects
coef(m, "fixef")

# extract the random effects
coef(m, "ranef")

# extract both
coef(m)
}
