% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model_evaluation.R
\name{evaluate_model_results}
\alias{evaluate_model_results}
\title{Summarise simulation results from extracted model estimates}
\usage{
evaluate_model_results(
  results,
  alpha = 0.05,
  ...,
  .summarise_standard_broom = FALSE,
  broom_cols = c("estimate", "std.error", "statistic", "df", "p.value")
)
}
\arguments{
\item{results}{A data frame of extracted model results, typically including columns
like \code{term}, \code{estimate}, \code{std.error}, \code{statistic}, and \code{p.value}.}

\item{alpha}{Significance level used to compute power. Defaults to \code{0.05}.}

\item{...}{Additional summary expressions to compute within \code{dplyr::summarise()}.
These may include calls to helper functions like \code{eval_bias()}, \code{eval_quantile()},
or direct summaries such as \code{mean(estimate, na.rm = TRUE)}.}

\item{.summarise_standard_broom}{Logical; if \code{TRUE}, computes mean and standard deviation
for standard \code{broom} columns present in the data (columns in \code{broom_cols}).
Defaults to \code{FALSE}.}

\item{broom_cols}{Character vector of standard \code{broom} columns to summarise if
\code{.summarise_standard_broom = TRUE}. Defaults to
\code{c("estimate", "std.error", "statistic", "df", "p.value")}.}
}
\value{
A summarised data frame containing:
\itemize{
\item \code{n_models}: the number of models summarised.
\item \code{power}: the proportion of p-values less than \code{alpha} (NA if all p-values are NA).
\item Additional columns corresponding to custom summaries provided in \code{...}.
\item Mean and SD summaries of \code{broom} columns if \code{.summarise_standard_broom = TRUE}.
}
}
\description{
Computes summary statistics (e.g., power, custom summaries) across a set of
extracted model results, typically from \code{extract_model_results()}, to facilitate
simulation evaluation and reporting.
}
\examples{
library(dplyr)
library(purrr)
library(broom.mixed)

# Simulate and fit models
sim_models <- tibble(
  id = 1:50,
  model = map(1:50, ~ lm(mpg ~ wt, data = mtcars))
) |>
  extract_model_results()

# Evaluate power and mean estimate for the slope
sim_models |>
  filter(term == "wt") |>
  group_by(term) |>
  evaluate_model_results(
    alpha = 0.05,
    mean_estimate = mean(estimate, na.rm = TRUE),
    sd_estimate = sd(estimate, na.rm = TRUE)
  )

# Evaluate with .summarise_standard_broom = TRUE
sim_models |>
  filter(term == "wt") |>
  group_by(term) |>
  evaluate_model_results(
    .summarise_standard_broom = TRUE
  )

# Evaluate with eval_bias to compute bias relative to the true value
# Suppose the true slope of wt is -5 (hypothetical)
sim_models |>
  filter(term == "wt") |>
  group_by(term) |>
  evaluate_model_results(
    bias = eval_bias(
      estimate,
      term = c("wt" = -5)
    )
  )

}
