% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/quadtree.R
\name{quadtree}
\alias{quadtree}
\alias{quadtree,ANY-method}
\title{Create a \code{Quadtree} from a raster or matrix}
\usage{
\S4method{quadtree}{ANY}(
  x,
  split_threshold = NULL,
  split_method = "range",
  split_fun = NULL,
  split_args = list(),
  split_if_any_na = TRUE,
  split_if_all_na = FALSE,
  combine_method = "mean",
  combine_fun = NULL,
  combine_args = list(),
  max_cell_length = NULL,
  min_cell_length = NULL,
  adj_type = "expand",
  resample_n_side = NULL,
  resample_pad_nas = TRUE,
  extent = NULL,
  projection = "",
  proj4string = NULL,
  template_quadtree = NULL
)
}
\arguments{
\item{x}{a \code{\link[raster:RasterLayer-class]{RasterLayer}},
\code{\link[terra:SpatRaster-class]{SpatRaster}}, or
\code{matrix}. If \code{x} is a \code{matrix}, the \code{extent} and
\code{projection} parameters can be used to set the extent and projection
of the quadtree. If \code{x} is a
\code{\link[raster:RasterLayer-class]{RasterLayer}} or
\code{\link[terra:SpatRaster-class]{SpatRaster}}, the extent and
projection are derived from the raster.}

\item{split_threshold}{numeric; the threshold value used by the split method
(specified by \code{split_method}) to decide whether to split a quadrant.
If the value for a quadrant is greater than this value, it is split into
its four child cells. If \code{split_method} is \code{"custom"}, this
parameter is ignored.}

\item{split_method}{character; one of \code{"range"} (the default),
\code{"sd"} (standard deviation), \code{"cv"} (coefficient of variation) or
\code{"custom"}. Determines the method used for calculating the value used
to determine whether or not to split a quadrant (this calculated value is
compared with \code{split_threshold} to decide whether to split a cell). If
\code{"custom"}, a function must be supplied to \code{split_fun}. See
'Details' for more.}

\item{split_fun}{function; function used on each quadrant to decide whether
or not to split the quadrant. Only used when \code{split_method} is
\code{"custom"}. Must take two arguments, \code{vals} (a numeric vector of
the cell values in a quadrant) and \code{args} (a named list of arguments
used within the function), and must output \code{TRUE} if the quadrant is
to be split and \code{FALSE} otherwise. It must be able to handle \code{NA}
values - if \code{NA} is ever returned, an error will occur.}

\item{split_args}{list; named list that contains the arguments needed by
\code{split_fun}. This list is given to the \code{args} parameter of
\code{split_fun}.}

\item{split_if_any_na}{boolean; if \code{TRUE} (the default), a quadrant is
automatically split if any of the values within the quadrant are \code{NA}.}

\item{split_if_all_na}{boolean; if \code{FALSE} (the default), a quadrant
that contains only \code{NA} values is not split. If \code{TRUE}, quadrants
that contain all \code{NA} values are split to the smallest possible cell
size.}

\item{combine_method}{character; one of \code{"mean"}, \code{"median"},
\code{"min"}, \code{"max"}, or \code{"custom"}. Determines the method used
for aggregating the values of multiple cells into a single value for a
larger, aggregated cell. Default is \code{"mean"}. If \code{"custom"}, a
function must be supplied to \code{combine_fun}.}

\item{combine_fun}{function; function used to calculate the value of a
quadrant. Only used when \code{combine_method} is \code{"custom"}. Must
take two arguments, \code{vals} (a numeric vector of the cell values in a
quadrant) and \code{args} (a named list of arguments used within the
function), and must output a single numeric value, which will be used as
the cell value.}

\item{combine_args}{list; named list that contains the arguments needed by
\code{combine_fun}. This list is given to the \code{args} parameter of
\code{combine_fun}.}

\item{max_cell_length}{numeric; the maximum side length allowed for a
quadtree cell. Any quadrants larger than \code{max_cell_length} will
automatically be split. If \code{NULL} (the default) no restrictions are
placed on the maximum cell length.}

\item{min_cell_length}{numeric; the minimum side length allowed for a
quadtree cell. A quadrant will not be split if its children would be
smaller than \code{min_cell_length}. If \code{NULL} (the default) no
restrictions are placed on the minimum cell length.}

\item{adj_type}{character; one of \code{"expand"} (the default),
\code{"resample"}, or \code{"none"}. Specifies the method used to adjust
\code{x} so that its dimensions are suitable for quadtree creation (i.e.
square and with the number of cells in each direction being a power of 2).
See 'Details' for more on the two methods of adjustment.}

\item{resample_n_side}{integer; if \code{adj_type} is \code{'resample'}, this
number is used to determine the dimensions to resample the raster to.}

\item{resample_pad_nas}{boolean; only applicable if \code{adj_type} is
\code{'resample'}. If \code{TRUE} (the default), \code{NA}s are added to
the shorter side of the raster to make it square before resampling. This
ensures that the cells of the resulting quadtree will be square. If
\code{FALSE}, no \code{NA}s are added - the cells in the quadtree will not
be square.}

\item{extent}{\code{\link[raster:Extent-class]{Extent}} object or else a
four-element numeric vector describing the extent of the data (in this
order: xmin, xmax, ymin, ymax). Only used when \code{x} is a matrix - this
parameter is ignored if \code{x} is a raster since the extent is derived
directly from the raster. If no value is provided and \code{x} is a matrix,
the extent is assumed to be \code{c(0,ncol(x),0,nrow(x))}.}

\item{projection}{character; string describing the projection of the
data. Only used when \code{x} is a matrix - this parameter is ignored if
\code{x} is a raster since the proj4ection of the raster is automatically
used. If no value is provided and \code{x} is a matrix, the projection of
the quadtree is set to \code{NA}.}

\item{proj4string}{deprecated. Use \code{projection} instead.}

\item{template_quadtree}{\code{\link{Quadtree}}; if provided, the new
quadtree will be created so that it has the exact same structure as the
template quadtree. Thus, no split function is used because the decision
about whether to split is pre-determined by the template quadtree. The
raster used to create the template quadtree should have the exact same
extent and dimensions as \code{x}. If \code{template_quadtree} is
non-\code{NULL}, all \code{split_}* parameters are disregarded, as are
\code{max_cell_length} and \code{min_cell_length}.}
}
\value{
a \code{\link{Quadtree}}
}
\description{
Creates a \code{\link{Quadtree}} from a 
\code{\link[terra:SpatRaster-class]{SpatRaster}},
\code{\link[raster:RasterLayer-class]{RasterLayer}} or a matrix.
}
\details{
The 'quadtree-creation' vignette contains detailed explanations and
  examples for all of the various creation options - run
  \code{vignette("quadtree-creation", package = "quadtree")} to view the
  vignette.

  If \code{adj_type} is \code{"expand"}, \code{NA} cells are added to the
  raster in order to create an expanded raster whose dimensions are a power
  of two. The smallest number that is a power of two but greater than the
  larger dimension is used as the dimensions of the expanded raster. If
  \code{adj_type} is \code{"resample"}, the raster is resampled to a raster
  with \code{resample_n_side} rows and columns. If \code{resample_pad_nas} is
  \code{TRUE}, \code{NA} rows or columns are are added to the shorter
  dimension before resampling to make the raster square. This ensures that
  the quadtree cells will be square (assuming the original raster cells were
  square).

  When \code{split_method} is \code{"range"}, the difference between the
  maximum and minimum cell values in a quadrant is calculated - if this value
  is greater than \code{split_threshold}, the quadrant is split. When
  \code{split_method} is \code{"sd"}, the standard deviation of the cell
  values in a quadrant is calculated - if this value is greater than
  \code{split_threshold}, the quadrant is split.
}
\examples{
####### NOTE #######
# see the "quadtree-creation" vignette for more details and examples of all
# the different parameter options:
# vignette("quadtree-creation", package = "quadtree")
####################

library(quadtree)
habitat <- terra::rast(system.file("extdata", "habitat.tif", package="quadtree"))

qt <- quadtree(habitat, .15)
plot(qt)
# we can make it look nicer by customizing the plotting parameters
plot(qt, crop = TRUE, na_col = NULL, border_lwd = .3)

# try a different splitting method
qt <- quadtree(habitat, .05, "sd")
plot(qt)

# ---- using a custom split function ----

# split a cell if any of the values are below a given value
split_fun = function(vals, args) {
  if (any(is.na(vals))) { # check for NAs first
    return(TRUE) # if there are any NAs we'll split automatically
  } else {
    return(any(vals < args$threshold))
  }
}

qt <- quadtree(habitat, split_method = "custom", split_fun = split_fun,
                split_args = list(threshold = .8))
plot(qt)
}
