% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pooldata.subset.R
\name{pooldata.subset}
\alias{pooldata.subset}
\title{Create a subset of the pooldata object that contains Pool-Seq data as a function of pool and/or SNP indexes}
\usage{
pooldata.subset(
  pooldata,
  pool.index = 1:pooldata@npools,
  snp.index = 1:pooldata@nsnp,
  min.maf = -1,
  min.cov = 0,
  max.cov = 1e+09,
  cov.qthres = c(0, 1),
  min.cov.per.pool = -1,
  max.cov.per.pool = 1e+06,
  cov.qthres.per.pool = c(0, 1),
  return.snp.idx = FALSE,
  verbose = TRUE
)
}
\arguments{
\item{pooldata}{A pooldata object containing Pool-Seq information}

\item{pool.index}{Indexes of the pools (at least two), that should be selected to create the new pooldata object (default=all the pools)}

\item{snp.index}{Indexes of the SNPs (at least two), that should be selected to create the new pooldata object (default=all the SNPs)}

\item{min.maf}{Minimal allowed Minor Allele Frequency (computed from the ratio over all read counts for the reference allele over the read coverage)}

\item{min.cov}{Minimal allowed read count (over all the pools).}

\item{max.cov}{Maximal allowed read count (over all the pools).}

\item{cov.qthres}{A two-elements vector containing the minimal (qmin) and maximal (qmax) quantile thresholds (0<=qmin<qmax<=1) for the overall coverage (i.e., summing over all pools). See details below}

\item{min.cov.per.pool}{Minimal allowed read count (per pool). If at least one pool is not covered by at least min.cov.perpool reads, the position is discarded}

\item{max.cov.per.pool}{Maximal allowed read count (per pool). If at least one pool is covered by more than min.cov.perpool reads, the position is discarded}

\item{cov.qthres.per.pool}{A two-elements vector containing the minimal (qmin) and maximal (qmax) quantile coverage thresholds applied to each pools (0<=qmin<qmax<=1). See details below}

\item{return.snp.idx}{If TRUE, the row.names of the snp.info slot of the returned pooldata object are named as "rsx" where x is the index of SNP in the initial pooldata object (default=FALSE)}

\item{verbose}{If TRUE return some information}
}
\value{
A pooldata object with 7 elements:
\enumerate{
\item "refallele.readcount": a matrix with nsnp rows and npools columns containing read counts for the reference allele (chosen arbitrarily) in each pool
\item "readcoverage": a matrix with nsnp rows and npools columns containing read coverage in each pool
\item "snp.info": a matrix with nsnp rows and four columns containing respectively the contig (or chromosome) name (1st column) and position (2nd column) of the SNP; the allele in the reference assembly (3rd column); the allele taken as reference in the refallele matrix.readcount matrix (4th column); and the alternative allele (5th column)
\item "poolsizes": a vector of length npools containing the haploid pool sizes
\item "poolnames": a vector of length npools containing the names of the pools
\item "nsnp": a scalar corresponding to the number of SNPs
\item "npools": a scalar corresponding to the number of pools
}
}
\description{
Create a subset of the pooldata object that contains Pool-Seq data as a function of pool and/or SNP indexes
}
\details{
This function subsets a \code{pooldata} object by selecting a subset of pools
and/or SNPs (e.g., based on genomic position).

Additional SNP-level filtering can be applied to the resulting subset to remove
poorly polymorphic SNPs using \code{min.maf}, or SNPs with low or excessively high
coverage using \code{min.cov}, \code{max.cov}, and \code{cov.qthres}.

Coverage filtering can also be performed on a per-pool basis with
\code{min.cov.per.pool}, \code{max.cov.per.pool}, and \code{cov.qthres.per.pool}.

For the \code{cov.qthres} and \code{cov.qthres.per.pool} arguments, empirical
coverage quantiles are computed and used as filtering thresholds.
SNPs with coverage outside the specified quantile range are discarded.

For example, if \code{qmax = 0.95} in \code{cov.qthres.per.pool}, a SNP is removed
in a given pool if its coverage exceeds the 95th percentile of the empirical
coverage distribution for that pool (computed over SNPs selected by
\code{snp.index}). Conversely, if \code{qmin = 0.05}, SNPs with coverage below the
5th percentile are discarded.

Quantile-based filtering is particularly useful when pools display heterogeneous
sequencing depth.
}
\examples{
 make.example.files(writing.dir=tempdir())
 pooldata=popsync2pooldata(sync.file=paste0(tempdir(),"/ex.sync.gz"),poolsizes=rep(50,15))
 subset.by.pools=pooldata.subset(pooldata,pool.index=c(1,2))
 subset.by.snps=pooldata.subset(pooldata,snp.index=10:100)
 subset.by.pools.and.snps=pooldata.subset(pooldata,pool.index=c(1,2),snp.index=10:100)
 subset.by.pools.qcov.thr=pooldata.subset(pooldata,pool.index=1:8,cov.qthres.per.pool=c(0.05,0.95)) 
}
\seealso{
To generate pooldata object, see \code{\link{vcf2pooldata}}, \code{\link{popsync2pooldata}}
}
