% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plotting.R
\name{parameter_summary}
\alias{parameter_summary}
\title{Parameter Calculation Comparison Summary}
\usage{
parameter_summary(f_param, f_pop, model_partition, resolution = 101L)
}
\arguments{
\item{f_param}{a function, \code{f(x)} which transforms the feature (e.g. age),
to yield the parameter values. Alternatively, a \code{data.frame} where the first
column is the feature and the second is the parameter; see \code{\link[=xy.coords]{xy.coords()}} for
details. If the latter, combined with \code{pars_interp_opts} to create a
parameter function.}

\item{f_pop}{like \code{f_param}, either a density function (though it does
not have to integrate to 1 like a pdf) or a \code{data.frame} of values. If the
latter, it is treated as a series of populations within intervals, and
then interpolated with \code{pop_interp_opts} to create a density function.}

\item{model_partition}{a numeric vector of cut points, which define the
partitioning that will be used in the model; must be length > 1}

\item{resolution}{the number of points to calculate for the underlying
\code{f_param} function. The default 101 points means 100 partitions.}
}
\value{
a \code{data.table}, columns:
\itemize{
\item \code{model_category}, a integer corresponding to which of the intervals of
\code{model_partition} the \code{x} value is in
\item \code{x}, a numeric series from the first to last elements of \code{model_partition}
with length \code{resolution}
\item \code{method}, a factor with levels:
\itemize{
\item \code{f_val}: \code{f_param(x)}
\item \code{f_mid}: \code{f_param(x_mid)}, where \code{x_mid} is the midpoint x of the
\code{model_category}
\item \code{f_mean}: \code{f_param(weighted.mean(x, w))}, where \code{w} defined by
\code{densities} and \code{model_category}
\item \code{mean_f}: \code{weighted.mean(f_param(x), w)}, same as previous
\item \code{wm_f}: the result as if having used \code{paramix::blend()}; this should be
very similar to \code{mean_f}, though will be slightly different since \code{blend}
uses \code{integrate()}
}
}
}
\description{
Implements several approaches to computing partition-aggregated parameters,
then tables them up for convenient plotting.
}
\examples{
# COVID IFR from Levin et al 2020 https://doi.org/10.1007/s10654-020-00698-1
f_param <- function(age_in_years) {
  (10^(-3.27 + 0.0524 * age_in_years))/100
}

densities <- data.frame(
  from = 0:101,
  weight = c(rep(1, 66), exp(-0.075 * 1:35), 0)
)

model_partition <- c(0, 5, 20, 65, 101)

ps_dt <- parameter_summary(f_param, densities, model_partition)
ps_dt

\dontshow{if (require(ggplot2)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
ggplot(ps_dt) + aes(x, y = value, color = method) +
  geom_line(data = function(dt) subset(dt, method == "f_val")) +
  geom_step(data = function(dt) subset(dt, method != "f_val")) +
  theme_bw() + theme(
    legend.position = "inside", legend.position.inside = c(0.05, 0.95),
    legend.justification = c(0, 1)
  ) + scale_color_discrete(
    "Method", labels = c(
      f_val = "f(x)", f_mid = "f(mid(x))", f_mean = "f(E[x])",
      mean_f = "discrete E[f(x)]", wm_f = "integrated E[f(x)]"
    )
  ) +
  scale_x_continuous("Age", breaks = seq(0, 100, by = 10)) +
  scale_y_log10("IFR", breaks = 10^c(-6, -4, -2, 0), limits = 10^c(-6, 0))
\dontshow{\}) # examplesIf}
}
