# Goodman, 1979, also Agresti #2.2
# Goodman's diagonals-parameter symmetry model


#' Fit's Goodman's diagonals parameter symmetry model.
#'
#' Goodman, L. A. (1979). Multiplicative models for square contingency
#' tables with ordered categories.  Biometrika, 66(3), 413-316.
#' @importFrom stats chisq.test
#' @param n the matrix of obsever counts
#' @returns a list containing
#'    individual_chisq: chi-square value for each diagonal
#'    individual_df: degrees of freedom for individual_chisq
#'    omnibus_chisq: overall chi-square for the model
#'    omnibus_df: degrees for freedom for omnibus_chisq
#'    equality_chisq: chi-square for test that all delta values are equal
#'    equality_df: degrees of freedom from equality_chisq
#'    delta: the vector of estimated delta values (without any equality constraints)
#' @export
#' @examples
#' Goodman_diagonals_parameter_symmetry(vision_data)
Goodman_diagonals_parameter_symmetry <- function(n) {
	c <- nrow(n)
	r <- matrix(nrow = c - 1,  ncol = 2)
	omnibus_chisq <- 0.0
	omnibus_df <- 0
	delta <- rep(0.0, c - 1)
	individual_chisq <- vector("double", c - 2)
	individual_df <- vector("double", c - 2)
	for (k in 1:(c - 2)) {
	  rk <- matrix(nrow=c - k, ncol=2)
	  for (i in 1:(c - k)) {
	    rk[i, 1] <- n[i, i + k]
	    rk[i, 2] <- n[i + k, i]
	  }
	  r[k, 1] = colSums(rk)[2]
	  r[k, 2] = colSums(rk)[1]
	  delta[k] <- colSums(rk)[2] / colSums(rk)[1]

	  chi <- chisq.test(rk, correct = FALSE)
	  individual_chisq[k] <- chi$statistic
	  individual_df[k] <- chi$parameter
	  omnibus_chisq <- omnibus_chisq + chi$statistic
	  omnibus_df <- omnibus_df + nrow(rk) - 1
	}
	delta[c - 1] <- n[c, 1] / n[1, c]
	r[c - 1, 1] <- n[c, 1]
	r[c - 1, 2] <- n[1, c]

	chi <- chisq.test(r, correct=FALSE)
	equality_chisq <- chi$statistic
	equality_df = c - 2

	list(individual_chisq=individual_chisq, individual_df=individual_df,
	     omnibus_chisq=omnibus_chisq, omnibus_df=omnibus_df,
	     equality_chisq=equality_chisq, equality_df=equality_df,
	     delta=delta)
}


#' Fits the model where some of the delta parameters are constrained to be equal to one another.
#'
#' @importFrom stats chisq.test
#' @param n the matrix of observed counts
#' @param equality logical vector indicating whether corresponding delta the parameter is part
#' of the equality set.
#' @returns a list containing
#'    pooled_chisq: Pearson chi-square for the pooled delta values
#'    pooled_df: degrees of freedom for pooled chisq
#'    omnibus_chisq: Pearson chi-square for overall model fit, subject to equality constraints
#'    omnibus_df; degrees of freedom for omnibus_chisq
#'    equality_chisq: Pearson chi-square for test that remaining deltas are all equal
#'    equality_df: degrees of freedom for equality_chisq
#;    delta: vector of all delta estimates
#'    delta_pooled: estimate of pooled delta
#' @export
#' @examples
#' equality = c(TRUE, TRUE, FALSE)
#' Goodman_diagonals_parameter_symmetry(vision_data)
Goodman_constrained_diagonals_parameter_symmetry <- function(n, equality) {
  c <- nrow(n)
  n_constraints <- sum(equality)
  if (n_constraints < 2) {
    warning("At least two levels must be contrained to be equal")
    return(NaN)
  }
  if (c - 1 < length(equality)) {
    warning(paste("At most", c - 1, "levels can be specified"))
    return(NaN)
  }

  n_tables <- c - 1
  n_levels <- 0
  for (i in 1:(c - 1)) {
    if (equality[i]) {
      n_levels <- n_levels + c - i
    }
  }
  r_pooled <- matrix(nrow = n_levels,  ncol=2)
  r <- matrix(nrow = n_tables - n_constraints + 1,  ncol = 2)

  omnibus_chisq <- 0.0
  omnibus_df <- 0
  delta <- rep(0.0, c - 2)
  index <- 1
  index2 <- 1
  for (k in 1:(c - 2)) {
    if (equality[k]) {
      for (i in 1:(c - k)) {
        r_pooled[index, 1] <- n[i, i + k]
        r_pooled[index, 2] <- n[i + k, i]
        index <- index + 1
      }
      next
    } else {
      rk <- matrix(nrow=c - k, ncol=2)
      for (i in 1:(c - k)) {
        rk[i, 1] <- n[i, i + k]
        rk[i, 2] <- n[i + k, i]
      }
      r[index2, 1] = colSums(rk)[2]
      r[index2, 2] = colSums(rk)[1]
      delta[index2] <- colSums(rk)[2] / colSums(rk)[1]
      index2 <- index2 + 1

      chi <- chisq.test(rk, correct = FALSE)
      omnibus_chisq <- omnibus_chisq + chi$statistic
      omnibus_df <- omnibus_df + nrow(rk) - 1
    }
  }
  if (equality[c - 1]) {
    r_pooled[index, 1] <- n[c, 1]
    r_pooled[index, 2] <- n[1, c]
  } else {
    delta[index2] <- n[c, 1] / n[1, c]
    r[index2, 1] <- n[c, 1]
    r[index2, 2] <- n[1, c]
    index2 <- index2 + 1
  }

  r[n_tables - n_constraints + 1, 1] <- colSums(r_pooled)[2]
  r[n_tables - n_constraints + 1, 2] <- colSums(r_pooled)[1]
  delta_pooled <- colSums(r_pooled)[2] / colSums(r_pooled)[1]
  delta[index2] <- delta_pooled

  chi <- chisq.test(r_pooled)
  pooled_chisq <- chi$statistic
  pooled_df <- chi$parameter

  omnibus_chisq <- omnibus_chisq + chi$statistic
  omnibus_df <- omnibus_df + chi$parameter

  chi <- chisq.test(r, correct=FALSE)
  equality_chisq <- chi$statistic
  equality_df <- chi$parameter

  list(pooled_chisq=pooled_chisq, pooled_df=pooled_df,
       omnibus_chisq=omnibus_chisq, omnibus_df=omnibus_df,
       equality_chisq=equality_chisq, equality_df=equality_df,
       delta=delta, delta_pooled=delta_pooled)
}


#' Fits the model with given parameters fixed to specific values.
#'
#' The model has simple closed form solutions when fitting either the unconstrained
#' version of the version that species equality of delta parameters.  However,
#' I could not see how to adapt that to the case where specific parameters were
#' constrained to have a specific value.  This routine is to fit that model. It will
#' also fit the unconstrained model, but Goodman gives the estimator for that case.
#' @param n the r X r matrix of observed counts
#' @param delta the vector of asymmetry r - 1 parameters
#' @param fixed r - 1 logical vector that specifies whether a delta parameter is fixed
#' (TRUE) or allowed to be estimated (FALSE).
#' @param max_iter maximum number of iterations, Default is 50.
#' @param convergence maximum change in a parameter across iterations. Default is 1.0e-4
#' @param verbose should progress information be printed to the console. Default is
#' FALSE, do not print.
#' @returns list containing phi, delta,
#'    max_change largest change in  parameter for last the iteration,
#'    chisq: Pearson chi-square
#'    g_squared: likelihood ratio G^2
#'    df: degrees of freedom
#' @export
#' @examples
#' fixed <- c(FALSE, TRUE, FALSE)
#' delta <- c(1.0, 1.0, 1.0)
#' phi <- matrix(0.0, nrow=4, ncol=4)
#' diag(phi) = rep(1.0, 4)
#' Goodman_fixed_parameter(vision_data, delta, fixed)
#' @seealso [Goodman_diagonals_parameter_symmetry()]
#' @seealso [Goodman_ml()]
Goodman_fixed_parameter <- function(n, delta, fixed, convergence=1.0e-4, max_iter=50,
                                    verbose=FALSE) {
  r <- nrow(n)
  phi <- matrix(0.1, nrow=r, ncol=r)
  for (i in 1:r) {
    phi[i, i] <- 1.0
  }
  pi_matrix <- Goodman_pi_matrix(phi, delta)
  if (verbose) {
    message(paste("initial chisq:", pearson_chisq(n, pi_matrix)))
  }

  pi <- Goodman_pi_matrix(phi, delta)
  logL <- log_likelihood(n, pi)
  if (verbose) {
    message(paste(0, logL))
  }

  for (iter in 1:max_iter) {
    result <- Goodman_ml(n, phi, delta, fixed)
    max_phi <- max(abs(phi - result$phi))
    phi <- result$phi
    max_delta <- max(abs(delta - result$delta))
    delta <- result$delta
    max_change <- max(max_phi, max_delta)
    pi <- Goodman_pi_matrix(phi, delta)
    logL <- log_likelihood(n, pi)
    if (verbose) {
      message(paste(iter, logL))
    }
    if (max_change <= convergence) {
      break
    }
  }
  pi_matrix <- Goodman_pi_matrix(phi, delta)
  chisq <- pearson_chisq(n, pi_matrix)
  g_squared <- likelihood_ratio_chisq(n, pi_matrix)
  if (verbose) {
    message(paste("  final chisq:", chisq))
  }
  k = sum(fixed)
  df <- k + (r - 1) * (r - 2) / 2
  list(phi=phi, delta=delta, max_change=max_change,
       chisq=chisq, g_squared=g_squared, df=df)
}


#' Performs ML estimation of the model.
#'
#' The model has simple closed form solutions when fitting either the unconstrained
#' version of the version that species equality of delta parameters.  However,
#' I could not see how to adapt that to the case where specific parameters were
#' constrained to have a specific value.  This routine is to fit that model. It will
#' also fit the unconstrained model, but Goodman gives the estimator for that case.
#' @param n the r X r matrix of observed counts
#' @param phi the symmetric matrix parameter
#' @param delta the vector of asymmetry r - 1 parameters
#' @param fixed r - 1 logical vector that specifies whether a delta parameter is fixed
#' (TRUE) or allowed to be estimated (FALSE).
#' @returns list containing new estimates of phi amd delta
#' @export
#' @examples
#' fixed <- c(FALSE, TRUE, FALSE)
#' delta <- c(1.0, 1.0, 1.0)
#' phi <- matrix(0.0, nrow=4, ncol=4)
#' for (i in 1:4) {
#'   phi[i, i] = 1.0
#' }
#' Goodman_ml(vision_data, phi, delta, fixed)
#' @seealso [Goodman_diagonals_parameter_symmetry()]
Goodman_ml <- function(n, phi, delta, fixed) {
  N <- sum(n)
  r <- nrow(n)
  delta_numer <- rep(0.0, length(delta))
  delta_denom <- rep(0.0, length(delta))
  for (i in 1:r) {
    for(j in 1:i) {
      k <- i - j
      if (k == 0) {
        phi[i, i] <- n[i, i] / N
        next
      }
      if (0 < k) {
        phi[i, j] <- (n[i, j] + n[j, i]) / (N * (1.0 + delta[k]))
        phi[j, i] <- phi[i, j]
        delta_numer[k] <- delta_numer[k] + n[i, j]
        delta_denom[k] <- delta_denom[k] + N * phi[i, j]
      }
    }
  }
  for (k in 1:(r - 1)) {
    if (!fixed[k]) {
      delta[k] <- delta_numer[k] / delta_denom[k]
    }
  }
  list(phi=phi, delta=delta)
}


#' Computes the full matrix of model-based cell probabilities.
#'
#' @param phi the symmetric matrix
#' @param delta the vector of asymmetry parameters
#' @returns matrix of model-based probabilities
Goodman_pi_matrix <- function(phi, delta) {
  r <- nrow(phi)
  pi <- matrix(nrow=r, ncol=r)
  for (i in 1:r) {
    for (j in 1:r) {
      pi[i, j] <- Goodman_pi(phi, delta, i, j)
    }
  }
  pi
}


#' Computes the model-based probability for cell i, j
#'
#' @param phi symmetry matrix
#' @param delta vector of asymmetry parameters
#' @param i row index
#' @param j column index
#' @returns pi for that cell
Goodman_pi <- function(phi, delta, i, j) {
  k <- i - j
  if (0 < k) {
    pi <- delta[k] * phi[i, j]
  } else {
    pi <- phi[i, j]
  }
  pi
}
