% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run_single_point_extra.R
\name{run_single_point_extra}
\alias{run_single_point_extra}
\title{Perform extended single-point pharmacokinetic calculations}
\usage{
run_single_point_extra(
  dat = NULL,
  half_life = NULL,
  single_point_base.lst = NULL,
  route = c("bolus", "oral", "infusion"),
  dose_type = NULL,
  pooled_ctrl = pooled_control(),
  ssctrl = ss_control()
)
}
\arguments{
\item{dat}{A data frame containing raw time–concentration data in the
standard nlmixr2 format.}

\item{half_life}{Optional numeric value representing the elimination half-life of the drug.
If not provided, half-life is estimated within \code{run_single_point_base()} using \code{get_hf()} applied
to the pharmacokinetic observations.}

\item{single_point_base.lst}{A list object returned from the base single-point calculation
that includes input data, preprocessing information, and initial estimates of CL and Vd.
If not supplied, the function will internally call the base routine using dat.}

\item{route}{Character string specifying the route of administration. Must be one of
"bolus", "oral", or "infusion".}

\item{dose_type}{Classified as "first_dose", "repeated_doses", or "combined_doses"
based on whether observed concentrations occur following the first administration,
during repeated dosing, or across both contexts. This parameter is passed directly to
\code{run_single_point_base()}.}

\item{pooled_ctrl}{A list of pooled-analysis control options created using \code{pooled_control()}.
These control time binning and time-after-dose rounding when pooled summaries are required.}

\item{ssctrl}{A list of steady-state control options created using \code{ss_control()}.
These govern assumptions and thresholds used when interpreting steady-state behavior
in single-point logic.}
}
\value{
A list containing:
\itemize{
\item singlepoint.results: A data frame with estimated ka, CL, Vd, computation start time,
processing time, and a descriptive message of the applied logic.
\item dat: The dataset used for processing.
\item single_point.ka.out: Output used for estimating the absorption rate constant
(for oral administration), if applicable.
\item single_point_cl_df: Data used for clearance estimation.
\item single_point_vd_df: Data used for volume of distribution estimation.
\item approx.vc.out: Data used for estimating the volume of distribution from maximum
concentration (Cmax) and dose when needed.
}
}
\description{
Extended single-point pharmacokinetic analysis for deriving CL, Vd, and ka
}
\details{
The function derives pharmacokinetic parameters using the following logic:
\itemize{
\item When both clearance (CL) and volume of distribution (Vd) are available from the base
calculation, these values are directly used without modification.
\item If Vd is missing but CL and elimination half-life are provided, Vd is calculated using:
\deqn{V_d = \frac{CL \cdot t_{1/2}}{\ln(2)}}
\item If CL is missing but Vd and half-life are available, CL is calculated using:
\deqn{CL = \frac{V_d \cdot \ln(2)}{t_{1/2}}}
\item If both CL and Vd are unavailable but half-life is provided, Vd is estimated using dose and Cmax:
\deqn{V_d = \frac{\mathrm{Dose}}{C_{\mathrm{max}}}}
and CL is subsequently derived:
\deqn{CL = \frac{V_d \cdot \ln(2)}{t_{1/2}}}
\item For oral administration, the absorption rate constant (ka) is estimated using a solution-based
approach implemented in \code{run_ka_solution()}. Only concentration–time data from the absorption
phase are used, defined as time after dose (tad) less than 20\% of the terminal half-life and
occurring prior to Tmax, where absorption is the dominant process.
}

The function supports bolus, oral, and infusion administration routes. For oral dosing,
only data within the absorption phase are used to estimate the absorption rate constant (ka),
specifically using concentration-time observations prior to the maximum concentration (Tmax).
}
\examples{
dat <- Bolus_1CPT
out <- processData(dat)
fdat <- out$dat
froute <- out$Datainfo$Value[out$Datainfo$Infometrics == "Dose Route"]
half_life <- get_hf(dat = fdat)$half_life_median
run_single_point_extra(
  dat = fdat,
  half_life = half_life,
  single_point_base.lst = run_single_point_base(
    dat = fdat,
    half_life = half_life,
    route = froute
  ),
  route = froute
)

}
\seealso{
\code{\link{run_single_point_base}}, \code{\link{run_single_point}}, \code{\link{run_ka_solution}}
}
\author{
Zhonghui Huang
}
