#### testing make.pmatrix function ---------------------------------------------

#### Expecting no errors

set.seed(12345)
X1 <- rnorm(100, 0, 5)
Z1 <- X1 + rnorm(100, 0, (1+sqrt(abs(X1))))

# pmatrix works with just one covariate
test_that("pmatrix function works with one covariate", {
  expect_no_error(make.pmatrix(Z1, X1))
})

X2 <- rnorm(100, 0, 1)
X3 <- rnorm(100, 0, 1)
X_df <- cbind(X1, X2, X3)
Z2 <- X1 + X3 + rnorm(100, 0, (1+sqrt(abs(X2))))

# pmatrix works with multiple covariates
test_that("pmatrix function works with multiple covariates", {
  expect_no_error(make.pmatrix(Z2, X_df))
})

#### Expecting errors

# error when Z is different length than X
Z_wrong <- Z2[1:99]
test_that("Error caused by Z and X differing lengths", {
  expect_error(make.pmatrix(Z_wrong, X_df), "Treatment vector is not same length as covariate matrix.")
})

# error when Z is non-numeric
Z_wrong2 <- rep("error", 100)
test_that("Error caused by Z being non-numeric", {
  expect_error(make.pmatrix(Z_wrong2, X_df), "Treatment is not numeric.")
})

# error when X has at least one non-numeric column
X_df_wrong <- X_df
X_df_wrong[1, 1] <- "error"
test_that("Error caused by Z being non-numeric", {
  expect_error(make.pmatrix(Z2, X_df_wrong), "Covariate matrix contains one or more non-numeric entries.")
})

#### testing nbp.caliper function ---------------------------------------------

#### Expecting no errors

pmat <- make.pmatrix(Z2, X_df)

# nbp.caliper with defaults for xi and M work
test_that("nbp.caliper runs with defaults", {
  expect_no_error(nbp.caliper(Z2, X_df, pmat))
})

# nbp.caliper with manually inputted xi and M works
test_that("nbp.caliper runs with defined xi and M", {
  expect_no_error(nbp.caliper(Z2, X_df, pmat, 0.1, 1000))
})

#### Expecting errors

# errors when pmat has the wrong dimensions

pmat_wrong <- pmat[-1, ]

test_that("Error from pmat not having enough rows", {
  expect_error(nbp.caliper(Z2, X_df, pmat_wrong), "pmat has the incorrect dimensions.")
})

pmat_wrong2 <- rbind(pmat, rep(0.5, 100))

test_that("Error from pmat having too many rows", {
  expect_error(nbp.caliper(Z2, X_df, pmat_wrong2), "pmat has the incorrect dimensions.")
})

pmat_wrong3 <- pmat[, -1]

test_that("Error from pmat not having enough columns", {
  expect_error(nbp.caliper(Z2, X_df, pmat_wrong3), "pmat has the incorrect dimensions.")
})

pmat_wrong4 <- cbind(pmat, rep(0.5, 100))

test_that("Error from pmat having too many columns", {
  expect_error(nbp.caliper(Z2, X_df, pmat_wrong4), "pmat has the incorrect dimensions.")
})

# error when pmat has inappropriate probability values

pmat_wrong5 <- pmat
pmat_wrong5[2, 1] <- 1.1
pmat_wrong5[1, 2] <- 1.1

test_that("Error from having probabilities above 1", {
  expect_error(nbp.caliper(Z2, X_df, pmat_wrong5), "One or more entries in pmat are not valid probability values.")
})

pmat_wrong6 <- pmat
pmat_wrong6[2, 1] <- -0.1
pmat_wrong6[1, 2] <- -0.1

test_that("Error from having negative values as probabilities", {
  expect_error(nbp.caliper(Z2, X_df, pmat_wrong6), "One or more entries in pmat are not valid probability values.")
})

pmat_wrong7 <- pmat
pmat_wrong7[2, 1] <- "error"
pmat_wrong7[1, 2] <- "error"

test_that("Error from non-numeric values in pmat matrix", {
  expect_error(nbp.caliper(Z2, X_df, pmat_wrong7), "One or more entries in pmat are not valid probability values.")
})

# error when xi is an inappropriate value

test_that("Error from xi being a negative number", {
  expect_error(nbp.caliper(Z2, X_df, pmat, -0.1), "Hyperparameter xi must be between 0 and 0.5, excluding 0.5.")
})

test_that("Error from xi being equal to 0.5", {
  expect_error(nbp.caliper(Z2, X_df, pmat, 0.5), "Hyperparameter xi must be between 0 and 0.5, excluding 0.5.")
})

test_that("Error from xi being too large", {
  expect_error(nbp.caliper(Z2, X_df, pmat, 0.6), "Hyperparameter xi must be between 0 and 0.5, excluding 0.5.")
})

test_that("Error from xi being non-numeric", {
  expect_error(nbp.caliper(Z2, X_df, pmat, "error"), "Hyperparameter xi must be between 0 and 0.5, excluding 0.5.")
})

# error when M is an inappropriate value

test_that("Error from M being a negative number", {
  expect_error(nbp.caliper(Z2, X_df, pmat, xi = 0, M = -0.1), "Hyperparameter M must be a nonnegative number.")
})

test_that("Error from M being non-numeric", {
  expect_error(nbp.caliper(Z2, X_df, pmat, xi = 0, M = "error"), "Hyperparameter M must be a numeric.")
})

#### testing classic.neyman ----------------------------------------------------

pairs <- nbp.caliper(Z2, X_df, pmat)
Y <- X1 + X2 + X3 + rnorm(100)

#### Expecting no errors

test_that("classic.neyman works as intended", {
  expect_no_error(classic.neyman(Y, Z2, pairs))
})

#### expecting errors

# Y vector and Z vector are differing lengths

test_that("Error since Y vector is shorter than Z vector", {
  expect_error(classic.neyman(Y[-1], Z2, pairs), "Outcome vector and treatment vector are different lengths.")
})

test_that("Error since Y vector is longer than Z vector", {
  expect_error(classic.neyman(c(Y, 1), Z2, pairs), "Outcome vector and treatment vector are different lengths.")
})

# number of matched pairs does not match length of Y or Z

pairs_wrong <- rbind(pairs, c(101, 102))
test_that("Error from having too many matched pairs", {
  expect_error(classic.neyman(Y, Z2, pairs_wrong), "Number of pairs is not half the length of the outcome vector.")
})

test_that("Error from having too few matched pairs", {
  expect_error(classic.neyman(Y, Z2, pairs[-1, ]), "Number of pairs is not half the length of the outcome vector.")
})

# matched pairs contains a duplicate pairing

pairs_wrong2 <- pairs
pairs_wrong2[2, ] <- pairs_wrong2[1, ]
test_that("Error from having duplicate pairs", {
  expect_error(classic.neyman(Y, Z2, pairs_wrong2),
  "pairs dataframe either has duplicated pairs, has indices outside of expected range, or is otherwise improperly formatted.")
})

pairs_wrong3 <- pairs_wrong2
pairs_wrong3[2, ] <- rev(pairs_wrong3[2, ])
test_that("Error from having reflected (duplicated) pairs", {
  expect_error(classic.neyman(Y, Z2, pairs_wrong3),
               "pairs dataframe either has duplicated pairs, has indices outside of expected range, or is otherwise improperly formatted.")
})

pairs_wrong4 <- pairs
pairs_wrong4[1, 1] <- 101
test_that("Error from having index outside of 1,...,N", {
  expect_error(classic.neyman(Y, Z2, pairs_wrong4),
               "pairs dataframe either has duplicated pairs, has indices outside of expected range, or is otherwise improperly formatted.")
})

# if my treatment vector is the same value for all observations, I should get
# 0 as my SATE estimate

Z_same <- rep(1, 100)
test_that("Treatment with no variation has SATE estimate of 0 by definition", {
  expect_equal(classic.neyman(Y, Z_same, pairs), 0)
})

#### testing bias.corrected.neyman ---------------------------------------------

test_that("bias.corrected.neyman works as intended", {
  expect_no_error(bias.corrected.neyman(Y, Z2, pairs, pmat, xi = 0.1))
})

#### testing covAdj.variance ---------------------------------------------

#### Errors not expected

# variance estimations works when I don't specify X or Q matrix
test_that("variance estimator works with no X or Q specified", {
  expect_no_error(covAdj.variance(Y, Z2, pair = pairs, pmat = pmat, xi = 0.1))
})

# variance estimations works when I give an X but no Q
test_that("variance estimator works with X specified, Q unspecified", {
  expect_no_error(covAdj.variance(Y, Z2, X_df, pair = pairs, pmat = pmat, xi = 0.1))
})

Qtest <- (X_df[pairs[, 1], ] - X_df[pairs[, 2], ])/2

# variance estimations works when just Q is specified
test_that("variance estimator works with Q specified, X unspecified", {
  expect_no_error(covAdj.variance(Y, Z2, Q = Qtest, pair = pairs, pmat = pmat, xi = 0.1))
})

# variance estimations works when both Q and X are specified
test_that("variance estimator works with X and Q specified", {
  expect_no_error(covAdj.variance(Y, Z2, X = X_df, Q = Qtest, pair = pairs, pmat = pmat, xi = 0.1))
})

#### Errors expected

Q_error <- matrix(1, nrow = 50, ncol = 51)

# variance estimations works when both Q and X are specified
test_that("Error since Q is a high-dimensional matrix (more columns than rows)", {
  expect_error(covAdj.variance(Y, Z2, X = X_df, Q = Q_error, pair = pairs, pmat = pmat, xi = 0.1),
               "Q matrix has more columns than rows.")
})


