% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/multiness_fit.R
\name{multiness_fit}
\alias{multiness_fit}
\title{Fit the MultiNeSS model}
\usage{
multiness_fit(A,model,self_loops,refit,tuning,tuning_opts,optim_opts)
}
\arguments{
\item{A}{An \eqn{n \times n \times m} array containing edge entries for
an undirected multiplex network on \eqn{n} nodes and \eqn{m} layers.}

\item{model}{A string which provides choice of model,
either \code{'gaussian'} or \code{'logistic'}. Defaults to \code{'gaussian'}.}

\item{self_loops}{A Boolean, if \code{FALSE}, all diagonal entries are ignored in
optimization. Defaults to \code{TRUE}.}

\item{refit}{A Boolean, if \code{TRUE}, a refitting step is performed to
debias the eigenvalues of the estimates. Defaults to \code{TRUE}.}

\item{tuning}{A string which provides the tuning method, valid options are
\code{'fixed'}, \code{'adaptive'}, or \code{'cv'}. Defaults to \code{'adaptive'}.}

\item{tuning_opts}{A list, containing additional optional arguments controlling
parameter tuning. The arguments used depends on the choice of tuning method.
If \code{tuning='fixed'}, \code{multiness_fit} will utilize the following
arguments:
\describe{
\item{lambda}{A positive scalar,
the \eqn{\lambda} parameter in the nuclear norm penalty, see Details.
Defaults to \code{2.309 * sqrt(n*m)}.}
\item{alpha}{A positive scalar or numeric vector of length \code{m}, the parameters \eqn{\alpha_k} in the
nuclear norm penalty, see Details. If a scalar is provided all \eqn{\alpha_k} parameters are set to that
value. Defaults to \code{1/sqrt(m)}}
}
If \code{tuning='adaptive'}, \code{multiness_fit} will utilize the following
arguments:
\describe{
\item{layer_wise}{A Boolean, if \code{TRUE}, the entry-wise variance
is estimated individually for each layer. Otherwise the estimates are
pooled. Defaults to \code{TRUE}.}
\item{penalty_const}{A positive scalar \eqn{C} which scales the
penalty parameters (see Details).
Defaults to \code{2.309}.}
\item{penalty_const_lambda}{A positive scalar \eqn{c} which scales only the \eqn{\lambda}
penalty parameter (see Details).
Defaults to \code{1}.}
}
If \code{tuning='cv'}, \code{multiness_fit} will utilize the following
arguments:
\describe{
\item{layer_wise}{A Boolean, if \code{TRUE}, the entry-wise variance
is estimated individually for each layer. Otherwise the estimates are
pooled. Defaults to \code{TRUE}.}
\item{N_cv}{A positive integer, the number of repetitions of edge
cross-validation performed for each parameter setting. Defaults to \code{3}.}
\item{p_cv}{A positive scalar in the interval (0,1), the proportion
of edge entries held out in edge cross-validation. Defaults to \eqn{0.1}.}
\item{penalty_const_lambda}{A positive scalar \eqn{c} which scales only the \eqn{\lambda}
penalty parameter (see Details).
Defaults to \code{1}.}
\item{penalty_const_vec}{A numeric vector with positive entries, the candidate
values of constant \eqn{C} to scale the penalty parameters (see Details).
An optimal constant is chosen by edge cross-validation. Defaults to
\code{c(1,1.5,...,3.5,4)}.}
\item{refit_cv}{A Boolean, if \code{TRUE}, a refitting step is
performed when fitting the model for edge cross-validation. Defaults
to \code{TRUE}}
\item{verbose_cv}{A Boolean, if \code{TRUE}, console output will
provide updates on the progress of edge cross-validation. Defaults
to \code{FALSE}.}
}}

\item{optim_opts}{A list, containing additional optional arguments controlling
the proximal gradient descent algorithm.
\describe{
\item{check_obj}{A Boolean, if \code{TRUE}, convergence is determined
by checking the decrease in the objective. Otherwise it is determined by
checking the average entry-wise difference in consecutive values of \eqn{F}.
Defaults to \code{TRUE}.}
\item{eig_maxitr}{A positive integer, maximum iterations for internal
eigenvalue solver. Defaults to \code{1000}.}
\item{eig_prec}{A positive scalar, estimated eigenvalues below this
threshold are set to zero. Defaults to \code{1e-2}.}
\item{eps}{A positive scalar, convergence threshold for proximal gradient
descent. Defaults to \code{1e-6}.}
\item{eta}{A positive scalar, step size for proximal gradient descent.
Defaults to \code{1} for the Gaussian model, \code{5} for the logistic
model.}
\item{init}{A string, initialization method. Valid options are
\code{'fix'} (using initializers \code{optim_opts$V_init} and
\code{optim_opts$U_init}), \code{'zero'} (initialize all parameters at zero),
or \code{'svd'} (initialize with a truncated SVD with rank \code{optim_opts$init_rank}).
Defaults to \code{'zero'}.}
\item{K_max}{A positive integer, maximum iterations for proximal gradient
descent. Defaults to \code{100}.}
\item{max_rank}{A positive integer, maximum rank for internal eigenvalue
solver. Defaults to \code{sqrt(n)}.}
\item{missing_pattern}{An \eqn{n \times n \times m} Boolean array with \code{TRUE}
for each observed entry and \code{FALSE} for missing entries. If unspecified, it
is set to \code{!is.na(A)}.}
\item{positive}{A Boolean, if \code{TRUE}, singular value thresholding only retains
positive eigenvalues. Defaults to \code{FALSE}.}
\item{return_posns}{A Boolean, if \code{TRUE}, returns estimates
of the latent positions based on ASE. Defaults to \code{FALSE}.}
\item{verbose}{A Boolean, if \code{TRUE}, console output will provide
updates on the progress of proximal gradient descent. Defaults to
\code{FALSE}.}
}}
}
\value{
A list is returned with the MultiNeSS model estimates, dimensions of
the common and individual latent spaces, and some additional optimization
output:
\item{F_hat}{An \eqn{n \times n} matrix estimating the common part of the expected
adjacency matrix, \eqn{F = VV^{T}}. If \code{optim_opts$return_posns}
is \code{TRUE}, this is not returned.}
\item{G_hat}{A list of length \eqn{m}, the collection of \eqn{n \times n} matrices
estimating the individual part of each adjacency matrix, \eqn{G_k = U_kU_k^{T}}.
If \code{optim_opts$return_posns}
is \code{TRUE}, this is not returned.}
\item{V_hat}{A matrix estimating the common latent positions.
Returned if \code{optim_opts$return_posns} is \code{TRUE}.}
\item{U_hat}{A list of length \eqn{m}, the collection of matrices
estimating the individual latent positions.
Returned if \code{optim_opts$return_posns} is \code{TRUE}.}
\item{d1}{A non-negative integer, the estimated common dimension of the
latent space.}
\item{d2}{An integer vector of length \eqn{m}, the estimated individual
dimension of the latent space for each layer.}
\item{K}{A positive integer, the number of iterations run in proximal
gradient descent.}
\item{convergence}{An integer convergence code, \code{0} if proximal
gradient descent converged in fewer than \code{optim_opts$K_max} iterations,
\code{1} otherwise.}
\item{lambda}{A positive scalar, the tuned \eqn{\lambda}
penalty parameter (see Details).}
\item{alpha}{A numeric vector of length \eqn{m}, the tuned \eqn{\alpha} penalty parameters
(see Details).}
}
\description{
\code{multiness_fit} fits the Gaussian or logistic MultiNeSS model
with various options for parameter tuning.
}
\details{
A MultiNeSS model is fit to an \eqn{n \times n \times m} array \eqn{A} of
symmetric adjacency matrices on a common set of nodes. Fitting
proceeds by convex proximal gradient descent on the entries of
\eqn{F = VV^{T}} and \eqn{G_k = U_kU_k^{T}}, see
\href{https://arxiv.org/abs/2012.14409}{MacDonald et al., (2020)},
Section 3.2. Additional optional arguments for
the gradient descent routine can be provided in \code{optim_opts}.
\code{refit} provides an option
to perform an additional refitting step to debias the eigenvalues
of the estimates, see
\href{https://arxiv.org/abs/2012.14409}{MacDonald et al., (2020)}, Section 3.3.

By default, \code{multiness_fit} will return estimates of the matrices
\eqn{F} and \eqn{G_k}. \code{optim_opts$return_posns} provides an option
to instead return estimates of latent positions \eqn{V} and \eqn{U_k}
based on the adjacency spectral embedding (if such a factorization exists).

Tuning parameters \eqn{\lambda} and \eqn{\alpha_k} in the nuclear norm penalty
\deqn{\lambda ||F||_* + \sum_k \lambda \alpha_k ||G_k||_*}
are either set by the
user (\code{tuning='fixed'}), selected adaptively using a
robust estimator of the
entry-wise variance (\code{tuning='adaptive'}), or
selected using edge cross-validation (\code{tuning='cv'}). For more details
see \href{https://arxiv.org/abs/2012.14409}{MacDonald et al., (2020)},
Section 3.4. Additional optional arguments for parameter tuning
can be provided in \code{tuning_opts}.
}
\examples{
# gaussian model data
data1 <- multiness_sim(n=100,m=4,d1=2,d2=2,
                     model="gaussian")

# multiness_fit with fixed tuning
fit1 <- multiness_fit(A=data1$A,
                      model="gaussian",
                      self_loops=TRUE,
                      refit=FALSE,
                      tuning="fixed",
                      tuning_opts=list(lambda=40,alpha=1/2),
                      optim_opts=list(max_rank=20,verbose=TRUE))

# multiness_fit with adaptive tuning
fit2 <- multiness_fit(A=data1$A,
                      refit=TRUE,
                      tuning="adaptive",
                      tuning_opts=list(layer_wise=FALSE),
                      optim_opts=list(return_posns=TRUE))

# logistic model data
data2 <- multiness_sim(n=100,m=4,d1=2,d2=2,
                       model="logistic",
                       self_loops=FALSE)

# multiness_fit with cv tuning
fit3 <- multiness_fit(A=data2$A,
                      model="logistic",
                      self_loops=FALSE,
                      tuning="cv",
                      tuning_opts=list(N_cv=2,
                                       penalty_const_vec=c(1,2,2.309,3),
                                       verbose_cv=TRUE))

}
