% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bottom_up.R
\name{multiscale.bottomUp}
\alias{multiscale.bottomUp}
\title{Multiscale MOSUM algorithm with bottom-up merging}
\usage{
multiscale.bottomUp(
  x,
  G = bandwidths.default(length(x), G.min = max(20, ceiling(0.05 * length(x)))),
  threshold = c("critical.value", "custom")[1],
  alpha = 0.1,
  threshold.function = NULL,
  eta = 0.4,
  do.confint = FALSE,
  level = 0.05,
  N_reps = 1000,
  ...
)
}
\arguments{
\item{x}{input data (a \code{numeric} vector or an object of classes \code{ts} and \code{timeSeries})}

\item{G}{a vector of (symmetric) bandwidths, given as either integers less than \code{length(x)/2},
or numbers between \code{0} and \code{0.5} describing the moving sum bandwidths relative to \code{length(x)}.
If the smallest bandwidth is smaller than \code{min(20, 0.05*length(x))} 
(\code{0.05} if relative bandwidths are given) and \code{threshold = "critical.value"},
it generates a warning message}

\item{threshold}{string indicating which threshold should be used to determine significance.
By default, it is chosen from the asymptotic distribution at the given significance level \code{alpha}.
Alternatively, it is possible to parse a user-defined function with \code{threshold.function}}

\item{alpha}{a numeric value for the significance level with
\code{0 <= alpha <= 1}; use iff \code{threshold = "critical.value"}}

\item{threshold.function}{function object of form \code{function(G, length(x), alpha)}, to compute a
threshold of significance for different bandwidths G; use iff \code{threshold = "custom"}}

\item{eta}{see \link[mosum]{mosum}}

\item{do.confint}{flag indicating whether to compute the confidence intervals for change points}

\item{level}{use iff \code{do.confint = TRUE}; a numeric value (\code{0 <= level <= 1}) with which
\code{100(1-level)\%} confidence interval is generated}

\item{N_reps}{use iff \code{do.confint = TRUE}; number of bootstrap replicates to be generated}

\item{...}{further arguments to be passed to the \link[mosum]{mosum} calls}
}
\value{
S3 object of class \code{multiscale.cpts}, which contains the following fields:
   \item{x}{input data}
   \item{cpts}{estimated change points}
   \item{cpts.info}{data frame containing information about estimated change points}
   \item{pooled.cpts}{set of change point candidates that have been considered by the algorithm}
   \item{G}{bandwidths}
   \item{threshold, alpha, threshold.function}{input parameters}
   \item{eta}{input parameters}
   \item{do.confint}{input parameter}
   \item{ci}{object of class \code{cpts.ci} containing confidence intervals for change points iff \code{do.confint = TRUE}}
}
\description{
Multiscale MOSUM procedure with symmetric bandwidths combined with
bottom-up bandwidth-based merging.
}
\details{
See Algorithm 1 in the first referenced paper for a comprehensive
description of the procedure and further details.
}
\examples{
x1 <- testData(lengths = c(100, 200, 300, 300), 
means = c(0, 1, 2, 2.7), sds = rep(1, 4), seed = 123)$x
mbu1 <- multiscale.bottomUp(x1)
plot(mbu1)
summary(mbu1)

x2 <- testData(model = "mix", seed = 1234)$x
threshold.custom <- function(G, n, alpha) {
mosum.criticalValue(n, G, G, alpha) * log(n/G)^0.1
}
mbu2 <- multiscale.bottomUp(x2, G = 10:40, threshold = "custom",
threshold.function = threshold.custom)
plot(mbu2)
summary(mbu2)

}
\references{
A. Meier, C. Kirch and H. Cho (2021)
mosum: A Package for Moving Sums in Change-point Analysis.
\emph{Journal of Statistical Software}, Volume 97, Number 8, pp. 1-42.
<doi:10.18637/jss.v097.i08>.

M. Messer et al. (2014)
A multiple filter test for the detection of rate changes in renewal processes with varying variance.
\emph{The Annals of Applied Statistics}, Volume 8, Number 4, pp. 2027-2067.

H. Cho and C. Kirch (2022) Bootstrap confidence intervals for multiple change points based on moving sum procedures. \emph{Computational Statistics & Data Analysis}, Volume 175, pp. 107552.
}
