#' Generates a cross section of a cylinder bundle
#'
#' Generates a cross section of a cylinder bundle with a given density and voxel
#' size. The cross section is generated by simulating a random distribution of
#' cylinders and computing the intersection of the cylinders with a plane. The
#' radii of the cylinders are drawn from a Gamma distribution fitted from data
#' retrieved by histology in the genu of the corpus callosum (Aboitiz et al.,
#' 1992). The number of cylinders is determined by the density and the voxel
#' size.
#'
#' @references Aboitiz, F., Scheibel, A. B., Fisher, R. S., & Zaidel, E. (1992).
#'   Fiber composition of the human corpus callosum. Brain research, 598(1-2),
#'   143-153.
#'
#' @param density A numeric value between 0 and 1 specifying the density of the
#'   cylinders in the voxel. Defaults to `0.5`.
#' @param voxel_size A numeric value specifying the size of the voxel in micro-
#'  meters. Defaults to `10`.
#' @param rel_tol A numeric value specifying the relative tolerance to reach the
#'   target volume defined as `density * voxel_size^2`. Defaults to `1e-3`.
#' @param verbose A logical value specifying whether to print messages. Defaults
#'   to `FALSE`.
#'
#' @return A list with the following components:
#' - `sections`: A numeric matrix with 3 columns:
#'   - `x`: The x-coordinates of the centers of the cylinders;
#'   - `y`: The y-coordinates of the centers of the cylinders;
#'   - `r`: The radii of the cylinders in micrometers.
#' - `voxel_size`: The size of the voxel in micrometers
#' @export
#'
#' @examples
#' density <- 0.5
#' voxel_size <- 5 # micrometers
#' withr::with_seed(1234, {
#'   out <- simulate_bundle(density, voxel_size)
#' })
#'
#' # Actual density in the simulated substrate
#' sum(out$sections[, "r"]^2 * pi) / voxel_size^2
simulate_bundle <- function(density = 0.5,
                            voxel_size = 10,
                            rel_tol = 1e-3,
                            verbose = FALSE) {
  target <- density * voxel_size^2
  kappa <- 5.3316
  scale <- 1.0242e-1
  tot_surf <- 0
  grd <- seq(0, 2 * pi, length.out = 100)
  unit_circle_sample <- cbind(cos(grd), sin(grd))
  x_centers <- numeric()
  y_centers <- numeric()
  radii <- numeric()
  while (abs(tot_surf - target) > rel_tol * target) {
    x <- stats::runif(1, min = -voxel_size / 2, max = voxel_size / 2)
    y <- stats::runif(1, min = -voxel_size / 2, max = voxel_size / 2)
    feasible <- TRUE
    if (length(radii) >= 1) {
      for (j in 1:length(radii)) {
        if ((x - x_centers[j])^2 + (y - y_centers[j])^2 < radii[j]^2) {
          feasible <- FALSE
          break
        }
      }
    }
    if (!feasible) next
    r <- stats::rgamma(1, shape = kappa, scale = scale)
    points <- cbind(
      x + r * unit_circle_sample[, 1],
      y + r * unit_circle_sample[, 2]
    )
    feasible <- all(points <= voxel_size / 2) && all(points >= -voxel_size / 2)
    if (!feasible) next
    if (length(radii) >= 1) {
      for (j in 1:length(radii)) {
        if (any((points[, 1] - x_centers[j])^2 +
                (points[, 2] - y_centers[j])^2 < radii[j]^2)) {
          feasible <- FALSE
          break
        }
      }
    }
    if (!feasible) next
    if (tot_surf + pi * r^2 > target)
      next
    tot_surf <- tot_surf + pi * r^2
    if (verbose)
      cli::cli_alert_info("Total surface area of {round(tot_surf, 3)} / {target}")
    x_centers <- c(x_centers, x)
    y_centers <- c(y_centers, y)
    radii <- c(radii, r)
  }

  sections <- cbind(x = x_centers, y = y_centers, r = radii)
  out <- list(sections = sections, voxel_size = voxel_size)
  class(out) <- c("bundle", class(out))
  out
}

#' Plots a cross section of a cylinder bundle using **ggplot2**
#'
#' Plots a cross section of a cylinder bundle from an object of class `bundle`
#' as generated by [`simulate_bundle()`] using
#' [**ggplot2**](https://ggplot2.tidyverse.org).
#'
#' @param object An object of class `bundle` as generated by
#'   [`simulate_bundle()`].
#' @param grid_size An integer value specifying the number of points on which
#'   the unit circle should be discretized to draw the spheres. Defaults to
#'   `100L`.
#' @param ... Additional arguments to be passed to the [`ggplot2::autoplot()`]
#'   method.
#'
#' @return A [`ggplot2::ggplot()`] object.
#' @export
#'
#' @importFrom ggplot2 autoplot
#' @importFrom rlang .data
#' @examples
#' density <- 0.5
#' voxel_size <- 5 # micrometers
#' withr::with_seed(1234, {
#'   out <- simulate_bundle(density, voxel_size)
#' })
#' ggplot2::autoplot(out)
autoplot.bundle <- function(object, grid_size = 100L, ...) {
  grd <- seq(0, 2 * pi, length.out = grid_size)
  voxel_size <- object$voxel_size
  sections <- object$sections |>
    purrr::array_tree(margin = 1) |>
    purrr::imap(\(vals, id) {
      res <- data.frame(
        x = vals[1] + vals[3] * cos(grd),
        y = vals[2] + vals[3] * sin(grd)
      )
      res$id <- id
      res
    })
  sections <- do.call(rbind, sections)
  sections |>
    ggplot2::ggplot(ggplot2::aes(.data$x, .data$y)) +
    ggplot2::geom_polygon(ggplot2::aes(group = .data$id)) +
    ggplot2::geom_hline(yintercept = -voxel_size / 2, color = "black") +
    ggplot2::geom_hline(yintercept =  voxel_size / 2, color = "black") +
    ggplot2::geom_vline(xintercept = -voxel_size / 2, color = "black") +
    ggplot2::geom_vline(xintercept =  voxel_size / 2, color = "black") +
    ggplot2::theme_void() +
    ggplot2::coord_equal() +
    ggplot2::theme(legend.position = "none")
}

#' Plots a cross section of a cylinder bundle
#'
#' @param x An object of class `bundle` as generated by
#'   [`simulate_bundle()`].
#' @inheritParams autoplot.bundle
#'
#' @return Nothing.
#' @export
#'
#' @importFrom graphics plot
#' @examples
#' density <- 0.5
#' voxel_size <- 5 # micrometers
#' withr::with_seed(1234, {
#'   out <- simulate_bundle(density, voxel_size)
#' })
#' plot(out)
plot.bundle <- function(x, grid_size = 100L, ...) {
  print(autoplot(x, grid_size, ...))
}

#' Plots a 3D representation of a cylinder bundle using **plotly**
#'
#' Plots a 3D representation of a cylinder bundle from an object of class
#' `bundle` as generated by [`simulate_bundle()`] using
#' [**plotly**](https://plotly.com/r/).
#'
#' @param b An object of class `bundle` as generated by
#'   [`simulate_bundle()`].
#' @param show_linear_mesh A logical value indicating whether the linear mesh of
#'   each cylinder should be displayed. Defaults to `FALSE` for computational
#'   efficiency.
#'
#' @return An HTML widget of class [`plotly::plotly`] storing the 3D
#'   visualization of the cylinder bundle.
#' @export
#'
#' @examples
#' density <- 0.5
#' voxel_size <- 5 # micrometers
#' withr::with_seed(1234, {
#'   out <- simulate_bundle(density, voxel_size)
#' })
#' plot3d(out)
plot3d <- function(b, show_linear_mesh = FALSE) {
  if (!inherits(b, "bundle"))
    cli::cli_abort("Input {.arg b} must be an object of class {.cls bundle}.")
  cylinders <- b$sections |>
    purrr::array_tree(margin = 1) |>
    purrr::map(\(v) cylinder_traces(
      r = v["r"], xs = v["x"], ys = v["y"], zs = 0, h = 1,
      surface_kw = list(opacity = 0.3), show_linear_mesh = show_linear_mesh,
      line_kw = list(opacity = 0.3, line = list(color = "#202020", width = 3))
    )) |>
    purrr::flatten()

  fig <- plotly::plot_ly()
  for (cylinder in cylinders)
    fig <- do.call(plotly::add_trace, c(list(fig), cylinder))
  fig
}
