#' Local mark correlation functions for inhomogeneous point patterns on Euclidean spaces.
#'
#' Local mark correlation functions for inhomogeneous point patterns on Euclidean spaces.
#'
#' @usage \method{lmcorrinhom}{ppp}(X,
#' ftype = c("variogram", "stoyan", "rcorr", "shimatani",
#'  "beisbart", "isham", "stoyancov", "schlather"),
#' r = NULL,
#' lambda = NULL,
#' method_lambda = c("kernel", "Voronoi"),
#' bw = bw.scott,
#' f = NULL,
#' method = c("density", "loess"),
#' correction = c("Ripley", "translate", "none"),
#' normalise = TRUE,
#' tol = 0.01,
#' ...)
#'
#' @param X An object of class ppp.
#' @param ftype Type of the test function \eqn{t_f}. Currently any selection of \code{"variogram", "stoyan", "rcorr", "shimatani", "beisbart", "isham", "stoyancov", "schlather"}.
#' @param r Optional. The values of the argument \code{r} at which the mark correlation function should be evaluated.
#' @param lambda Estimated intensity at data points. If not given, it will be estimated internally. See details.
#' @param method_lambda The method to be used for estimating intensity at data points, if \code{lambda = NULL}.
#' @param bw Bandwidth method to be used for estimating intensity at data points if \code{lambda = NULL} and \code{method_lambda = "kernel"}.
#' @param f  Optional. Test function \eqn{t_f} used in the definition of the mark correlation function. If \code{ftype} is given, \eqn{t_f} should be \code{NULL}.
#' @param method Type of smoothing, either \code{density} or \code{loess}.
#' @param correction Type of edge correction to be applied, either of \code{"Ripley", "translate", "none"}.
#' @param normalise If \code{normalise=FALSE}, only the numerator of the expression for the mark correlation function will be computed.
#' @param tol Tolerance used in the calculation of the conditional mean of marks. This is used only if \code{ftype} is \code{schlather}.
#' @param ... Arguments passed to \code{\link[spatstat.univar]{unnormdensity}} or \code{\link[stats]{loess}}.
#' @details
#' This function computes local mark correlation functions for an inhomogeneous point pattern in \eqn{\mathbb{R}^2}. See the details of test functions used in \code{\link[markstat]{mcorrinhom.ppp}}. Technical details are given in Eckardt and Moradi (2025) and Moradi and Eckardt (2025).
#' @examples
#'  library(spatstat.geom)
#'  library(spatstat.random)
#'  library(spatstat.explore)
#'  X <- rpoispp(function(x,y) {100 * exp(-3*x)}, 100)
#'  marks(X) <- runif(npoints(X), 1, 10)
#'  mcorrinhom.ppp(X, ftype = "stoyan",
#'                 method = "density", correction = "translate",
#'                 method_lambda = "kernel", bw = bw.scott)

#' @return A \code{data.frame} where the first column contains the values of the argument \eqn{r} at which the mark correlation function is evaluated, and the remaining columns contain the estimated values of the mark correlation function for each data point; column names correspond to the IDs of the data points.
#' If there are multiple numeric marks, the result will instead be a list, with each element corresponding to one of the marks.
#' 
#' @author Mehdi Moradi \email{m2.moradi@yahoo.com} and Matthias Eckardt
#' @references 
#' Moradi, M., & Eckardt, M. (2025). Inhomogeneous mark correlation functions for general marked point processes. arXiv e-prints, arXiv-2505.
#' 
#' Eckardt, M., & Moradi, M. (2025). Local indicators of mark association for marked spatial point processes.
#' @seealso \code{\link[markstat]{mcorrinhom.ppp}}.


#' @import spatstat.univar
#' @import spatstat.random
#' @import spatstat.explore
#' @import spatstat.geom
#' @import spatstat.utils
#' @import stats
#' @export

lmcorrinhom.ppp <- function(X,
                            ftype = c("variogram", "stoyan", "rcorr", "shimatani", "beisbart", "isham", "stoyancov", "schlather"),
                            r = NULL,
                            lambda = NULL,
                            method_lambda = c("kernel", "Voronoi"),
                            bw = bw.scott,
                            f = NULL,
                            method = c("density", "loess"),
                            correction = c("Ripley", "translate", "none"),
                            normalise = TRUE,
                            tol = 0.01,
                            ...){

  if (all(class(X) != "ppp")) stop("object X should be of class ppp.")
  
  if (is.null(f) & missing(ftype)) stop("ftype must be provided if 'f' is NULL.")
  
  if (missing(method)) stop("smoothing method should be chosen.")
  
  lambda_given <- lambda
  
  correction <- match.arg(correction, correction)
  n <- npoints(X)
  d <- pairdist(X)
  
  if(is.null(lambda)){
    
    if(method_lambda=="kernel"){
      
      lambda <- as.numeric(density(unmark(X), sigma = bw(X), at="points", diggle=T))
      
    }else if(method_lambda=="Voronoi"){
      
      lambda <- as.numeric(densityVoronoi(X, f=0.2, nrep = 100)[X])
      
    }else{
      
      stop("You need to pick a method for intensity estimation!")
      
    }
  }else{
    lambda <- lambda
  }
  
  if(is.null(r)){
    W <- X$window
    rmaxdefault <- rmax.rule("K", W, n/area(W))
    if(length(rmaxdefault)==0) {rmaxdefault <- 0.5 * max(d)}
    breaks <- handle.r.b.args(r, NULL, W, rmaxdefault = rmaxdefault)
    r <- breaks$r
  }
  
  rmax <- max(r)
  

  m <- marks(X)

  if (any(class(m) == "hyperframe" | class(m) == "data.frame")){
    m <- as.data.frame(m)
    num_cols <- unlist(sapply(m, is.numeric))
    s <- which(num_cols)
    
    out <- list()
    for (i in 1:length(s)) {
      marks(X) <- as.numeric(m[,s[i]])
      out[[i]] <- lmcorrinhom.ppp(X, ftype = ftype, r = r, lambda = lambda, f = f,
                                  method = method, correction = correction, normalise = normalise,  tol = tol, ...)
    }

    names(out) <- names(s)
    
    class(out) <- "mc"
    attr(out, "mtype") <- "real-valued"
    attr(out, "type") <- "local"
    attr(out, "ftype") <- ftype
    attr(out, "method") <- method
    attr(out, "lambda") <- lambda_given
    attr(out, "normalise") <- normalise
    attr(out, "method_lambda") <- method_lambda
    attr(out, "correction") <- correction
    attr(out, "bw") <- bw
    attr(out, "nmark") <- length(s)
    
    return(out)
  }

  if (is.null(f)) {
    if (ftype == "variogram") {
      f <- function(m1, m2, mu = NULL) 0.5 * ((m1 - m2)^2)
    } else if (ftype == "stoyan") {
      f <- function(m1, m2, mu = NULL) m1 * m2
    } else if (ftype == "rcorr") {
      f <- function(m1, m2, mu = NULL) m1
    } else if (ftype == "shimatani") {
      f <- function(m1, m2, mu = NULL) (m1 - mean(m)) * (m2 - mean(m))
    } else if (ftype == "beisbart") {
      f <- function(m1, m2, mu = NULL) m1 + m2
    } else if (ftype == "isham") {
      f <- function(m1, m2, mu = NULL) m1 * m2 - (mean(m))^2
    } else if (ftype == "stoyancov") {
      f <- function(m1, m2, mu = NULL) m1 * m2 - (mean(m))^2
    } else if (ftype == "schlather") {
      f <- function(m1, m2, mu = NULL) m1 * m2 - mu * (m1 + m2) + mu^2
    } else {
      stop("Your ftype is not supported!")
    }
  } else {
    warning("Your given test function is not among the default ones; only unnormalised version will be calculated.")
  }
  

  if(correction=="translate"){

    close <- closepairs(X, rmax)
    XI <- ppp(close$xi, close$yi, window = X$window, check = FALSE)
    XJ <- ppp(close$xj, close$yj, window = X$window, check = FALSE)
    edgewt <- edge.Trans(XI, XJ, paired = TRUE)

  }else if(correction=="Ripley"){

    close <- closepairs(X, rmax)
    XI <- ppp(close$xi, close$yi, window = X$window, check = FALSE)
    edgewt <- edge.Ripley(XI, matrix(close$d, ncol = 1))

  }else if(correction=="none"){

    close <- closepairs(X, rmax)
    edgewt <- rep(1, length(close$d))

  }

  df <- cbind(
    dist = as.vector(d),
    id.row = rep(c(1:n), each=n),
    id.col = rep(c(1:n), n),
    int_i = rep(lambda, each=n),
    int_j = rep(lambda, n)
  )

  df.filter <- df[df[,1]<= rmax & df[,1]>0,]
  m1 <- m[df.filter[,2]]
  m2 <- m[df.filter[,3]]

  if (ftype=="schlather"){
    df.filter <- cbind(df.filter,
                       mu = as.numeric(unlist(sapply(df.filter[,1], function(d) {
                         matched <- df.filter[,3][abs(df.filter[,1] - d) <= tol]
                         paste(mean(m[matched]), collapse = ",")
                       }))))
    mu <- df.filter[,6]
    dfvario <- data.frame(d = df.filter[,1],
                          ff = (f(m1, m2, mu)),
                          int = df.filter[,"int_i"]*df.filter[,"int_j"],
                          w = edgewt,
                          id.row=df.filter[,"id.row"],
                          id.col=df.filter[,"id.col"])
  }else{
    dfvario <- data.frame(d = df.filter[,1],
                          ff = (f(m1,m2)),
                          int = df.filter[,"int_i"]*df.filter[,"int_j"],
                          w = edgewt,
                          id.row=df.filter[,"id.row"],
                          id.col=df.filter[,"id.col"]
    )
  }


  Eff <- list()
  out <- list()

  for(i in 1:n){

    data <- dfvario[dfvario$id.row==i,]

    if(method=="density"){
      Kf <- unnormdensity(data$d, weights = data$w*data$ff/data$int,
                          from=min(r), to=max(r), n=length(r),...
      )$y
      ## smooth estimate of kappa_1
      K1 <- unnormdensity(data$d, weights=data$w/data$int,
                          from=min(r), to=max(r), n=length(r),...
      )$y
      Eff[[i]] <-  Kf/K1
    }
    else if(method=="loess"){
      lo <- loess(ff~d, data = data,
                  control = loess.control(surface="direct"))
      Eff[[i]] <- predict(lo, newdata=data.frame(d=r))
    }


    if(normalise){
      if(ftype=="stoyan"){
        mean.i <- mean(m[data[,"id.row"][1]] * m[data[,"id.col"]])
        out[[i]] <- Eff[[i]]/mean.i
      } else if(ftype=="variogram" | ftype=="isham" | ftype=="schlather" | ftype=="shimatani"){
        var.i <- var(c(m[data[,"id.row"][1]], m[data[,"id.col"]]))
        out[[i]] <- Eff[[i]]/var.i
      }else if(ftype=="rcorr"){
        mean.i <- mean(c(m[data[,"id.row"][1]], m[data[,"id.col"]]))
        out[[i]] <- Eff[[i]]/mean.i
      }else if(ftype=="Beisbart"){
        mean.i <- 2*mean(c(m[data[,"id.row"][1]], m[data[,"id.col"]]))
        out[[i]] <- Eff[[i]]/mean.i
      }else if(ftype=="stoyancov"){
        out[[i]] <- Eff[[i]]
      }
      else{
        stop("your ftype is not supported!!")
      }
    }else{
      out[[i]] <- Eff[[i]]
    }
  }

  out <- do.call(cbind, out)
  colnames(out) <- as.character(c(1:n))
  out <- cbind(r, out)
  
  if(ncol(out) == npoints(X) + 1 ) type <- "local" else type <- "global"
  
  class(out) <- "mc"
  attr(out, "mtype") <- "real-valued"
  attr(out, "type") <- "local"
  attr(out, "ftype") <- ftype
  attr(out, "method") <- method
  attr(out, "lambda") <- lambda_given
  attr(out, "normalise") <- normalise
  attr(out, "method_lambda") <- method_lambda
  attr(out, "correction") <- correction
  attr(out, "bw") <- bw
  
  
  return(out)
}
