% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mapboxer.R
\name{mapboxer-shiny}
\alias{mapboxer-shiny}
\alias{mapboxerOutput}
\alias{renderMapboxer}
\title{Shiny bindings for mapboxer}
\usage{
mapboxerOutput(outputId, width = "100\%", height = "400px")

renderMapboxer(expr, env = parent.frame(), quoted = FALSE)
}
\arguments{
\item{outputId}{output variable to read from}

\item{width, height}{Must be a valid CSS unit (like \code{'100\%'},
\code{'400px'}, \code{'auto'}) or a number, which will be coerced to a
string and have \code{'px'} appended.}

\item{expr}{An expression that generates a mapboxer}

\item{env}{The environment in which to evaluate \code{expr}.}

\item{quoted}{Is \code{expr} a quoted expression (with \code{quote()})? This
is useful if you want to save an expression in a variable.}
}
\description{
Output and render functions for using mapboxer within Shiny
applications and interactive Rmd documents.
}
\examples{
library(shiny)
library(mapboxer)

LAYER_ID <- "mvc"

view <- basicPage(
  h1("mapboxer"),
  sliderInput(
    "slider",
    "Number of persons injured",
    min = 0,
    max = max(motor_vehicle_collisions_nyc$injured),
    step = 1,
    value = 0
  ),
  checkboxInput("hide", "Hide layer"),
  mapboxerOutput("map"),
  htmlOutput("datetime")
)

server <- function(input, output) {
  output$map <- renderMapboxer({
      as_mapbox_source(motor_vehicle_collisions_nyc) \%>\%
      mapboxer(
        center = c(-73.9165, 40.7114),
        zoom = 10,
        style = basemap_raster_style(stamen_raster_tiles())
      ) \%>\%
      add_circle_layer(
        circle_color = "black",
        popup = "Number of persons injured {{injured}}",
        id = LAYER_ID
      ) \%>\%
      add_mouse_position_control(
        "Lng: {{lng}}</br>Lat: {{lat}}",
        css_text = "text-align: left; width: 180px;"
      ) \%>\%
      add_navigation_control(pos = "top-left")
  })

  observeEvent(input$slider, {
    mapboxer_proxy("map") \%>\%
      set_filter(LAYER_ID, list("==", "injured", input$slider)) \%>\%
      update_mapboxer()
  })

  observeEvent(input$hide, {
    mapboxer_proxy("map") \%>\%
      set_layout_property(LAYER_ID, "visibility", !input$hide) \%>\%
      update_mapboxer()
  })

  output$datetime <- renderText({
    props <- input$map_onclick$props
    sprintf("<p>\%s \%s</p>", props$date, props$time)
  })
}

if (interactive()) shinyApp(view, server)
}
