% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kde_adaptive.R
\name{adaptive_fnhat}
\alias{adaptive_fnhat}
\title{Specialized ``Workhorse'' Function for Kernel Adaptive Density Estimators}
\usage{
adaptive_fnhat(x, data, K, h, sigma, Ai, Bj, fnx, ticker = FALSE,
  plot = FALSE, parlist = NULL, ...)
}
\arguments{
\item{x}{Numeric vector \eqn{(x_1, \ldots, x_k)} of location(s) at which the
density estimate is to be computed.}

\item{data}{Numeric vector \eqn{(X_1, \ldots, X_n)} of the data from which
the estimate is to be computed.}

\item{K}{Kernel function with vectorized in- & output.}

\item{h}{Numeric scalar, where (usually) \eqn{h = n^{-1/5}}.}

\item{sigma}{Numeric vector \eqn{(\sigma_1, \ldots, \sigma_s)} with
\eqn{s \ge 1}.}

\item{Ai}{Numeric matrix expecting in its i-th row \eqn{(x_i - X_1, \ldots,
x_i - X_n)/h}, where (usually) \eqn{x_1, \ldots, x_k} with
\eqn{k =} \code{length(x)} are the points at which the density is
to be estimated for the data \eqn{X_1, \ldots, X_n} with
\eqn{h = n^{-1/5}}.}

\item{Bj}{Numeric vector expecting \eqn{(-J(1/n), \ldots, -J(n/n))} in
case of the rank transformation method, but \eqn{(\hat \theta -
X_1, \ldots, \hat \theta - X_n)} in case of the non-robust
Srihera-Stute-method.}

\item{fnx}{Numeric vector expecting \eqn{(f_n(x_1), \ldots, f_n(x_k))} with
\eqn{f_n(x_i)} the Parzen-Rosenblatt estimator at \eqn{x_i}, i.e.,
\eqn{f_n(x_i) =} \code{mean(K(Ai[i,]))/h} where here typically
\code{h} \eqn{= n^{-1/5}}.}

\item{ticker}{Logical; determines if a 'ticker' documents the iteration
progress through \code{sigma}. Defaults to FALSE.}

\item{plot}{Logical or character or numeric and indicates if graphical
output should be produced. Defaults to \code{FALSE} (i.e., no
graphical output is produced). If it is a character string or
a numeric value, graphical output will be written to numbered
pdf-files (one for each element of \code{x}, in the current
working directory) whose names start with the provided
``value'' after converting it into a character string
followed by the index number of the pertaining
\code{x}-element. (Parts of the graphical output are
generated by \code{\link{minimize_MSEHat}}.)}

\item{parlist}{A list of graphical parameters; affects only the pdf-files
(if any are created at all). Default: \code{NULL}.}

\item{\ldots}{Possible further arguments passed to \code{minimize_MSEHat()}
(where they are currently ignored).}
}
\value{
A list of as many lists as elements in \code{x}, each with components
  \code{x}, \code{y}, \code{sigma.adap}, \code{msehat.min},
  \code{discr.min.smaller}, and \code{sig.range.adj} whose meanings are as
  follows:
  \tabular{ll}{
   \code{x} \tab the n coordinates of the points where the density is
            estimated. \cr
   \code{y} \tab the estimate of the density value \eqn{f(x)}. \cr
   \code{sigma.adap} \tab Minimizer of MSE-estimator (from function
                     \code{\link{minimize_MSEHat}}). \cr
   \code{msehat.min} \tab Minimum of MSE-estimator (from function
                     \code{\link{minimize_MSEHat}}). \cr
   \code{discr.min.smaller} \tab TRUE iff the numerically found minimum was
                            smaller than the discrete one (from function
                            \code{\link{minimize_MSEHat}}). \cr
   \code{sig.range.adj} \tab Number of adjustments of sigma-range. \cr
   }
}
\description{
Common specialized computational ``workhorse'' function to compute the kernel
adaptive density estimators both in eq. (1.6) of Srihera & Stute (2011) and
in eq. (4) of Eichner & Stute (2013) (together with several related
quantities) with a \eqn{\sigma} that minimizes the estimated MSE using an
estimated \eqn{\theta}. This function is ``specialized'' in that it expects
some pre-computed quantities (in addition to the point(s) at which the
density is to be estimated, the data, etc.). In particular, the estimator of
\eqn{\theta} (which is typically the arithmetic mean of the data) is
expected to be already ``contained'' in those pre-computed quantities, which
increases the computational efficiency.
}
\details{
The computational procedure in this function can be highly iterative because
for each point in \code{x} (and hence for each row of matrix \code{Ai}) the
MSE estimator is computed as a function of \eqn{\sigma} on a (usually fine)
\eqn{\sigma}-grid provided through \code{sigma}. This happens by repeated
calls to \code{\link{bias_AND_scaledvar}()}. The minimization in \eqn{\sigma}
is then performed by \code{\link{minimize_MSEHat}()} using both a discrete
grid-search and the numerical optimization routine implemented in base R's
\code{optimize()}. Finally, \code{\link{compute_fnhat}()} yields the actual
value of the density estimator for the adapted \eqn{\sigma}, i.e., for the
MSE-estimator-minimizing \eqn{\sigma}.
(If necessary the computation over the \eqn{\sigma}-grid is repeated after
extending the range of the grid until the estimator functions for both bias
and variance are \emph{not constant} across the \eqn{\sigma}-grid.)
}
\examples{
\dontrun{
require(stats)

 # Kernel adaptive density estimators for simulated N(0,1)-data
 # computed on an x-grid using the rank transformation and the
 # non-robust method:
set.seed(2017);     n <- 100;     Xdata <- sort(rnorm(n))
x <- seq(-4, 4, by = 0.5);     Sigma <- seq(0.01, 10, length = 51)
h <- n^(-1/5)

x.X_h <- outer(x/h, Xdata/h, "-")
fnx <- rowMeans(dnorm(x.X_h)) / h   # Parzen-Rosenblatt estim. at
                                    # x_j, j = 1, ..., length(x).
 # non-robust method:
theta.X <- mean(Xdata) - Xdata
adaptive_fnhat(x = x, data = Xdata, K = dnorm, h = h, sigma = Sigma,
  Ai = x.X_h, Bj = theta.X, fnx = fnx, ticker = TRUE, plot = TRUE)

 # rank transformation-based method (requires sorted data):
negJ <- -J_admissible(1:n / n)   # rank trafo
adaptive_fnhat(x = x, data = Xdata, K = dnorm, h = h, sigma = Sigma,
  Ai = x.X_h, Bj = negJ, fnx = fnx, ticker = TRUE, plot = TRUE)
}

}
\references{
Srihera & Stute (2011) and Eichner & Stute (2013): see
            \link{kader}.
}
