#' Launch the Interactive Shiny Application
#'
#' @description
#' Launches the interactive Shiny dashboard for personalized multi-state
#' joint modeling with semi-parametric association surfaces. The app provides
#' data upload, exploration, model fitting, personalized prediction, and
#' surface visualization.
#'
#' @param ... Additional arguments passed to \code{shiny::runApp}.
#'
#' @return No return value, called for the side effect of launching an
#'   interactive Shiny application in the user's default web browser.
#'
#' @details
#' The Shiny app requires additional packages: \code{shiny}, \code{shinydashboard},
#' \code{shinyWidgets}, \code{ggplot2}, \code{viridis}, \code{plotly}, \code{dplyr},
#' \code{tidyr}, \code{DT}, \code{gridExtra}.
#'
#' @export
run_shiny_app <- function(...) {
  app_dir <- system.file("shiny", package = "jmSurface")
  if (app_dir == "") {
    stop("Shiny app not found. Try reinstalling jmSurface.")
  }
  if (!requireNamespace("shiny", quietly = TRUE)) {
    stop("Package 'shiny' is required. Install with: install.packages('shiny')")
  }
  shiny::runApp(app_dir, ...)
}


#' jmSurface: Semi-Parametric Association Surfaces for Joint Models
#'
#' @description
#' The \pkg{jmSurface} package implements interpretable multi-biomarker fusion
#' in joint longitudinal-survival models via semi-parametric association surfaces
#' for multi-state disease processes.
#'
#' @section Main functions:
#' \describe{
#'   \item{\code{\link{jmSurf}}}{Fit the two-stage joint model}
#'   \item{\code{\link{fit_longitudinal}}}{Stage 1: fit longitudinal submodels}
#'   \item{\code{\link{fit_gam_cox}}}{Stage 2: fit GAM-Cox with tensor-product surface}
#'   \item{\code{\link{edf_diagnostics}}}{Extract EDF and complexity diagnostics}
#'   \item{\code{\link{dynPred}}}{Dynamic prediction of transition probabilities}
#'   \item{\code{\link{plot_surface}}}{3D association surface visualization}
#'   \item{\code{\link{contour_heatmap}}}{Contour heatmap of danger zones}
#'   \item{\code{\link{marginal_slices}}}{Marginal effect slice plots}
#'   \item{\code{\link{simulate_jmSurface}}}{Simulate multi-state data}
#'   \item{\code{\link{run_shiny_app}}}{Launch interactive Shiny dashboard}
#' }
#'
#' @references
#' Bhattacharjee, A. (2025). Interpretable Multi-Biomarker Fusion in Joint
#' Longitudinal-Survival Models via Semi-Parametric Association Surfaces.
#'
#' @importFrom nlme lme lmeControl fixef ranef VarCorr
#' @importFrom survival Surv coxph basehaz survfit
#' @importFrom mgcv gam te s predict.gam summary.gam
#' @importFrom stats coef predict fitted residuals model.matrix
#'   rnorm runif rexp rbinom quantile median setNames
#'   as.formula complete.cases pnorm qnorm
#' @importFrom utils read.csv
#' @importFrom graphics par plot lines points legend title axis
#'   contour filled.contour image text abline mtext persp
#' @importFrom grDevices colorRampPalette hcl.colors
#'
#' @docType package
#' @name jmSurface-package
"_PACKAGE"
