#' Calculate the rho value that reaches the maximum type I error rate in the improved trimmed Hochberg method when k value is given
#'
#' @param k a pre-specified constant in the improved trimmed weighted Hochberg procedure
#' @param alpha the significance level
#' @param alphavec a numeric vector of two values representing the weighted significance levels assigned to the two hypotheses
#'
#' @returns the rho value that makes the type I error rate reaches the maximum value `rho_opt` and the type I error rate `errorrate`
#' @export
#'
#' @importFrom stats uniroot
#' @author Jiangtao Gou
#' @author Fengqing Zhang
#' @references
#' Gou, J., Chang, Y., Li, T., and Zhang, F. (2025). Improved trimmed weighted Hochberg procedures with two endpoints and sample size optimization. Technical Report.
#' @examples
#' optrho(k = 2/3, alpha = 0.025)
optrho <- function (k, alpha, alphavec = c(alpha/2, alpha/2)) {
  # R20241109b
  # Rfun_optk
  find_rho_result <- stats::uniroot(f = find_rho_target_mvtnorm, interval = c(0, 0.5), tol = .Machine$double.eps^0.8, k = k, alpha = alpha, alphavec = alphavec)
  rho_opt <- find_rho_result$root
  #
  errorrate <- find_k_given_rho_target_mvtnorm(k = k, rho = rho_opt, alpha = alpha, alphavec = alphavec) + alpha
  #
  return (list(rho_opt = rho_opt, errorrate = errorrate))
}
