% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pr_0_total_ibd.R
\name{pr_0_total_ibd}
\alias{pr_0_total_ibd}
\title{Probability of no IBD across one or more chromosomes}
\usage{
pr_0_total_ibd(
  relationship_type = c("cousin", "grandparent", "halfsib", "uncle"),
  degree,
  removal,
  removal1,
  removal2,
  chromosome_length,
  log10 = FALSE
)
}
\arguments{
\item{relationship_type}{One of \code{"cousin"}, \code{"halfsib"},
\code{"grandparent"} or \code{"uncle"}.}

\item{degree}{See details.}

\item{removal}{See details.}

\item{removal1}{See details.}

\item{removal2}{See details.}

\item{chromosome_length}{Default is 267.77 cM (an estimate of the length of chromosome 1).}

\item{log10}{Should the log10 probability be returned? Default is \code{FALSE}.}
}
\value{
Numeric
}
\description{
Donnelly (1983) studies the probability that relatives of certain types have
no segments of a chromosome in common and provides expressions that can be
efficiently computed.
}
\details{
Types of relationships supported are:
\describe{
\item{\code{cousin}}{Use \code{degree = a} and \code{removal = b} for \code{a}'th cousins \code{b} times removed with degree at least one. Default is \code{degree = 1}.}
\item{\code{halfsib}}{Use \code{removal = 0} (default) for half-siblings and \code{removal = a} for half-cousins \code{a} times removed.}
\item{\code{grandparent}}{\code{degree = 1} (default) for grandparents, 2 for great-grandparents and so on.}
\item{\code{uncle}}{Use \code{degree = 0} (default) for uncle and \code{degree = d} for great^d uncle}
}
}
\examples{
## Cousin-type: third cousins on a chromosome of length 100 cM
degree <- 3

p0_3C <- pr_0_total_ibd("cousin", degree = 3, chromosome_length = 100)
p0_3C

# verify
p0_3C_manual <- d_cibd(x = 100, ibd = 0,
                       pedigree = pedtools::cousinPed(degree = 3))

p0_3C_manual
stopifnot(all.equal(p0_3C, p0_3C_manual))

## Half-sib type: half-cousins twice removed
p0_H1C_2R <- pr_0_total_ibd("halfsib",
                            degree = 1, removal = 2, chromosome_length = 100)
p0_H1C_2R

p0_H1C_2R_manual <- d_cibd(x = 100, ibd = 0,
                           pedigree = pedtools::halfCousinPed(removal = 2))

p0_H1C_2R_manual

stopifnot(all.equal(p0_H1C_2R, p0_H1C_2R_manual))

## Grandparent-type: great grandparents (degree = 2)
p0_GGP <- pr_0_total_ibd("grandparent", degree = 2, chromosome_length = 100)
p0_GGP

# GGP is a third generation ancestor so n = 3
p0_GGP_manual <- d_cibd(x = 100, ibd = 0,
                    pedigree = pedtools::linearPed(n = 3),
                    ids = c(1, pedtools::leaves(pedtools::linearPed(n = 3))))

p0_GGP_manual

stopifnot(all.equal(p0_GGP, p0_GGP_manual))

## Uncle-type: degree = 0 for uncle
p0_AV <- pr_0_total_ibd("uncle", chromosome_length = 100)
p0_AV

p0_AV_manual <- d_cibd(x = 100, ibd = 0, pedigree = pedtools::avuncularPed())

p0_AV_manual

stopifnot(all.equal(p0_AV, p0_AV_manual))

## Reproduce Table 1 of Donnelly (1983)
# (historic) chromosome lengths (cM) used in Donnelly (1983)
L <- 33 * c(9.12, 8.53, 7.16, 6.59, 6.15, 5.87, 5.31, 4.92, 4.81, 4.71, 4.60,
            4.47, 3.56, 3.60, 3.40, 3.20, 3.12, 2.72, 2.48, 2.27, 1.77, 1.64)
k <- 4:15

tab1 <- data.frame(k=k)

tab1$cousin <- pr_0_total_ibd(relationship_type = "cousin",
                              degree = rep(2:7, each = 2),
                              removal = rep(0:1, times = 6),
                              chromosome_length = L)

tab1$uncle <- pr_0_total_ibd(relationship_type = "uncle",
                             degree = k - 1, chromosome_length = L)

# Note the removal on one side only
tab1$halfsib <- pr_0_total_ibd(relationship_type = "halfsib",
                               removal1 = k - 1,
                               removal2 = rep(0, length(k)),
                               chromosome_length = L)

# Poisson approximation
tab1$`exp(-33 * k / 2^k)` <- exp(-33 * k / 2^k)

# Note that k corresponds to great^k grandparent,
# i.e. a (k+2)'th generation ancestor
# (not great^(k-1) and (k+1)'th generation ancestor as printed)

tab1$grandparent <- pr_0_total_ibd(relationship_type = "grandparent",
                                   degree = k, chromosome_length = L)

tab1
}
\references{
Donnelly K. P. (1983). The probability that related individuals
share some section of genome identical by descent.
Theoretical population biology, 23(1), 34–63.
https://doi.org/10.1016/0040-5809(83)90004-7
}
\seealso{
\link{pr_all_genes_survive} for the probability that all autosomal genes
are passed on to the next generation (offspring column in Table 1
of Donnelly (1983))
}
