/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5ESprivate.h" 
#include "H5FLprivate.h" 
#include "H5FOprivate.h" 
#include "H5Iprivate.h"  
#include "H5Lprivate.h"  
#include "H5MMprivate.h" 
#include "H5Oprivate.h"  
#include "H5Pprivate.h"  
#include "H5Tpkg.h"      
#include "H5VLprivate.h" 

static herr_t H5T__commit_api_common(hid_t loc_id, const char *name, hid_t type_id, hid_t lcpl_id,
                                     hid_t tcpl_id, hid_t tapl_id, void **token_ptr,
                                     H5VL_object_t **_vol_obj_ptr);
static hid_t  H5T__open_api_common(hid_t loc_id, const char *name, hid_t tapl_id, void **token_ptr,
                                   H5VL_object_t **_vol_obj_ptr);
static H5T_t *H5T__open_oid(const H5G_loc_t *loc);
static herr_t H5T_destruct_datatype(void *datatype, H5VL_connector_t *vol_connector);

static herr_t
H5T__commit_api_common(hid_t loc_id, const char *name, hid_t type_id, hid_t lcpl_id, hid_t tcpl_id,
                       hid_t tapl_id, void **token_ptr, H5VL_object_t **_vol_obj_ptr)
{
    void           *data        = NULL; 
    H5VL_object_t  *new_obj     = NULL; 
    H5T_t          *dt          = NULL; 
    H5VL_object_t  *tmp_vol_obj = NULL; 
    H5VL_object_t **vol_obj_ptr =
        (_vol_obj_ptr ? _vol_obj_ptr : &tmp_vol_obj); 
    H5VL_loc_params_t loc_params;                     
    herr_t            ret_value = SUCCEED;            

    FUNC_ENTER_PACKAGE

    
    if (!name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "name parameter cannot be NULL");
    if (!*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "name parameter cannot be an empty string");
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (H5T_is_named(dt))
        HGOTO_ERROR(H5E_ARGS, H5E_CANTSET, FAIL, "datatype is already committed");

    
    if (H5P_DEFAULT == lcpl_id)
        lcpl_id = H5P_LINK_CREATE_DEFAULT;
    else if (true != H5P_isa_class(lcpl_id, H5P_LINK_CREATE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not link creation property list");

    
    if (H5P_DEFAULT == tcpl_id)
        tcpl_id = H5P_DATATYPE_CREATE_DEFAULT;
    else if (true != H5P_isa_class(tcpl_id, H5P_DATATYPE_CREATE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not datatype creation property list");

    
    H5CX_set_lcpl(lcpl_id);

    
    if (H5VL_setup_acc_args(loc_id, H5P_CLS_TACC, true, &tapl_id, vol_obj_ptr, &loc_params) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSET, FAIL, "can't set object access arguments");

    
    if (NULL == (data = H5VL_datatype_commit(*vol_obj_ptr, &loc_params, name, type_id, lcpl_id, tcpl_id,
                                             tapl_id, H5P_DATASET_XFER_DEFAULT, token_ptr)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to commit datatype");

    
    if (NULL == (new_obj = H5VL_create_object(data, H5VL_OBJ_CONNECTOR(*vol_obj_ptr))))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL, "can't create VOL object for committed datatype");

    
    dt->vol_obj = new_obj;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Tcommit2(hid_t loc_id, const char *name, hid_t type_id, hid_t lcpl_id, hid_t tcpl_id, hid_t tapl_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if ((ret_value = H5T__commit_api_common(loc_id, name, type_id, lcpl_id, tcpl_id, tapl_id, NULL, NULL)) <
        0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPENOBJ, FAIL, "unable to commit datatype synchronously");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Tcommit_async(const char *app_file, const char *app_func, unsigned app_line, hid_t loc_id, const char *name,
                hid_t type_id, hid_t lcpl_id, hid_t tcpl_id, hid_t tapl_id, hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    herr_t         ret_value = SUCCEED;         

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token;

    
    if ((ret_value = H5T__commit_api_common(loc_id, name, type_id, lcpl_id, tcpl_id, tapl_id, token_ptr,
                                            &vol_obj)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPENOBJ, FAIL, "unable to commit datatype asynchronously");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE10(__func__, "*s*sIui*siiiii", app_file, app_func, app_line, loc_id, name, type_id, lcpl_id, tcpl_id, tapl_id, es_id)) < 0)
            
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINSERT, FAIL, "can't insert token into event set");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5T__commit_named(const H5G_loc_t *loc, const char *name, H5T_t *dt, hid_t lcpl_id, hid_t tcpl_id)
{
    H5O_obj_create_t ocrt_info;           
    H5T_obj_create_t tcrt_info;           
    H5T_state_t      old_state;           
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(loc);
    assert(name && *name);
    assert(dt);
    assert(lcpl_id != H5P_DEFAULT);
    assert(tcpl_id != H5P_DEFAULT);

    
    old_state = dt->shared->state;

    
    tcrt_info.dt      = dt;
    tcrt_info.tcpl_id = tcpl_id;

    
    ocrt_info.obj_type = H5O_TYPE_NAMED_DATATYPE;
    ocrt_info.crt_info = &tcrt_info;
    ocrt_info.new_obj  = NULL;

    
    if (H5L_link_object(loc, name, &ocrt_info, lcpl_id) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to create and link to named datatype");
    assert(ocrt_info.new_obj);

done:
    
    if (ret_value < 0 && (NULL != ocrt_info.new_obj)) {
        if (dt->shared->state == H5T_STATE_OPEN && dt->sh_loc.type == H5O_SHARE_TYPE_COMMITTED) {
            
            if (H5FO_top_decr(dt->sh_loc.file, dt->sh_loc.u.loc.oh_addr) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "can't decrement count for object");
            if (H5FO_delete(dt->sh_loc.file, dt->sh_loc.u.loc.oh_addr) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL,
                            "can't remove dataset from list of open objects");

            
            if (H5O_close(&(dt->oloc), NULL) < 0)
                HDONE_ERROR(H5E_DATATYPE, H5E_CLOSEERROR, FAIL, "unable to release object header");

            
            if (H5O_delete(dt->sh_loc.file, dt->sh_loc.u.loc.oh_addr) < 0)
                HDONE_ERROR(H5E_DATATYPE, H5E_CANTDELETE, FAIL, "unable to delete object header");

            
            if (H5T_set_loc(dt, NULL, H5T_LOC_MEMORY))
                HDONE_ERROR(H5E_DATATYPE, H5E_CANTDELETE, FAIL, "unable to return datatype to memory");
            dt->sh_loc.type   = H5O_SHARE_TYPE_UNSHARED;
            dt->shared->state = old_state;
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Tcommit_anon(hid_t loc_id, hid_t type_id, hid_t tcpl_id, hid_t tapl_id)
{
    void             *dt      = NULL; 
    H5VL_object_t    *new_obj = NULL; 
    H5T_t            *type    = NULL; 
    H5VL_object_t    *vol_obj = NULL; 
    H5VL_loc_params_t loc_params;
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (type = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (H5T_is_named(type))
        HGOTO_ERROR(H5E_ARGS, H5E_CANTSET, FAIL, "datatype is already committed");

    
    if (H5P_DEFAULT == tcpl_id)
        tcpl_id = H5P_DATATYPE_CREATE_DEFAULT;
    else if (true != H5P_isa_class(tcpl_id, H5P_DATATYPE_CREATE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not datatype creation property list");

    if (H5P_DEFAULT == tapl_id)
        tapl_id = H5P_DATATYPE_ACCESS_DEFAULT;
    else if (true != H5P_isa_class(tapl_id, H5P_DATATYPE_ACCESS))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not datatype access property list");

    
    if (H5CX_set_apl(&tapl_id, H5P_CLS_TACC, loc_id, true) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSET, FAIL, "can't set access property list info");

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid file identifier");

    
    if (NULL == (dt = H5VL_datatype_commit(vol_obj, &loc_params, NULL, type_id, H5P_LINK_CREATE_DEFAULT,
                                           tcpl_id, tapl_id, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to commit datatype");

    
    if (NULL == (new_obj = H5VL_create_object(dt, H5VL_OBJ_CONNECTOR(vol_obj))))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL, "can't create VOL object for committed datatype");

    
    type->vol_obj = new_obj;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5T__commit_anon(H5F_t *file, H5T_t *type, hid_t tcpl_id)
{
    H5O_loc_t *oloc;                
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(file);
    assert(type);
    assert(tcpl_id != H5P_DEFAULT);

    
    if (H5T__commit(file, type, tcpl_id) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to commit datatype");

    

    
    if (NULL == (oloc = H5T_oloc(type)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, FAIL, "unable to get object location of committed datatype");

    
    if (H5O_dec_rc_by_loc(oloc) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDEC, FAIL, "unable to decrement refcount on newly created object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T__commit(H5F_t *file, H5T_t *type, hid_t tcpl_id)
{
    H5O_t     *oh = NULL;            
    H5O_loc_t  temp_oloc;            
    H5G_name_t temp_path;            
    bool       loc_init     = false; 
    bool       ohdr_created = false; 
    size_t     dtype_size;           
    herr_t     ret_value = SUCCEED;  

    FUNC_ENTER_PACKAGE

    assert(file);
    assert(type);
    assert(tcpl_id != H5P_DEFAULT);

    
    if (0 == (H5F_INTENT(file) & H5F_ACC_RDWR))
        HGOTO_ERROR(H5E_DATATYPE, H5E_WRITEERROR, FAIL, "no write intent on file");

    
    if (H5T_STATE_NAMED == type->shared->state || H5T_STATE_OPEN == type->shared->state)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "datatype is already committed");
    if (H5T_STATE_IMMUTABLE == type->shared->state)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "datatype is immutable");

    
    if (H5T_is_sensible(type) <= 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "datatype is not sensible");

    
    if (H5T_set_loc(type, H5F_VOL_OBJ(file), H5T_LOC_DISK) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "cannot mark datatype on disk");

    
    if (H5O_loc_reset(&temp_oloc) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTRESET, FAIL, "unable to initialize location");
    if (H5G_name_reset(&temp_path) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTRESET, FAIL, "unable to initialize path");
    loc_init = true;

    
    if (H5T_set_version(file, type) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set version of datatype");

    
    dtype_size = H5O_msg_size_f(file, tcpl_id, H5O_DTYPE_ID, type, (size_t)0);
    assert(dtype_size);

    
    if (H5O_create(file, dtype_size, (size_t)1, tcpl_id, &temp_oloc) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to create datatype object header");
    ohdr_created = true;

    
    if (NULL == (oh = H5O_pin(&temp_oloc)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTPIN, FAIL, "unable to pin object header");

    
    if (!(H5O_has_chksum(oh) || (H5F_RFIC_FLAGS(file) & H5F_RFIC_UNUSUAL_NUM_UNUSED_NUMERIC_BITS)) &&
        H5T_is_numeric_with_unusual_unused_bits(type))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL,
                    "creating committed datatype with unusual datatype, see documentation for "
                    "H5Pset_relax_file_integrity_checks for details.");

    
    if (H5O_msg_append_oh(file, oh, H5O_DTYPE_ID, H5O_MSG_FLAG_CONSTANT | H5O_MSG_FLAG_DONTSHARE,
                          H5O_UPDATE_TIME, type) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to insert type header message");

    
    if (H5O_loc_copy_shallow(&(type->oloc), &temp_oloc) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to copy datatype location");
    if (H5G_name_copy(&(type->path), &temp_path, H5_COPY_SHALLOW) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to copy datatype location");
    loc_init = false;

    
    H5T_update_shared(type);
    type->shared->state    = H5T_STATE_OPEN;
    type->shared->fo_count = 1;

    
    if (H5FO_top_incr(type->sh_loc.file, type->sh_loc.u.loc.oh_addr) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINC, FAIL, "can't incr object ref. count");
    if (H5FO_insert(type->sh_loc.file, type->sh_loc.u.loc.oh_addr, type->shared, true) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINSERT, FAIL, "can't insert datatype into list of open objects");

    
    if (H5T_set_loc(type, NULL, H5T_LOC_MEMORY) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "cannot mark datatype in memory");

done:
    if (oh && H5O_unpin(oh) < 0)
        HDONE_ERROR(H5E_DATATYPE, H5E_CANTUNPIN, FAIL, "unable to unpin object header");

    if (ret_value < 0) {
        
        if (ohdr_created) {
            H5O_loc_t *oloc_ptr; 

            
            if (loc_init)
                oloc_ptr = &temp_oloc;
            else
                oloc_ptr = &(type->oloc);
            if (H5O_dec_rc_by_loc(oloc_ptr) < 0)
                HDONE_ERROR(H5E_DATATYPE, H5E_CANTDEC, FAIL,
                            "unable to decrement refcount on newly created object");
            if (H5O_close(oloc_ptr, NULL) < 0)
                HDONE_ERROR(H5E_DATATYPE, H5E_CLOSEERROR, FAIL, "unable to release object header");
            if (H5O_delete(file, oloc_ptr->addr) < 0)
                HDONE_ERROR(H5E_DATATYPE, H5E_CANTDELETE, FAIL, "unable to delete object header");
        }

        
        if (loc_init) {
            H5O_loc_free(&temp_oloc);
            H5G_name_free(&temp_path);
        }

        
        if ((type->shared->state == H5T_STATE_TRANSIENT || type->shared->state == H5T_STATE_RDONLY) &&
            (type->sh_loc.type == H5O_SHARE_TYPE_COMMITTED))
            type->sh_loc.type = H5O_SHARE_TYPE_UNSHARED;
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5Tcommitted(hid_t type_id)
{
    H5T_t *type;      
    htri_t ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (type = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");

    
    ret_value = H5T_is_named(type);

done:
    FUNC_LEAVE_API(ret_value)
} 

int
H5T_link(const H5T_t *type, int adjust)
{
    int ret_value = -1; 

    FUNC_ENTER_NOAPI((-1))

    assert(type);
    assert(type->sh_loc.type == H5O_SHARE_TYPE_COMMITTED);

    
    if ((ret_value = H5O_link(&type->oloc, adjust)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_LINKCOUNT, (-1), "unable to adjust named datatype link count");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static hid_t
H5T__open_api_common(hid_t loc_id, const char *name, hid_t tapl_id, void **token_ptr,
                     H5VL_object_t **_vol_obj_ptr)
{
    void           *dt          = NULL; 
    H5VL_object_t  *tmp_vol_obj = NULL; 
    H5VL_object_t **vol_obj_ptr =
        (_vol_obj_ptr ? _vol_obj_ptr : &tmp_vol_obj); 
    H5VL_loc_params_t loc_params;
    hid_t             ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_PACKAGE

    
    if (!name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "name parameter cannot be NULL");
    if (!*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "name parameter cannot be an empty string");

    
    if (H5VL_setup_acc_args(loc_id, H5P_CLS_TACC, false, &tapl_id, vol_obj_ptr, &loc_params) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSET, H5I_INVALID_HID, "can't set object access arguments");

    
    if (NULL == (dt = H5VL_datatype_open(*vol_obj_ptr, &loc_params, name, tapl_id, H5P_DATASET_XFER_DEFAULT,
                                         token_ptr)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPENOBJ, H5I_INVALID_HID, "unable to open named datatype");

    
    if ((ret_value = H5VL_register(H5I_DATATYPE, dt, H5VL_OBJ_CONNECTOR(*vol_obj_ptr), true)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register named datatype");

done:
    
    if (H5I_INVALID_HID == ret_value)
        if (dt && H5T_destruct_datatype(dt, H5VL_OBJ_CONNECTOR(*vol_obj_ptr)) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CLOSEERROR, H5I_INVALID_HID, "unable to release datatype");

    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5Topen2(hid_t loc_id, const char *name, hid_t tapl_id)
{
    hid_t ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if ((ret_value = H5T__open_api_common(loc_id, name, tapl_id, NULL, NULL)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPENOBJ, H5I_INVALID_HID,
                    "unable to open named datatype synchronously");
done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Topen_async(const char *app_file, const char *app_func, unsigned app_line, hid_t loc_id, const char *name,
              hid_t tapl_id, hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    hid_t          ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token; 

    
    if ((ret_value = H5T__open_api_common(loc_id, name, tapl_id, token_ptr, &vol_obj)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPENOBJ, H5I_INVALID_HID,
                    "unable to open named datatype asynchronously");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE7(__func__, "*s*sIui*sii", app_file, app_func, app_line, loc_id, name, tapl_id, es_id)) < 0) {
            
            if (H5I_dec_app_ref_always_close(ret_value) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDEC, H5I_INVALID_HID,
                            "can't decrement count on datatype ID");
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINSERT, H5I_INVALID_HID, "can't insert token into event set");
        } 

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Tget_create_plist(hid_t dtype_id)
{
    H5T_t *type      = NULL;            
    htri_t is_named  = FAIL;            
    hid_t  ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (type = (H5T_t *)H5I_object_verify(dtype_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not a datatype");

    
    if (FAIL == (is_named = H5T_is_named(type)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, H5I_INVALID_HID, "can't check whether datatype is committed");

    
    if (false == is_named) {
        H5P_genplist_t *tcpl_plist = NULL;

        
        if (NULL == (tcpl_plist = (H5P_genplist_t *)H5I_object(H5P_LST_DATATYPE_CREATE_ID_g)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "can't get default creation property list");
        if ((ret_value = H5P_copy_plist(tcpl_plist, true)) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, H5I_INVALID_HID,
                        "unable to copy the creation property list");
    } 
    
    else {
        H5VL_object_t           *vol_obj = type->vol_obj;
        H5VL_datatype_get_args_t vol_cb_args; 

        
        vol_cb_args.op_type               = H5VL_DATATYPE_GET_TCPL;
        vol_cb_args.args.get_tcpl.tcpl_id = H5I_INVALID_HID;

        
        if (H5VL_datatype_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, H5I_INVALID_HID, "can't get object creation info");

        
        ret_value = vol_cb_args.args.get_tcpl.tcpl_id;
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Tflush(hid_t type_id)
{
    H5T_t *dt;                  
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (!H5T_is_named(dt))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a committed datatype");

    
    if (dt->vol_obj) {
        H5VL_datatype_specific_args_t vol_cb_args; 

        
        if (H5CX_set_loc(type_id) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSET, FAIL, "can't set access property list info");

        
        vol_cb_args.op_type            = H5VL_DATATYPE_FLUSH;
        vol_cb_args.args.flush.type_id = type_id;

        if (H5VL_datatype_specific(dt->vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTFLUSH, FAIL, "unable to flush datatype");
    }

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Trefresh(hid_t type_id)
{
    H5T_t *dt;                  
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (!H5T_is_named(dt))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a committed datatype");

    
    if (dt->vol_obj) {
        H5VL_datatype_specific_args_t vol_cb_args; 

        
        if (H5CX_set_loc(type_id) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSET, FAIL, "can't set access property list info");

        
        vol_cb_args.op_type              = H5VL_DATATYPE_REFRESH;
        vol_cb_args.args.refresh.type_id = type_id;

        if (H5VL_datatype_specific(dt->vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTLOAD, FAIL, "unable to refresh datatype");
    }

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5T__get_create_plist(const H5T_t *type)
{
    H5P_genplist_t *tcpl_plist;                    
    H5P_genplist_t *new_plist;                     
    hid_t           new_tcpl_id = FAIL;            
    hid_t           ret_value   = H5I_INVALID_HID; 

    FUNC_ENTER_PACKAGE

    
    assert(type);

    
    if (NULL == (tcpl_plist = (H5P_genplist_t *)H5I_object(H5P_LST_DATATYPE_CREATE_ID_g)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, H5I_INVALID_HID, "can't get default creation property list");
    if ((new_tcpl_id = H5P_copy_plist(tcpl_plist, true)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, H5I_INVALID_HID, "unable to copy the creation property list");

    
    if (NULL == (new_plist = (H5P_genplist_t *)H5I_object(new_tcpl_id)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, H5I_INVALID_HID, "can't get property list");

    
    if (H5O_get_create_plist(&type->oloc, new_plist) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, H5I_INVALID_HID, "can't get object creation info");

    
    ret_value = new_tcpl_id;

done:
    if (ret_value < 0)
        if (new_tcpl_id > 0)
            if (H5I_dec_app_ref(new_tcpl_id) < 0)
                HDONE_ERROR(H5E_DATATYPE, H5E_CANTDEC, H5I_INVALID_HID, "unable to close temporary object");

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5T_t *
H5T__open_name(const H5G_loc_t *loc, const char *name)
{
    H5T_t     *dt = NULL;         
    H5G_name_t path;              
    H5O_loc_t  oloc;              
    H5G_loc_t  type_loc;          
    H5O_type_t obj_type;          
    bool       obj_found = false; 
    H5T_t     *ret_value = NULL;  

    FUNC_ENTER_PACKAGE

    
    assert(loc);
    assert(name);

    
    type_loc.oloc = &oloc;
    type_loc.path = &path;
    H5G_loc_reset(&type_loc);

    
    if (H5G_loc_find(loc, name, &type_loc ) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_NOTFOUND, NULL, "not found");
    obj_found = true;

    
    if (H5O_obj_type(&oloc, &obj_type) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, NULL, "can't get object type");
    if (obj_type != H5O_TYPE_NAMED_DATATYPE)
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, NULL, "not a named datatype");

    
    if (NULL == (dt = H5T_open(&type_loc)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPENOBJ, NULL, "unable to open named datatype");

    ret_value = dt;

done:
    
    if (NULL == ret_value)
        if (obj_found && H5_addr_defined(type_loc.oloc->addr))
            if (H5G_loc_free(&type_loc) < 0)
                HDONE_ERROR(H5E_DATATYPE, H5E_CANTRELEASE, NULL, "can't free location");

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5T_t *
H5T_open(const H5G_loc_t *loc)
{
    H5T_shared_t *shared_fo = NULL;
    H5T_t        *dt        = NULL;
    H5T_t        *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    assert(loc);

    
    if (NULL == (shared_fo = (H5T_shared_t *)H5FO_opened(loc->oloc->file, loc->oloc->addr))) {
        
        if (NULL == (dt = H5T__open_oid(loc)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_NOTFOUND, NULL, "not found");

        
        if (H5FO_insert(dt->sh_loc.file, dt->sh_loc.u.loc.oh_addr, dt->shared, false) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINSERT, NULL,
                        "can't insert datatype into list of open objects");

        
        if (H5FO_top_incr(dt->sh_loc.file, dt->sh_loc.u.loc.oh_addr) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINC, NULL, "can't increment object count");

        
        if (H5T_set_loc(dt, NULL, H5T_LOC_MEMORY) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "invalid datatype location");

        dt->shared->fo_count = 1;
    } 
    else {
        if (NULL == (dt = H5FL_MALLOC(H5T_t)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "can't allocate space for datatype");
        dt->vol_obj = NULL;

#if defined(H5_USING_MEMCHECKER) || !defined(NDEBUG)
        
        if (H5O_loc_reset(&(dt->oloc)) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPENOBJ, NULL, "unable to reset location");

        
        if (H5G_name_reset(&(dt->path)) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPENOBJ, NULL, "unable to reset path");
#endif 

        
        if (H5O_loc_copy_shallow(&dt->oloc, loc->oloc) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, NULL, "can't copy object location");

        
        if (H5G_name_copy(&(dt->path), loc->path, H5_COPY_SHALLOW) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, NULL, "can't copy path");

        
        H5T_update_shared(dt);

        
        dt->shared = shared_fo;

        
        if (H5T_set_loc(dt, NULL, H5T_LOC_MEMORY) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "invalid datatype location");

        
        shared_fo->fo_count++;

        
        if (H5FO_top_count(dt->sh_loc.file, dt->sh_loc.u.loc.oh_addr) == 0) {
            
            if (H5O_open(&(dt->oloc)) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPENOBJ, NULL, "unable to open object header");
        } 

        
        if (H5FO_top_incr(dt->sh_loc.file, dt->sh_loc.u.loc.oh_addr) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINC, NULL, "can't increment object count");
    } 

    ret_value = dt;

done:
    if (ret_value == NULL) {
        if (dt) {
            if (shared_fo == NULL) { 
                if (dt->shared->owned_vol_obj && H5VL_free_object(dt->shared->owned_vol_obj) < 0)
                    HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, NULL, "unable to close owned VOL object");
                dt->shared = H5FL_FREE(H5T_shared_t, dt->shared);
            } 

            H5O_loc_free(&(dt->oloc));
            H5G_name_free(&(dt->path));

            dt = H5FL_FREE(H5T_t, dt);
        } 

        if (shared_fo)
            shared_fo->fo_count--;
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5T_t *
H5T__open_oid(const H5G_loc_t *loc)
{
    H5T_t *dt        = NULL; 
    H5T_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE_TAG(loc->oloc->addr)

    assert(loc);

    
    if (H5O_open(loc->oloc) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPENOBJ, NULL, "unable to open named datatype");

    
    if (NULL == (dt = (H5T_t *)H5O_msg_read(loc->oloc, H5O_DTYPE_ID, NULL)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "unable to load type message from object header");

    
    dt->shared->state = H5T_STATE_OPEN;

    
    if (H5O_loc_copy_shallow(&dt->oloc, loc->oloc) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, NULL, "can't copy object location");

    
    if (H5G_name_copy(&(dt->path), loc->path, H5_COPY_SHALLOW) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, NULL, "can't copy path");

    
    H5T_update_shared(dt);

    
    ret_value = dt;

done:
    if (ret_value == NULL)
        if (dt == NULL)
            H5O_close(loc->oloc, NULL);

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5T_update_shared(H5T_t *dt)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(dt);

    
    H5O_UPDATE_SHARED(&(dt->sh_loc), H5O_SHARE_TYPE_COMMITTED, dt->oloc.file, H5O_DTYPE_ID, 0, dt->oloc.addr)

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

H5T_t *
H5T_construct_datatype(H5VL_object_t *vol_obj)
{
    H5T_t                   *dt = NULL;        
    H5VL_datatype_get_args_t vol_cb_args;      
    size_t                   nalloc    = 0;    
    void                    *buf       = NULL; 
    H5T_t                   *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    
    vol_cb_args.op_type                   = H5VL_DATATYPE_GET_BINARY_SIZE;
    vol_cb_args.args.get_binary_size.size = &nalloc;

    
    if (H5VL_datatype_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "unable to get datatype serialized size");

    
    if (NULL == (buf = (void *)H5MM_calloc(nalloc)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "can't allocate space for datatype");

    
    vol_cb_args.op_type                  = H5VL_DATATYPE_GET_BINARY;
    vol_cb_args.args.get_binary.buf      = buf;
    vol_cb_args.args.get_binary.buf_size = nalloc;

    
    if (H5VL_datatype_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "unable to get serialized datatype");

    
    if (NULL == (dt = H5T_decode(nalloc, buf)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "can't decode datatype");
    dt->vol_obj = vol_obj;

    
    ret_value = dt;

done:
    if (buf)
        buf = H5MM_xfree(buf);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T_destruct_datatype(void *datatype, H5VL_connector_t *vol_connector)
{
    H5VL_object_t *vol_obj   = NULL;
    herr_t         ret_value = FAIL;

    FUNC_ENTER_NOAPI(FAIL)

    if (NULL == (vol_obj = H5VL_create_object(datatype, vol_connector)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL, "can't create VOL object for committed datatype");

    if (H5VL_datatype_close(vol_obj, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CLOSEERROR, FAIL, "unable to release datatype");

done:
    if (vol_obj && H5VL_free_object(vol_obj) < 0)
        HDONE_ERROR(H5E_DATATYPE, H5E_CANTFREE, FAIL, "can't free VOL object");

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5VL_object_t *
H5T_get_named_type(const H5T_t *dt)
{
    H5VL_object_t *ret_value = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (NULL != dt->vol_obj)
        ret_value = dt->vol_obj;

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5T_t *
H5T_get_actual_type(H5T_t *dt)
{
    H5T_t *ret_value = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    if (NULL == dt->vol_obj)
        ret_value = dt;
    else
        ret_value = (H5T_t *)H5VL_object_data(dt->vol_obj);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T_save_refresh_state(hid_t tid, H5O_shared_t *cached_H5O_shared)
{
    H5T_t *dt        = NULL;    
    H5T_t *vol_dt    = NULL;    
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(cached_H5O_shared);

    if (NULL == (dt = (H5T_t *)H5I_object_verify(tid, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "tid is not a datatype ID");
    vol_dt = H5T_get_actual_type(dt);
    if (NULL == vol_dt)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "tid is not not a named datatype ID");

    
    vol_dt->shared->fo_count += 1;

    
    if (H5FO_top_incr(vol_dt->sh_loc.file, vol_dt->sh_loc.u.loc.oh_addr) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINC, FAIL, "can't increment object count");

    
    H5MM_memcpy(cached_H5O_shared, &(vol_dt->sh_loc), sizeof(H5O_shared_t));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T_restore_refresh_state(hid_t tid, H5O_shared_t *cached_H5O_shared)
{
    H5T_t *dt        = NULL;    
    H5T_t *vol_dt    = NULL;    
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(cached_H5O_shared);

    if (NULL == (dt = (H5T_t *)H5I_object_verify(tid, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "tid not a datatype ID");
    vol_dt = H5T_get_actual_type(dt);
    if (NULL == vol_dt)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "tid is not not a named datatype ID");

    
    H5MM_memcpy(&(vol_dt->sh_loc), cached_H5O_shared, sizeof(H5O_shared_t));

    
    if (H5FO_top_decr(vol_dt->sh_loc.file, vol_dt->sh_loc.u.loc.oh_addr) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDEC, FAIL, "can't decrement object count");

    
    vol_dt->shared->fo_count -= 1;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

bool
H5T_already_vol_managed(const H5T_t *dt)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(dt);

    FUNC_LEAVE_NOAPI(dt->vol_obj != NULL)
} 

herr_t
H5T_invoke_vol_optional(H5T_t *dt, H5VL_optional_args_t *args, hid_t dxpl_id, void **req,
                        H5VL_object_t **vol_obj_ptr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (!H5T_is_named(dt))
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL, "not a committed datatype");

    
    if (dt->vol_obj)
        if (H5VL_datatype_optional_op(dt->vol_obj, args, dxpl_id, req, vol_obj_ptr) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPERATE, FAIL, "unable to execute datatype optional callback");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
