/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Pmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Gprivate.h" 
#include "H5Oprivate.h" 
#include "H5Ppkg.h"     

#define H5G_CRT_GROUP_INFO_ENC H5P__gcrt_group_info_enc
#define H5G_CRT_GROUP_INFO_DEC H5P__gcrt_group_info_dec
#define H5G_CRT_LINK_INFO_ENC  H5P__gcrt_link_info_enc
#define H5G_CRT_LINK_INFO_DEC  H5P__gcrt_link_info_dec

static herr_t H5P__gcrt_reg_prop(H5P_genclass_t *pclass);

static herr_t H5P__gcrt_group_info_enc(const void *value, void **_pp, size_t *size);
static herr_t H5P__gcrt_group_info_dec(const void **_pp, void *value);
static herr_t H5P__gcrt_link_info_enc(const void *value, void **_pp, size_t *size);
static herr_t H5P__gcrt_link_info_dec(const void **_pp, void *value);

const H5P_libclass_t H5P_CLS_GCRT[1] = {{
    "group create",        
    H5P_TYPE_GROUP_CREATE, 

    &H5P_CLS_OBJECT_CREATE_g,   
    &H5P_CLS_GROUP_CREATE_g,    
    &H5P_CLS_GROUP_CREATE_ID_g, 
    &H5P_LST_GROUP_CREATE_ID_g, 
    H5P__gcrt_reg_prop,         

    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL  
}};

static const H5O_ginfo_t H5G_def_ginfo_g = H5G_CRT_GROUP_INFO_DEF; 
static const H5O_linfo_t H5G_def_linfo_g = H5G_CRT_LINK_INFO_DEF;  

static herr_t
H5P__gcrt_reg_prop(H5P_genclass_t *pclass)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5P__register_real(pclass, H5G_CRT_GROUP_INFO_NAME, H5G_CRT_GROUP_INFO_SIZE, &H5G_def_ginfo_g, NULL,
                           NULL, NULL, H5G_CRT_GROUP_INFO_ENC, H5G_CRT_GROUP_INFO_DEC, NULL, NULL, NULL,
                           NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5G_CRT_LINK_INFO_NAME, H5G_CRT_LINK_INFO_SIZE, &H5G_def_linfo_g, NULL,
                           NULL, NULL, H5G_CRT_LINK_INFO_ENC, H5G_CRT_LINK_INFO_DEC, NULL, NULL, NULL,
                           NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Pset_local_heap_size_hint(hid_t plist_id, size_t size_hint)
{
    H5P_genplist_t *plist;               
    H5O_ginfo_t     ginfo;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_GROUP_CREATE, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_get(plist, H5G_CRT_GROUP_INFO_NAME, &ginfo) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get group info");

    
    H5_CHECKED_ASSIGN(ginfo.lheap_size_hint, uint32_t, size_hint, size_t);

    
    if (H5P_set(plist, H5G_CRT_GROUP_INFO_NAME, &ginfo) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set group info");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_local_heap_size_hint(hid_t plist_id, size_t *size_hint )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    if (size_hint) {
        H5P_genplist_t *plist; 
        H5O_ginfo_t     ginfo; 

        
        if (NULL == (plist = H5P_object_verify(plist_id, H5P_GROUP_CREATE, true)))
            HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

        
        if (H5P_get(plist, H5G_CRT_GROUP_INFO_NAME, &ginfo) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get group info");

        
        *size_hint = ginfo.lheap_size_hint;
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_link_phase_change(hid_t plist_id, unsigned max_compact, unsigned min_dense)
{
    H5P_genplist_t *plist;               
    H5O_ginfo_t     ginfo;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (max_compact < min_dense)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "max compact value must be >= min dense value");
    if (max_compact > 65535)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "max compact value must be < 65536");
    if (min_dense > 65535)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "min dense value must be < 65536");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_GROUP_CREATE, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_get(plist, H5G_CRT_GROUP_INFO_NAME, &ginfo) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get group info");

    
    if (max_compact != H5G_CRT_GINFO_MAX_COMPACT || min_dense != H5G_CRT_GINFO_MIN_DENSE)
        ginfo.store_link_phase_change = true;
    else
        ginfo.store_link_phase_change = false;
    ginfo.max_compact = (uint16_t)max_compact;
    ginfo.min_dense   = (uint16_t)min_dense;

    
    if (H5P_set(plist, H5G_CRT_GROUP_INFO_NAME, &ginfo) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set group info");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_link_phase_change(hid_t plist_id, unsigned *max_compact , unsigned *min_dense )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (max_compact || min_dense) {
        H5P_genplist_t *plist; 
        H5O_ginfo_t     ginfo; 

        
        if (NULL == (plist = H5P_object_verify(plist_id, H5P_GROUP_CREATE, true)))
            HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

        
        if (H5P_get(plist, H5G_CRT_GROUP_INFO_NAME, &ginfo) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get group info");

        if (max_compact)
            *max_compact = ginfo.max_compact;
        if (min_dense)
            *min_dense = ginfo.min_dense;
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_est_link_info(hid_t plist_id, unsigned est_num_entries, unsigned est_name_len)
{
    H5P_genplist_t *plist;               
    H5O_ginfo_t     ginfo;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (est_num_entries > 65535)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "est. number of entries must be < 65536");
    if (est_name_len > 65535)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "est. name length must be < 65536");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_GROUP_CREATE, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_get(plist, H5G_CRT_GROUP_INFO_NAME, &ginfo) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get group info");

    
    if (est_num_entries != H5G_CRT_GINFO_EST_NUM_ENTRIES || est_name_len != H5G_CRT_GINFO_EST_NAME_LEN)
        ginfo.store_est_entry_info = true;
    else
        ginfo.store_est_entry_info = false;
    ginfo.est_num_entries = (uint16_t)est_num_entries;
    ginfo.est_name_len    = (uint16_t)est_name_len;

    
    if (H5P_set(plist, H5G_CRT_GROUP_INFO_NAME, &ginfo) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set group info");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_est_link_info(hid_t plist_id, unsigned *est_num_entries , unsigned *est_name_len )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (est_num_entries || est_name_len) {
        H5P_genplist_t *plist; 
        H5O_ginfo_t     ginfo; 

        
        if (NULL == (plist = H5P_object_verify(plist_id, H5P_GROUP_CREATE, true)))
            HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

        
        if (H5P_get(plist, H5G_CRT_GROUP_INFO_NAME, &ginfo) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get group info");

        if (est_num_entries)
            *est_num_entries = ginfo.est_num_entries;
        if (est_name_len)
            *est_name_len = ginfo.est_name_len;
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_link_creation_order(hid_t plist_id, unsigned crt_order_flags)
{
    H5P_genplist_t *plist;               
    H5O_linfo_t     linfo;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!(crt_order_flags & H5P_CRT_ORDER_TRACKED) && (crt_order_flags & H5P_CRT_ORDER_INDEXED))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "tracking creation order is required for index");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_GROUP_CREATE, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_get(plist, H5G_CRT_LINK_INFO_NAME, &linfo) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get link info");

    
    linfo.track_corder = (bool)((crt_order_flags & H5P_CRT_ORDER_TRACKED) ? true : false);
    linfo.index_corder = (bool)((crt_order_flags & H5P_CRT_ORDER_INDEXED) ? true : false);

    
    if (H5P_set(plist, H5G_CRT_LINK_INFO_NAME, &linfo) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set link info");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_link_creation_order(hid_t plist_id, unsigned *crt_order_flags )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (crt_order_flags) {
        H5P_genplist_t *plist; 
        H5O_linfo_t     linfo; 

        
        *crt_order_flags = 0;

        
        if (NULL == (plist = H5P_object_verify(plist_id, H5P_GROUP_CREATE, true)))
            HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

        
        if (H5P_get(plist, H5G_CRT_LINK_INFO_NAME, &linfo) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get link info");

        *crt_order_flags |= linfo.track_corder ? H5P_CRT_ORDER_TRACKED : 0;
        *crt_order_flags |= linfo.index_corder ? H5P_CRT_ORDER_INDEXED : 0;
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5P__gcrt_group_info_enc(const void *value, void **_pp, size_t *size)
{
    const H5O_ginfo_t *ginfo = (const H5O_ginfo_t *)value; 
    uint8_t          **pp    = (uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    if (NULL != *pp) {
        UINT32ENCODE(*pp, ginfo->lheap_size_hint);
        UINT16ENCODE(*pp, ginfo->max_compact);
        UINT16ENCODE(*pp, ginfo->min_dense);
        UINT16ENCODE(*pp, ginfo->est_num_entries);
        UINT16ENCODE(*pp, ginfo->est_name_len);
    } 

    *size += sizeof(uint16_t) * 4 + sizeof(uint32_t);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__gcrt_group_info_dec(const void **_pp, void *_value)
{
    H5O_ginfo_t    *ginfo     = (H5O_ginfo_t *)_value; 
    const uint8_t **pp        = (const uint8_t **)_pp;
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_NOERR

    
    memset(ginfo, 0, sizeof(H5O_ginfo_t));
    *ginfo = H5G_def_ginfo_g;

    UINT32DECODE(*pp, ginfo->lheap_size_hint);
    UINT16DECODE(*pp, ginfo->max_compact);
    UINT16DECODE(*pp, ginfo->min_dense);
    UINT16DECODE(*pp, ginfo->est_num_entries);
    UINT16DECODE(*pp, ginfo->est_name_len);

    
    if (ginfo->max_compact != H5G_CRT_GINFO_MAX_COMPACT || ginfo->min_dense != H5G_CRT_GINFO_MIN_DENSE)
        ginfo->store_link_phase_change = true;
    else
        ginfo->store_link_phase_change = false;

    if (ginfo->est_num_entries != H5G_CRT_GINFO_EST_NUM_ENTRIES ||
        ginfo->est_name_len != H5G_CRT_GINFO_EST_NAME_LEN)
        ginfo->store_est_entry_info = true;
    else
        ginfo->store_est_entry_info = false;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__gcrt_link_info_enc(const void *value, void **_pp, size_t *size)
{
    const H5O_linfo_t *linfo = (const H5O_linfo_t *)value; 
    uint8_t          **pp    = (uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    if (NULL != *pp) {
        unsigned crt_order_flags = 0;

        crt_order_flags |= linfo->track_corder ? H5P_CRT_ORDER_TRACKED : 0;
        crt_order_flags |= linfo->index_corder ? H5P_CRT_ORDER_INDEXED : 0;

        
        *(*pp)++ = (uint8_t)sizeof(unsigned);

        
        H5_ENCODE_UNSIGNED(*pp, crt_order_flags);
    } 

    *size += (1 + sizeof(unsigned));

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__gcrt_link_info_dec(const void **_pp, void *_value)
{
    H5O_linfo_t    *linfo = (H5O_linfo_t *)_value; 
    const uint8_t **pp    = (const uint8_t **)_pp;
    unsigned        crt_order_flags;
    unsigned        enc_size;
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    enc_size = *(*pp)++;
    if (enc_size != sizeof(unsigned))
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "unsigned value can't be decoded");

    
    memset(linfo, 0, sizeof(H5O_linfo_t));
    *linfo = H5G_def_linfo_g;

    H5_DECODE_UNSIGNED(*pp, crt_order_flags);

    
    linfo->track_corder = (bool)((crt_order_flags & H5P_CRT_ORDER_TRACKED) ? true : false);
    linfo->index_corder = (bool)((crt_order_flags & H5P_CRT_ORDER_INDEXED) ? true : false);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
