/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Cmodule.h" 

#include "H5private.h"   
#include "H5Cpkg.h"      
#include "H5Clog.h"      
#include "H5Eprivate.h"  
#include "H5MMprivate.h" 

#define H5C_MAX_TRACE_LOG_MSG_SIZE 4096

typedef struct H5C_log_trace_udata_t {
    FILE *outfile;
    char *message;
} H5C_log_trace_udata_t;

static herr_t H5C__trace_write_log_message(H5C_log_trace_udata_t *trace_udata);

static herr_t H5C__trace_tear_down_logging(H5C_log_info_t *log_info);
static herr_t H5C__trace_write_expunge_entry_log_msg(void *udata, haddr_t address, int type_id,
                                                     herr_t fxn_ret_value);
static herr_t H5C__trace_write_flush_cache_log_msg(void *udata, herr_t fxn_ret_value);
static herr_t H5C__trace_write_insert_entry_log_msg(void *udata, haddr_t address, int type_id, unsigned flags,
                                                    size_t size, herr_t fxn_ret_value);
static herr_t H5C__trace_write_mark_entry_dirty_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                                        herr_t fxn_ret_value);
static herr_t H5C__trace_write_mark_entry_clean_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                                        herr_t fxn_ret_value);
static herr_t H5C__trace_write_mark_unserialized_entry_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                                               herr_t fxn_ret_value);
static herr_t H5C__trace_write_mark_serialized_entry_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                                             herr_t fxn_ret_value);
static herr_t H5C__trace_write_move_entry_log_msg(void *udata, haddr_t old_addr, haddr_t new_addr,
                                                  int type_id, herr_t fxn_ret_value);
static herr_t H5C__trace_write_pin_entry_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                                 herr_t fxn_ret_value);
static herr_t H5C__trace_write_create_fd_log_msg(void *udata, const H5C_cache_entry_t *parent,
                                                 const H5C_cache_entry_t *child, herr_t fxn_ret_value);
static herr_t H5C__trace_write_protect_entry_log_msg(void *udata, const H5C_cache_entry_t *entry, int type_id,
                                                     unsigned flags, herr_t fxn_ret_value);
static herr_t H5C__trace_write_resize_entry_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                                    size_t new_size, herr_t fxn_ret_value);
static herr_t H5C__trace_write_unpin_entry_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                                   herr_t fxn_ret_value);
static herr_t H5C__trace_write_destroy_fd_log_msg(void *udata, const H5C_cache_entry_t *parent,
                                                  const H5C_cache_entry_t *child, herr_t fxn_ret_value);
static herr_t H5C__trace_write_unprotect_entry_log_msg(void *udata, haddr_t address, int type_id,
                                                       unsigned flags, herr_t fxn_ret_value);
static herr_t H5C__trace_write_set_cache_config_log_msg(void *udata, const H5AC_cache_config_t *config,
                                                        herr_t fxn_ret_value);
static herr_t H5C__trace_write_remove_entry_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                                    herr_t fxn_ret_value);

static const H5C_log_class_t H5C_trace_log_class_g = {"trace",
                                                      H5C__trace_tear_down_logging,
                                                      NULL, 
                                                      NULL, 
                                                      NULL, 
                                                      NULL, 
                                                      NULL, 
                                                      NULL, 
                                                      NULL, 
                                                      H5C__trace_write_expunge_entry_log_msg,
                                                      H5C__trace_write_flush_cache_log_msg,
                                                      H5C__trace_write_insert_entry_log_msg,
                                                      H5C__trace_write_mark_entry_dirty_log_msg,
                                                      H5C__trace_write_mark_entry_clean_log_msg,
                                                      H5C__trace_write_mark_unserialized_entry_log_msg,
                                                      H5C__trace_write_mark_serialized_entry_log_msg,
                                                      H5C__trace_write_move_entry_log_msg,
                                                      H5C__trace_write_pin_entry_log_msg,
                                                      H5C__trace_write_create_fd_log_msg,
                                                      H5C__trace_write_protect_entry_log_msg,
                                                      H5C__trace_write_resize_entry_log_msg,
                                                      H5C__trace_write_unpin_entry_log_msg,
                                                      H5C__trace_write_destroy_fd_log_msg,
                                                      H5C__trace_write_unprotect_entry_log_msg,
                                                      H5C__trace_write_set_cache_config_log_msg,
                                                      H5C__trace_write_remove_entry_log_msg};

static herr_t
H5C__trace_write_log_message(H5C_log_trace_udata_t *trace_udata)
{
    size_t n_chars;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(trace_udata);
    assert(trace_udata->outfile);
    assert(trace_udata->message);

    
    n_chars = strlen(trace_udata->message);
    if ((int)n_chars != Rfprintf(trace_udata->outfile, "%s", trace_udata->message))
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "error writing log message");
    memset((void *)(trace_udata->message), 0, (size_t)(n_chars * sizeof(char)));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C__log_trace_set_up(H5C_log_info_t *log_info, const char log_location[], int mpi_rank)
{
    H5C_log_trace_udata_t *trace_udata = NULL;
    char                  *file_name   = NULL;
    size_t                 n_chars;
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(log_info);
    assert(log_location);

    
    log_info->cls = &H5C_trace_log_class_g;

    
    if (NULL == (log_info->udata = H5MM_calloc(sizeof(H5C_log_trace_udata_t))))
        HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, FAIL, "memory allocation failed");
    trace_udata = (H5C_log_trace_udata_t *)(log_info->udata);

    
    if (NULL == (trace_udata->message = (char *)H5MM_calloc(H5C_MAX_TRACE_LOG_MSG_SIZE * sizeof(char))))
        HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, FAIL, "memory allocation failed");

    
    n_chars = strlen(log_location) + 1 + 39 + 1;
    if (NULL == (file_name = (char *)H5MM_calloc(n_chars * sizeof(char))))
        HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, FAIL,
                    "can't allocate memory for mdc log file name manipulation");

    
    if (-1 == mpi_rank)
        snprintf(file_name, n_chars, "%s", log_location);
    else
        snprintf(file_name, n_chars, "%s.%d", log_location, mpi_rank);

    
    if (NULL == (trace_udata->outfile = fopen(file_name, "w")))
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "can't create mdc log file");
    setbuf(trace_udata->outfile, NULL);

    
    Rfprintf(trace_udata->outfile, "### HDF5 metadata cache trace file version 1 ###\n");

done:
    if (file_name)
        H5MM_xfree(file_name);

    
    if (FAIL == ret_value) {
        
        if (trace_udata && trace_udata->message)
            H5MM_xfree(trace_udata->message);
        if (trace_udata)
            H5MM_xfree(trace_udata);

        
        log_info->udata = NULL;
        log_info->cls   = NULL;
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__trace_tear_down_logging(H5C_log_info_t *log_info)
{
    H5C_log_trace_udata_t *trace_udata = NULL;
    herr_t                 ret_value   = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(log_info);

    
    trace_udata = (H5C_log_trace_udata_t *)(log_info->udata);

    
    H5MM_xfree(trace_udata->message);

    
    if (EOF == fclose(trace_udata->outfile))
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "problem closing mdc log file");
    trace_udata->outfile = NULL;

    
    H5MM_xfree(trace_udata);

    
    log_info->cls   = NULL;
    log_info->udata = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__trace_write_expunge_entry_log_msg(void *udata, haddr_t address, int type_id, herr_t fxn_ret_value)
{
    H5C_log_trace_udata_t *trace_udata = (H5C_log_trace_udata_t *)(udata);
    herr_t                 ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(trace_udata);
    assert(trace_udata->message);

    
    snprintf(trace_udata->message, H5C_MAX_TRACE_LOG_MSG_SIZE, "H5AC_expunge_entry 0x%lx %d %d\n",
             (unsigned long)address, type_id, (int)fxn_ret_value);

    
    if (H5C__trace_write_log_message(trace_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__trace_write_flush_cache_log_msg(void *udata, herr_t fxn_ret_value)
{
    H5C_log_trace_udata_t *trace_udata = (H5C_log_trace_udata_t *)(udata);
    herr_t                 ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(trace_udata);
    assert(trace_udata->message);

    
    snprintf(trace_udata->message, H5C_MAX_TRACE_LOG_MSG_SIZE, "H5AC_flush %d\n", (int)fxn_ret_value);

    
    if (H5C__trace_write_log_message(trace_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__trace_write_insert_entry_log_msg(void *udata, haddr_t address, int type_id, unsigned flags, size_t size,
                                      herr_t fxn_ret_value)
{
    H5C_log_trace_udata_t *trace_udata = (H5C_log_trace_udata_t *)(udata);
    herr_t                 ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(trace_udata);
    assert(trace_udata->message);

    
    snprintf(trace_udata->message, H5C_MAX_TRACE_LOG_MSG_SIZE, "H5AC_insert_entry 0x%lx %d 0x%x %d %d\n",
             (unsigned long)address, type_id, flags, (int)size, (int)fxn_ret_value);

    
    if (H5C__trace_write_log_message(trace_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__trace_write_mark_entry_dirty_log_msg(void *udata, const H5C_cache_entry_t *entry, herr_t fxn_ret_value)
{
    H5C_log_trace_udata_t *trace_udata = (H5C_log_trace_udata_t *)(udata);
    herr_t                 ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(trace_udata);
    assert(trace_udata->message);
    assert(entry);

    
    snprintf(trace_udata->message, H5C_MAX_TRACE_LOG_MSG_SIZE, "H5AC_mark_entry_dirty 0x%lx %d\n",
             (unsigned long)(entry->addr), (int)fxn_ret_value);

    
    if (H5C__trace_write_log_message(trace_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__trace_write_mark_entry_clean_log_msg(void *udata, const H5C_cache_entry_t *entry, herr_t fxn_ret_value)
{
    H5C_log_trace_udata_t *trace_udata = (H5C_log_trace_udata_t *)(udata);
    herr_t                 ret_value   = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(trace_udata);
    assert(trace_udata->message);
    assert(entry);

    
    snprintf(trace_udata->message, H5C_MAX_TRACE_LOG_MSG_SIZE, "H5AC_mark_entry_clean 0x%lx %d\n",
             (unsigned long)(entry->addr), (int)fxn_ret_value);

    
    if (H5C__trace_write_log_message(trace_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__trace_write_mark_unserialized_entry_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                                 herr_t fxn_ret_value)
{
    H5C_log_trace_udata_t *trace_udata = (H5C_log_trace_udata_t *)(udata);
    herr_t                 ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(trace_udata);
    assert(trace_udata->message);
    assert(entry);

    
    snprintf(trace_udata->message, H5C_MAX_TRACE_LOG_MSG_SIZE, "H5AC_mark_entry_unserialized 0x%lx %d\n",
             (unsigned long)(entry->addr), (int)fxn_ret_value);

    
    if (H5C__trace_write_log_message(trace_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__trace_write_mark_serialized_entry_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                               herr_t fxn_ret_value)
{
    H5C_log_trace_udata_t *trace_udata = (H5C_log_trace_udata_t *)(udata);
    herr_t                 ret_value   = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(trace_udata);
    assert(trace_udata->message);
    assert(entry);

    
    snprintf(trace_udata->message, H5C_MAX_TRACE_LOG_MSG_SIZE, "H5AC_mark_entry_serialized 0x%lx %d\n",
             (unsigned long)(entry->addr), (int)fxn_ret_value);

    
    if (H5C__trace_write_log_message(trace_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__trace_write_move_entry_log_msg(void *udata, haddr_t old_addr, haddr_t new_addr, int type_id,
                                    herr_t fxn_ret_value)
{
    H5C_log_trace_udata_t *trace_udata = (H5C_log_trace_udata_t *)(udata);
    herr_t                 ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(trace_udata);
    assert(trace_udata->message);

    
    snprintf(trace_udata->message, H5C_MAX_TRACE_LOG_MSG_SIZE, "H5AC_move_entry 0x%lx 0x%lx %d %d\n",
             (unsigned long)old_addr, (unsigned long)new_addr, type_id, (int)fxn_ret_value);

    
    if (H5C__trace_write_log_message(trace_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__trace_write_pin_entry_log_msg(void *udata, const H5C_cache_entry_t *entry, herr_t fxn_ret_value)
{
    H5C_log_trace_udata_t *trace_udata = (H5C_log_trace_udata_t *)(udata);
    herr_t                 ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(trace_udata);
    assert(trace_udata->message);
    assert(entry);

    
    snprintf(trace_udata->message, H5C_MAX_TRACE_LOG_MSG_SIZE, "H5AC_pin_protected_entry 0x%lx %d\n",
             (unsigned long)(entry->addr), (int)fxn_ret_value);

    
    if (H5C__trace_write_log_message(trace_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__trace_write_create_fd_log_msg(void *udata, const H5C_cache_entry_t *parent,
                                   const H5C_cache_entry_t *child, herr_t fxn_ret_value)
{
    H5C_log_trace_udata_t *trace_udata = (H5C_log_trace_udata_t *)(udata);
    herr_t                 ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(trace_udata);
    assert(trace_udata->message);
    assert(parent);
    assert(child);

    
    snprintf(trace_udata->message, H5C_MAX_TRACE_LOG_MSG_SIZE,
             "H5AC_create_flush_dependency 0x%lx 0x%lx %d\n", (unsigned long)(parent->addr),
             (unsigned long)(child->addr), (int)fxn_ret_value);

    
    if (H5C__trace_write_log_message(trace_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__trace_write_protect_entry_log_msg(void *udata, const H5C_cache_entry_t *entry, int type_id,
                                       unsigned flags, herr_t fxn_ret_value)
{
    H5C_log_trace_udata_t *trace_udata = (H5C_log_trace_udata_t *)(udata);
    herr_t                 ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(trace_udata);
    assert(trace_udata->message);
    assert(entry);

    
    snprintf(trace_udata->message, H5C_MAX_TRACE_LOG_MSG_SIZE, "H5AC_protect 0x%lx %d 0x%x %d %d\n",
             (unsigned long)(entry->addr), type_id, flags, (int)(entry->size), (int)fxn_ret_value);

    
    if (H5C__trace_write_log_message(trace_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__trace_write_resize_entry_log_msg(void *udata, const H5C_cache_entry_t *entry, size_t new_size,
                                      herr_t fxn_ret_value)
{
    H5C_log_trace_udata_t *trace_udata = (H5C_log_trace_udata_t *)(udata);
    herr_t                 ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(trace_udata);
    assert(trace_udata->message);
    assert(entry);

    
    snprintf(trace_udata->message, H5C_MAX_TRACE_LOG_MSG_SIZE, "H5AC_resize_entry 0x%lx %d %d\n",
             (unsigned long)(entry->addr), (int)new_size, (int)fxn_ret_value);

    
    if (H5C__trace_write_log_message(trace_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__trace_write_unpin_entry_log_msg(void *udata, const H5C_cache_entry_t *entry, herr_t fxn_ret_value)
{
    H5C_log_trace_udata_t *trace_udata = (H5C_log_trace_udata_t *)(udata);
    herr_t                 ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(trace_udata);
    assert(trace_udata->message);
    assert(entry);

    
    snprintf(trace_udata->message, H5C_MAX_TRACE_LOG_MSG_SIZE, "H5AC_unpin_entry 0x%lx %d\n",
             (unsigned long)(entry->addr), (int)fxn_ret_value);

    
    if (H5C__trace_write_log_message(trace_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__trace_write_destroy_fd_log_msg(void *udata, const H5C_cache_entry_t *parent,
                                    const H5C_cache_entry_t *child, herr_t fxn_ret_value)
{
    H5C_log_trace_udata_t *trace_udata = (H5C_log_trace_udata_t *)(udata);
    herr_t                 ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(trace_udata);
    assert(trace_udata->message);
    assert(parent);
    assert(child);

    
    snprintf(trace_udata->message, H5C_MAX_TRACE_LOG_MSG_SIZE,
             "H5AC_destroy_flush_dependency 0x%lx 0x%lx %d\n", (unsigned long)(parent->addr),
             (unsigned long)(child->addr), (int)fxn_ret_value);

    
    if (H5C__trace_write_log_message(trace_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__trace_write_unprotect_entry_log_msg(void *udata, haddr_t address, int type_id, unsigned flags,
                                         herr_t fxn_ret_value)
{
    H5C_log_trace_udata_t *trace_udata = (H5C_log_trace_udata_t *)(udata);
    herr_t                 ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(trace_udata);
    assert(trace_udata->message);

    
    snprintf(trace_udata->message, H5C_MAX_TRACE_LOG_MSG_SIZE, "H5AC_unprotect 0x%lx %d 0x%x %d\n",
             (unsigned long)(address), type_id, flags, (int)fxn_ret_value);

    
    if (H5C__trace_write_log_message(trace_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__trace_write_set_cache_config_log_msg(void *udata, const H5AC_cache_config_t *config,
                                          herr_t fxn_ret_value)
{
    H5C_log_trace_udata_t *trace_udata = (H5C_log_trace_udata_t *)(udata);
    herr_t                 ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(trace_udata);
    assert(trace_udata->message);
    assert(config);

    
    snprintf(trace_udata->message, H5C_MAX_TRACE_LOG_MSG_SIZE,
             "H5AC_set_cache_auto_resize_config %d %d %d %d \"%s\" %d %d %d %f %d %d %ld %d %f %f %d %f %f "
             "%d %d %d %f %f %d %d %d %d %f %llu %d %d\n",
             config->version, (int)(config->rpt_fcn_enabled), (int)(config->open_trace_file),
             (int)(config->close_trace_file), config->trace_file_name, (int)(config->evictions_enabled),
             (int)(config->set_initial_size), (int)(config->initial_size), config->min_clean_fraction,
             (int)(config->max_size), (int)(config->min_size), config->epoch_length, (int)(config->incr_mode),
             config->lower_hr_threshold, config->increment, (int)(config->flash_incr_mode),
             config->flash_multiple, config->flash_threshold, (int)(config->apply_max_increment),
             (int)(config->max_increment), (int)(config->decr_mode), config->upper_hr_threshold,
             config->decrement, (int)(config->apply_max_decrement), (int)(config->max_decrement),
             config->epochs_before_eviction, (int)(config->apply_empty_reserve), config->empty_reserve,
             (unsigned long long)config->dirty_bytes_threshold, config->metadata_write_strategy, (int)fxn_ret_value);

    
    if (H5C__trace_write_log_message(trace_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__trace_write_remove_entry_log_msg(void *udata, const H5C_cache_entry_t *entry, herr_t fxn_ret_value)
{
    H5C_log_trace_udata_t *trace_udata = (H5C_log_trace_udata_t *)(udata);
    herr_t                 ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(trace_udata);
    assert(trace_udata->message);
    assert(entry);

    
    snprintf(trace_udata->message, H5C_MAX_TRACE_LOG_MSG_SIZE, "H5AC_remove_entry 0x%lx %d\n",
             (unsigned long)(entry->addr), (int)fxn_ret_value);

    
    if (H5C__trace_write_log_message(trace_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
