/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Fmodule.h" 

#include "H5private.h"   
#include "H5ACprivate.h" 
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5ESprivate.h" 
#include "H5Fpkg.h"      
#include "H5FLprivate.h" 
#include "H5Iprivate.h"  
#include "H5Pprivate.h"  
#include "H5VLprivate.h" 

#include "H5VLnative_private.h" 

typedef struct {
    size_t   obj_count; 
    unsigned types;     
} H5F_trav_obj_cnt_t;

typedef struct {
    size_t max_objs;  
    hid_t *oid_list;  
    size_t obj_count; 
} H5F_trav_obj_ids_t;

static int H5F__get_all_count_cb(void H5_ATTR_UNUSED *obj_ptr, hid_t H5_ATTR_UNUSED obj_id, void *key);

static int H5F__get_all_ids_cb(void H5_ATTR_UNUSED *obj_ptr, hid_t obj_id, void *key);

static herr_t H5F__post_open_api_common(H5VL_object_t *vol_obj, void **token_ptr);
static hid_t  H5F__create_api_common(const char *filename, unsigned flags, hid_t fcpl_id, hid_t fapl_id,
                                     void **token_ptr);
static hid_t  H5F__open_api_common(const char *filename, unsigned flags, hid_t fapl_id, void **token_ptr);
static hid_t  H5F__reopen_api_common(hid_t file_id, void **token_ptr);
static herr_t H5F__flush_api_common(hid_t object_id, H5F_scope_t scope, void **token_ptr,
                                    H5VL_object_t **_vol_obj_ptr);

hid_t
H5Fget_create_plist(hid_t file_id)
{
    H5VL_object_t       *vol_obj;                     
    H5VL_file_get_args_t vol_cb_args;                 
    hid_t                ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid file identifier");

    
    vol_cb_args.op_type               = H5VL_FILE_GET_FCPL;
    vol_cb_args.args.get_fcpl.fcpl_id = H5I_INVALID_HID;

    
    if (H5VL_file_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTCOPY, H5I_INVALID_HID, "unable to retrieve file creation properties");

    
    ret_value = vol_cb_args.args.get_fcpl.fcpl_id;

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Fget_access_plist(hid_t file_id)
{
    H5VL_object_t       *vol_obj;                     
    H5VL_file_get_args_t vol_cb_args;                 
    hid_t                ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid file identifier");

    
    vol_cb_args.op_type               = H5VL_FILE_GET_FAPL;
    vol_cb_args.args.get_fapl.fapl_id = H5I_INVALID_HID;

    
    if (H5VL_file_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, H5I_INVALID_HID, "can't get file access property list");

    
    ret_value = vol_cb_args.args.get_fapl.fapl_id;

done:
    FUNC_LEAVE_API(ret_value)
} 

static int
H5F__get_all_count_cb(void H5_ATTR_UNUSED *obj_ptr, hid_t H5_ATTR_UNUSED obj_id, void *key)
{
    H5F_trav_obj_cnt_t *udata     = (H5F_trav_obj_cnt_t *)key;
    int                 ret_value = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE_NOERR

    udata->obj_count++;

    FUNC_LEAVE_NOAPI(ret_value)
} 

ssize_t
H5Fget_obj_count(hid_t file_id, unsigned types)
{
    ssize_t ret_value = 0; 

    FUNC_ENTER_API((-1))

    
    if (0 == (types & H5F_OBJ_ALL))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, (-1), "not an object type");

    
    
    if (file_id != (hid_t)H5F_OBJ_ALL) {
        H5VL_object_t       *vol_obj;     
        size_t               count = 0;   
        H5VL_file_get_args_t vol_cb_args; 

        
        if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, (-1), "not a file id");

        
        vol_cb_args.op_type                  = H5VL_FILE_GET_OBJ_COUNT;
        vol_cb_args.args.get_obj_count.types = types;
        vol_cb_args.args.get_obj_count.count = &count;

        
        if (H5VL_file_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, (-1), "unable to get object count in file(s)");

        
        ret_value = (ssize_t)count;
    }
    
    else {
        H5F_trav_obj_cnt_t udata;

        
        udata.types     = types | H5F_OBJ_LOCAL;
        udata.obj_count = 0;

        if (types & H5F_OBJ_FILE)
            if (H5I_iterate(H5I_FILE, H5F__get_all_count_cb, &udata, true) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_BADITER, (-1), "iteration over file IDs failed");
        if (types & H5F_OBJ_DATASET)
            if (H5I_iterate(H5I_DATASET, H5F__get_all_count_cb, &udata, true) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_BADITER, (-1), "iteration over dataset IDs failed");
        if (types & H5F_OBJ_GROUP)
            if (H5I_iterate(H5I_GROUP, H5F__get_all_count_cb, &udata, true) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_BADITER, (-1), "iteration over group IDs failed");
        if (types & H5F_OBJ_DATATYPE)
            if (H5I_iterate(H5I_DATATYPE, H5F__get_all_count_cb, &udata, true) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_BADITER, (-1), "iteration over datatype IDs failed");
        if (types & H5F_OBJ_ATTR)
            if (H5I_iterate(H5I_ATTR, H5F__get_all_count_cb, &udata, true) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_BADITER, (-1), "iteration over attribute IDs failed");

        
        ret_value = (ssize_t)udata.obj_count;
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

static int
H5F__get_all_ids_cb(void H5_ATTR_UNUSED *obj_ptr, hid_t obj_id, void *key)
{
    H5F_trav_obj_ids_t *udata     = (H5F_trav_obj_ids_t *)key;
    int                 ret_value = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE_NOERR

    if (udata->obj_count >= udata->max_objs)
        HGOTO_DONE(H5_ITER_STOP);

    
    udata->oid_list[udata->obj_count] = obj_id;
    udata->obj_count++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

ssize_t
H5Fget_obj_ids(hid_t file_id, unsigned types, size_t max_objs, hid_t *oid_list )
{
    ssize_t ret_value = 0; 

    FUNC_ENTER_API((-1))

    
    if (0 == (types & H5F_OBJ_ALL))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, (-1), "not an object type");
    if (!oid_list)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, (-1), "object ID list cannot be NULL");

    
    
    if (file_id != (hid_t)H5F_OBJ_ALL) {
        H5VL_object_t       *vol_obj;     
        size_t               count = 0;   
        H5VL_file_get_args_t vol_cb_args; 

        
        if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, (-1), "invalid file identifier");

        
        vol_cb_args.op_type                   = H5VL_FILE_GET_OBJ_IDS;
        vol_cb_args.args.get_obj_ids.types    = types;
        vol_cb_args.args.get_obj_ids.max_objs = max_objs;
        vol_cb_args.args.get_obj_ids.oid_list = oid_list;
        vol_cb_args.args.get_obj_ids.count    = &count;

        
        if (H5VL_file_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, (-1), "unable to get object ids in file(s)");

        
        ret_value = (ssize_t)count;
    } 
    
    else {
        H5F_trav_obj_ids_t udata;

        
        udata.max_objs  = max_objs;
        udata.oid_list  = oid_list;
        udata.obj_count = 0;

        if (types & H5F_OBJ_FILE)
            if (H5I_iterate(H5I_FILE, H5F__get_all_ids_cb, &udata, true) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_BADITER, (-1), "iteration over file IDs failed");
        if (types & H5F_OBJ_DATASET)
            if (H5I_iterate(H5I_DATASET, H5F__get_all_ids_cb, &udata, true) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_BADITER, (-1), "iteration over dataset IDs failed");
        if (types & H5F_OBJ_GROUP)
            if (H5I_iterate(H5I_GROUP, H5F__get_all_ids_cb, &udata, true) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_BADITER, (-1), "iteration over group IDs failed");
        if (types & H5F_OBJ_DATATYPE)
            if (H5I_iterate(H5I_DATATYPE, H5F__get_all_ids_cb, &udata, true) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_BADITER, (-1), "iteration over datatype IDs failed");
        if (types & H5F_OBJ_ATTR)
            if (H5I_iterate(H5I_ATTR, H5F__get_all_ids_cb, &udata, true) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_BADITER, (-1), "iteration over attribute IDs failed");

        
        ret_value = (ssize_t)udata.obj_count;
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fget_vfd_handle(hid_t file_id, hid_t fapl_id, void **file_handle )
{
    H5VL_object_t                   *vol_obj;             
    H5VL_optional_args_t             vol_cb_args;         
    H5VL_native_file_optional_args_t file_opt_args;       
    herr_t                           ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file_handle)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid file handle pointer");

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid file identifier");

    
    file_opt_args.get_vfd_handle.fapl_id     = fapl_id;
    file_opt_args.get_vfd_handle.file_handle = file_handle;
    vol_cb_args.op_type                      = H5VL_NATIVE_FILE_GET_VFD_HANDLE;
    vol_cb_args.args                         = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to get VFD handle");

done:
    FUNC_LEAVE_API(ret_value)
} 

htri_t
H5Fis_accessible(const char *filename, hid_t fapl_id)
{
    H5VL_file_specific_args_t vol_cb_args;           
    bool                      is_accessible = false; 
    htri_t                    ret_value;             

    FUNC_ENTER_API(FAIL)

    
    if (!filename || !*filename)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "no file name specified");

    
    if (H5P_DEFAULT == fapl_id)
        fapl_id = H5P_FILE_ACCESS_DEFAULT;
    else if (true != H5P_isa_class(fapl_id, H5P_FILE_ACCESS))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not file access property list");

    
    vol_cb_args.op_type                       = H5VL_FILE_IS_ACCESSIBLE;
    vol_cb_args.args.is_accessible.filename   = filename;
    vol_cb_args.args.is_accessible.fapl_id    = fapl_id;
    vol_cb_args.args.is_accessible.accessible = &is_accessible;

    
    if (H5VL_file_specific(NULL, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_NOTHDF5, FAIL, "unable to determine if file is accessible as HDF5");

    
    ret_value = (htri_t)is_accessible;

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5F__post_open_api_common(H5VL_object_t *vol_obj, void **token_ptr)
{
    uint64_t supported;           
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    supported = 0;
    if (H5VL_introspect_opt_query(vol_obj, H5VL_SUBCLS_FILE, H5VL_NATIVE_FILE_POST_OPEN, &supported) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't check for 'post open' operation");
    if (supported & H5VL_OPT_QUERY_SUPPORTED) {
        H5VL_optional_args_t vol_cb_args; 

        
        vol_cb_args.op_type = H5VL_NATIVE_FILE_POST_OPEN;
        vol_cb_args.args    = NULL;

        
        if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, token_ptr) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "unable to make file 'post open' callback");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static hid_t
H5F__create_api_common(const char *filename, unsigned flags, hid_t fcpl_id, hid_t fapl_id, void **token_ptr)
{
    void                 *new_file = NULL;             
    H5P_genplist_t       *plist;                       
    H5VL_connector_prop_t connector_prop;              
    hid_t                 ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_PACKAGE

    
    if (!filename || !*filename)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "invalid file name");

    
    if (flags & ~(H5F_ACC_EXCL | H5F_ACC_TRUNC | H5F_ACC_SWMR_WRITE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "invalid flags");

    
    if ((flags & H5F_ACC_EXCL) && (flags & H5F_ACC_TRUNC))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "mutually exclusive flags for file creation");

    
    if (H5P_DEFAULT == fcpl_id)
        fcpl_id = H5P_FILE_CREATE_DEFAULT;
    else if (true != H5P_isa_class(fcpl_id, H5P_FILE_CREATE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not file create property list");

    
    if (H5CX_set_apl(&fapl_id, H5P_CLS_FACC, H5I_INVALID_HID, true) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, H5I_INVALID_HID, "can't set access property list info");

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object(fapl_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not a file access property list");
    if (H5P_peek(plist, H5F_ACS_VOL_CONN_NAME, &connector_prop) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, H5I_INVALID_HID, "can't get VOL connector info");

    
    if (H5CX_set_vol_connector_prop(&connector_prop) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, H5I_INVALID_HID, "can't set VOL connector info in API context");

    
    if (0 == (flags & (H5F_ACC_EXCL | H5F_ACC_TRUNC)))
        flags |= H5F_ACC_EXCL; 
    flags |= H5F_ACC_RDWR | H5F_ACC_CREAT;

    
    if (NULL == (new_file = H5VL_file_create(connector_prop.connector, filename, flags, fcpl_id, fapl_id,
                                             H5P_DATASET_XFER_DEFAULT, token_ptr)))
        HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, H5I_INVALID_HID, "unable to create file");

    
    if ((ret_value = H5VL_register(H5I_FILE, new_file, connector_prop.connector, true)) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register file handle");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5Fcreate(const char *filename, unsigned flags, hid_t fcpl_id, hid_t fapl_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    hid_t          ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if ((ret_value = H5F__create_api_common(filename, flags, fcpl_id, fapl_id, NULL)) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTCREATE, H5I_INVALID_HID, "unable to synchronously create file");

    
    if (NULL == (vol_obj = H5VL_vol_object(ret_value)))
        HGOTO_ERROR(H5E_FILE, H5E_BADTYPE, H5I_INVALID_HID, "invalid object identifier");

    
    if (H5F__post_open_api_common(vol_obj, NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, H5I_INVALID_HID, "'post open' operation failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Fcreate_async(const char *app_file, const char *app_func, unsigned app_line, const char *filename,
                unsigned flags, hid_t fcpl_id, hid_t fapl_id, hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    hid_t          ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token; 

    
    if ((ret_value = H5F__create_api_common(filename, flags, fcpl_id, fapl_id, token_ptr)) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTCREATE, H5I_INVALID_HID, "unable to asynchronously create file");

    
    if (NULL == (vol_obj = H5VL_vol_object(ret_value)))
        HGOTO_ERROR(H5E_FILE, H5E_BADTYPE, H5I_INVALID_HID, "invalid object identifier");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE8(__func__, "*s*sIu*sIuiii", app_file, app_func, app_line, filename, flags, fcpl_id, fapl_id, es_id)) < 0) {
            
            if (H5I_dec_app_ref(ret_value) < 0)
                HDONE_ERROR(H5E_FILE, H5E_CANTDEC, H5I_INVALID_HID, "can't decrement count on file ID");
            HGOTO_ERROR(H5E_FILE, H5E_CANTINSERT, H5I_INVALID_HID, "can't insert token into event set");
        } 

    
    
    token = NULL;

    
    if (H5F__post_open_api_common(vol_obj, token_ptr) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, H5I_INVALID_HID, "'post open' operation failed");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE8(__func__, "*s*sIu*sIuiii", app_file, app_func, app_line, filename, flags, fcpl_id, fapl_id, es_id)) < 0)
            
            HGOTO_ERROR(H5E_FILE, H5E_CANTINSERT, H5I_INVALID_HID, "can't insert token into event set");

done:
    FUNC_LEAVE_API(ret_value)
} 

static hid_t
H5F__open_api_common(const char *filename, unsigned flags, hid_t fapl_id, void **token_ptr)
{
    void                 *new_file = NULL;             
    H5P_genplist_t       *plist;                       
    H5VL_connector_prop_t connector_prop;              
    hid_t                 ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_PACKAGE

    
    if (!filename || !*filename)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "invalid file name");
    
    if ((flags & ~H5F_ACC_PUBLIC_FLAGS) || (flags & H5F_ACC_TRUNC) || (flags & H5F_ACC_EXCL))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "invalid file open flags");
    
    
    if ((flags & H5F_ACC_SWMR_WRITE) && 0 == (flags & H5F_ACC_RDWR))
        HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, H5I_INVALID_HID,
                    "SWMR write access on a file open for read-only access is not allowed");
    
    if ((flags & H5F_ACC_SWMR_READ) && (flags & H5F_ACC_RDWR))
        HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, H5I_INVALID_HID,
                    "SWMR read access on a file open for read-write access is not allowed");

    
    if (H5CX_set_apl(&fapl_id, H5P_CLS_FACC, H5I_INVALID_HID, true) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, H5I_INVALID_HID, "can't set access property list info");

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object(fapl_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not a file access property list");
    if (H5P_peek(plist, H5F_ACS_VOL_CONN_NAME, &connector_prop) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, H5I_INVALID_HID, "can't get VOL connector info");

    
    if (H5CX_set_vol_connector_prop(&connector_prop) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, H5I_INVALID_HID, "can't set VOL connector info in API context");

    
    if (NULL == (new_file = H5VL_file_open(connector_prop.connector, filename, flags, fapl_id,
                                           H5P_DATASET_XFER_DEFAULT, token_ptr)))
        HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, H5I_INVALID_HID, "unable to open file");

    
    if ((ret_value = H5VL_register(H5I_FILE, new_file, connector_prop.connector, true)) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register file handle");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5Fopen(const char *filename, unsigned flags, hid_t fapl_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    hid_t          ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if ((ret_value = H5F__open_api_common(filename, flags, fapl_id, NULL)) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, H5I_INVALID_HID, "unable to synchronously open file");

    
    if (NULL == (vol_obj = H5VL_vol_object(ret_value)))
        HGOTO_ERROR(H5E_FILE, H5E_BADTYPE, H5I_INVALID_HID, "invalid object identifier");

    
    if (H5F__post_open_api_common(vol_obj, NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, H5I_INVALID_HID, "'post open' operation failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Fopen_async(const char *app_file, const char *app_func, unsigned app_line, const char *filename,
              unsigned flags, hid_t fapl_id, hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    hid_t          ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token; 

    
    if ((ret_value = H5F__open_api_common(filename, flags, fapl_id, token_ptr)) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, H5I_INVALID_HID, "unable to asynchronously open file");

    
    if (NULL == (vol_obj = H5VL_vol_object(ret_value)))
        HGOTO_ERROR(H5E_FILE, H5E_BADTYPE, H5I_INVALID_HID, "invalid object identifier");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE7(__func__, "*s*sIu*sIuii", app_file, app_func, app_line, filename, flags, fapl_id, es_id)) < 0) {
            
            if (H5I_dec_app_ref(ret_value) < 0)
                HDONE_ERROR(H5E_FILE, H5E_CANTDEC, H5I_INVALID_HID, "can't decrement count on file ID");
            HGOTO_ERROR(H5E_FILE, H5E_CANTINSERT, H5I_INVALID_HID, "can't insert token into event set");
        } 

    
    
    token = NULL;

    
    if (H5F__post_open_api_common(vol_obj, token_ptr) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, H5I_INVALID_HID, "'post open' operation failed");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE7(__func__, "*s*sIu*sIuii", app_file, app_func, app_line, filename, flags, fapl_id, es_id)) < 0)
            
            HGOTO_ERROR(H5E_FILE, H5E_CANTINSERT, H5I_INVALID_HID, "can't insert token into event set");

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5F__flush_api_common(hid_t object_id, H5F_scope_t scope, void **token_ptr, H5VL_object_t **_vol_obj_ptr)
{
    H5VL_object_t  *tmp_vol_obj = NULL; 
    H5VL_object_t **vol_obj_ptr =
        (_vol_obj_ptr ? _vol_obj_ptr : &tmp_vol_obj); 
    H5I_type_t                obj_type;               
    H5VL_file_specific_args_t vol_cb_args;            
    herr_t                    ret_value = SUCCEED;    

    FUNC_ENTER_PACKAGE

    
    obj_type = H5I_get_type(object_id);
    if (H5I_FILE != obj_type && H5I_GROUP != obj_type && H5I_DATATYPE != obj_type &&
        H5I_DATASET != obj_type && H5I_ATTR != obj_type)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file or file object");

    
    if (NULL == (*vol_obj_ptr = H5VL_vol_object(object_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid object identifier");

    
    vol_cb_args.op_type             = H5VL_FILE_FLUSH;
    vol_cb_args.args.flush.obj_type = obj_type;
    vol_cb_args.args.flush.scope    = scope;

    
    if (H5VL_file_specific(*vol_obj_ptr, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, token_ptr) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTFLUSH, FAIL, "unable to flush file");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Fflush(hid_t object_id, H5F_scope_t scope)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5F__flush_api_common(object_id, scope, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTFLUSH, FAIL, "unable to synchronously flush file");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fflush_async(const char *app_file, const char *app_func, unsigned app_line, hid_t object_id,
               H5F_scope_t scope, hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    herr_t         ret_value = SUCCEED;         

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token; 

    
    if (H5F__flush_api_common(object_id, scope, token_ptr, &vol_obj) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTFLUSH, FAIL, "unable to asynchronously flush file");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                H5ARG_TRACE6(__func__, "*s*sIuiFsi", app_file, app_func, app_line, object_id, scope, es_id)) < 0)
            
            HGOTO_ERROR(H5E_FILE, H5E_CANTINSERT, FAIL, "can't insert token into event set");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fclose(hid_t file_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5I_FILE != H5I_get_type(file_id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file ID");

    
    if (H5I_dec_app_ref(file_id) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTCLOSEFILE, FAIL, "decrementing file ID failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fclose_async(const char *app_file, const char *app_func, unsigned app_line, hid_t file_id, hid_t es_id)
{
    H5VL_object_t    *vol_obj   = NULL;            
    H5VL_connector_t *connector = NULL;            
    void             *token     = NULL;            
    void            **token_ptr = H5_REQUEST_NULL; 
    herr_t            ret_value = SUCCEED;         

    FUNC_ENTER_API(FAIL)

    
    if (H5I_FILE != H5I_get_type(file_id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file ID");

    
    if (H5ES_NONE != es_id) {
        
        if (NULL == (vol_obj = H5VL_vol_object(file_id)))
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't get VOL object for file");

        
        connector = H5VL_OBJ_CONNECTOR(vol_obj);
        H5VL_conn_inc_rc(connector);

        
        token_ptr = &token;
    } 

    
    if (H5I_dec_app_ref_async(file_id, token_ptr) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTCLOSEFILE, FAIL, "decrementing file ID failed");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE5(__func__, "*s*sIuii", app_file, app_func, app_line, file_id, es_id)) < 0)
            
            HGOTO_ERROR(H5E_FILE, H5E_CANTINSERT, FAIL, "can't insert token into event set");

done:
    if (connector && H5VL_conn_dec_rc(connector) < 0)
        HDONE_ERROR(H5E_FILE, H5E_CANTDEC, FAIL, "can't decrement ref count on connector");

    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fdelete(const char *filename, hid_t fapl_id)
{
    H5P_genplist_t           *plist;                 
    H5VL_connector_prop_t     connector_prop;        
    H5VL_file_specific_args_t vol_cb_args;           
    bool                      is_accessible = false; 
    herr_t                    ret_value     = SUCCEED;

    FUNC_ENTER_API(FAIL)

    
    if (!filename || !*filename)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "no file name specified");

    
    if (H5CX_set_apl(&fapl_id, H5P_CLS_FACC, H5I_INVALID_HID, true) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "can't set access property list info");

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object_verify(fapl_id, H5I_GENPROP_LST)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");
    if (H5P_peek(plist, H5F_ACS_VOL_CONN_NAME, &connector_prop) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't get VOL connector info");

    
    if (H5CX_set_vol_connector_prop(&connector_prop) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "can't set VOL connector info in API context");

    
    vol_cb_args.op_type                       = H5VL_FILE_IS_ACCESSIBLE;
    vol_cb_args.args.is_accessible.filename   = filename;
    vol_cb_args.args.is_accessible.fapl_id    = fapl_id;
    vol_cb_args.args.is_accessible.accessible = &is_accessible;

    
    if (H5VL_file_specific(NULL, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_NOTHDF5, FAIL, "unable to determine if file is accessible as HDF5");
    if (!is_accessible)
        HGOTO_ERROR(H5E_FILE, H5E_NOTHDF5, FAIL, "not an HDF5 file");

    
    vol_cb_args.op_type           = H5VL_FILE_DELETE;
    vol_cb_args.args.del.filename = filename;
    vol_cb_args.args.del.fapl_id  = fapl_id;

    
    if (H5VL_file_specific(NULL, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTDELETEFILE, FAIL, "unable to delete the file");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fmount(hid_t loc_id, const char *name, hid_t child_id, hid_t plist_id)
{
    H5VL_object_t             *loc_vol_obj   = NULL; 
    H5VL_object_t             *child_vol_obj = NULL; 
    H5VL_group_specific_args_t vol_cb_args;          
    void                      *grp = NULL;           
    H5I_type_t                 loc_type;             
    htri_t                     same_connector; 
    herr_t                     ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    loc_type = H5I_get_type(loc_id);
    if (H5I_FILE != loc_type && H5I_GROUP != loc_type)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "loc_id parameter not a file or group ID");
    if (!name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "name parameter cannot be NULL");
    if (!*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "name parameter cannot be the empty string");
    if (H5I_FILE != H5I_get_type(child_id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "child_id parameter not a file ID");
    if (H5P_DEFAULT == plist_id)
        plist_id = H5P_FILE_MOUNT_DEFAULT;
    else if (true != H5P_isa_class(plist_id, H5P_FILE_MOUNT))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "plist_id is not a file mount property list ID");

    
    if (H5CX_set_loc(loc_id) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "can't set collective metadata read info");

    
    
    if (H5I_FILE == loc_type) {
        H5VL_object_t    *vol_obj;    
        H5VL_loc_params_t loc_params; 

        
        if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

        
        loc_params.type     = H5VL_OBJECT_BY_SELF;
        loc_params.obj_type = loc_type;

        
        if (NULL == (grp = H5VL_group_open(vol_obj, &loc_params, "/", H5P_GROUP_ACCESS_DEFAULT,
                                           H5P_DATASET_XFER_DEFAULT, NULL)))
            HGOTO_ERROR(H5E_FILE, H5E_CANTOPENOBJ, FAIL, "unable to open group");

        
        if (NULL == (loc_vol_obj = H5VL_create_object(grp, H5VL_OBJ_CONNECTOR(vol_obj))))
            HGOTO_ERROR(H5E_FILE, H5E_CANTOPENOBJ, FAIL, "can't create VOL object for root group");
    } 
    else {
        assert(H5I_GROUP == loc_type);
        if (NULL == (loc_vol_obj = H5VL_vol_object(loc_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "could not get location object");
    } 

    
    if (NULL == (child_vol_obj = H5VL_vol_object(child_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "could not get child object");

    
    if ((same_connector =
             H5VL_conn_same_class(H5VL_OBJ_CONNECTOR(loc_vol_obj), H5VL_OBJ_CONNECTOR(child_vol_obj))) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTCOMPARE, FAIL, "can't compare connector classes");
    if (!same_connector)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                    "can't mount file onto object from different VOL connector");

    
    vol_cb_args.op_type         = H5VL_GROUP_MOUNT;
    vol_cb_args.args.mount.name = name;
    vol_cb_args.args.mount.child_file =
        H5VL_OBJ_DATA(child_vol_obj); 
    vol_cb_args.args.mount.fmpl_id = plist_id;

    
    
    if (H5VL_group_specific(loc_vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_MOUNT, FAIL, "unable to mount file");

done:
    
    if (grp) {
        assert(loc_vol_obj);
        if (H5VL_group_close(loc_vol_obj, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HDONE_ERROR(H5E_FILE, H5E_CLOSEERROR, FAIL, "unable to release group");
        if (H5VL_free_object(loc_vol_obj) < 0)
            HDONE_ERROR(H5E_FILE, H5E_CANTDEC, FAIL, "unable to free VOL object");
    } 

    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Funmount(hid_t loc_id, const char *name)
{
    H5VL_object_t             *loc_vol_obj = NULL;  
    H5VL_group_specific_args_t vol_cb_args;         
    void                      *grp = NULL;          
    H5I_type_t                 loc_type;            
    herr_t                     ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    loc_type = H5I_get_type(loc_id);
    if (H5I_FILE != loc_type && H5I_GROUP != loc_type)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "loc_id parameter not a file or group ID");
    if (!name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "name parameter cannot be NULL");
    if (!*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "name parameter cannot be the empty string");

    
    if (H5CX_set_loc(loc_id) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "can't set collective metadata read info");

    
    
    if (H5I_FILE == loc_type) {
        H5VL_object_t    *vol_obj;    
        H5VL_loc_params_t loc_params; 

        
        if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

        
        loc_params.type     = H5VL_OBJECT_BY_SELF;
        loc_params.obj_type = loc_type;

        
        if (NULL == (grp = H5VL_group_open(vol_obj, &loc_params, "/", H5P_GROUP_ACCESS_DEFAULT,
                                           H5P_DATASET_XFER_DEFAULT, NULL)))
            HGOTO_ERROR(H5E_FILE, H5E_CANTOPENOBJ, FAIL, "unable to open group");

        
        if (NULL == (loc_vol_obj = H5VL_create_object(grp, H5VL_OBJ_CONNECTOR(vol_obj))))
            HGOTO_ERROR(H5E_FILE, H5E_CANTOPENOBJ, FAIL, "can't create VOL object for root group");
    } 
    else {
        assert(H5I_GROUP == loc_type);
        if (NULL == (loc_vol_obj = H5VL_vol_object(loc_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "could not get location object");
    } 

    
    vol_cb_args.op_type           = H5VL_GROUP_UNMOUNT;
    vol_cb_args.args.unmount.name = name;

    
    
    if (H5VL_group_specific(loc_vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_MOUNT, FAIL, "unable to unmount file");

done:
    
    if (grp) {
        assert(loc_vol_obj);
        if (H5VL_group_close(loc_vol_obj, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HDONE_ERROR(H5E_FILE, H5E_CLOSEERROR, FAIL, "unable to release group");
        if (H5VL_free_object(loc_vol_obj) < 0)
            HDONE_ERROR(H5E_FILE, H5E_CANTDEC, FAIL, "unable to free VOL object");
    } 

    FUNC_LEAVE_API(ret_value)
} 

static hid_t
H5F__reopen_api_common(hid_t file_id, void **token_ptr)
{
    H5VL_object_t            *vol_obj = NULL;                
    H5VL_file_specific_args_t vol_cb_args;                   
    void                     *reopen_file = NULL;            
    hid_t                     ret_value   = H5I_INVALID_HID; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid file identifier");

    
    vol_cb_args.op_type          = H5VL_FILE_REOPEN;
    vol_cb_args.args.reopen.file = &reopen_file;

    
    if (H5VL_file_specific(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, token_ptr) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, H5I_INVALID_HID, "unable to reopen file via the VOL connector");

    
    if (NULL == reopen_file)
        HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, H5I_INVALID_HID, "unable to reopen file");

    
    if ((ret_value = H5VL_register(H5I_FILE, reopen_file, H5VL_OBJ_CONNECTOR(vol_obj), true)) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register file handle");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5Freopen(hid_t file_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    hid_t          ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if ((ret_value = H5F__reopen_api_common(file_id, NULL)) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, H5I_INVALID_HID, "unable to synchronously reopen file");

    
    if (NULL == (vol_obj = H5VL_vol_object(ret_value)))
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, H5I_INVALID_HID, "can't get handle for re-opened file");

    
    if (H5F__post_open_api_common(vol_obj, NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, H5I_INVALID_HID, "'post open' operation failed");

done:
    
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Freopen_async(const char *app_file, const char *app_func, unsigned app_line, hid_t file_id, hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    hid_t          ret_value;                   

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token; 

    
    if ((ret_value = H5F__reopen_api_common(file_id, token_ptr)) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, H5I_INVALID_HID, "unable to asynchronously reopen file");

    
    if (NULL == (vol_obj = H5VL_vol_object(ret_value)))
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, H5I_INVALID_HID, "can't get handle for re-opened file");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE5(__func__, "*s*sIuii", app_file, app_func, app_line, file_id, es_id)) < 0) {
            
            if (H5I_dec_app_ref(ret_value) < 0)
                HDONE_ERROR(H5E_FILE, H5E_CANTDEC, H5I_INVALID_HID, "can't decrement count on file ID");
            HGOTO_ERROR(H5E_FILE, H5E_CANTINSERT, H5I_INVALID_HID, "can't insert token into event set");
        } 

    
    
    token = NULL;

    
    if (H5F__post_open_api_common(vol_obj, token_ptr) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, H5I_INVALID_HID, "'post open' operation failed");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE5(__func__, "*s*sIuii", app_file, app_func, app_line, file_id, es_id)) < 0)
            
            HGOTO_ERROR(H5E_FILE, H5E_CANTINSERT, H5I_INVALID_HID, "can't insert token into event set");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fget_intent(hid_t file_id, unsigned *intent_flags )
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_API(FAIL)

    
    if (intent_flags) {
        H5VL_object_t       *vol_obj;     
        H5VL_file_get_args_t vol_cb_args; 

        
        if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid file identifier");

        
        vol_cb_args.op_type               = H5VL_FILE_GET_INTENT;
        vol_cb_args.args.get_intent.flags = intent_flags;

        
        if (H5VL_file_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to get file's intent flags");
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fget_fileno(hid_t file_id, unsigned long *fnumber )
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_API(FAIL)

    
    if (fnumber) {
        H5VL_object_t       *vol_obj;     
        H5VL_file_get_args_t vol_cb_args; 

        
        if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid file identifier");

        
        vol_cb_args.op_type                = H5VL_FILE_GET_FILENO;
        vol_cb_args.args.get_fileno.fileno = fnumber;

        
        if (H5VL_file_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to get file's 'file number'");
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

hssize_t
H5Fget_freespace(hid_t file_id)
{
    H5VL_object_t                   *vol_obj = NULL;
    H5VL_optional_args_t             vol_cb_args;        
    H5VL_native_file_optional_args_t file_opt_args;      
    hsize_t                          file_freespace = 0; 
    hssize_t                         ret_value;          

    FUNC_ENTER_API((-1))

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, (-1), "invalid file identifier");

    
    file_opt_args.get_freespace.size = &file_freespace;
    vol_cb_args.op_type              = H5VL_NATIVE_FILE_GET_FREE_SPACE;
    vol_cb_args.args                 = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, (-1), "unable to get file free space");

    
    ret_value = (hssize_t)file_freespace;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fget_filesize(hid_t file_id, hsize_t *size )
{
    H5VL_object_t                   *vol_obj;             
    H5VL_optional_args_t             vol_cb_args;         
    H5VL_native_file_optional_args_t file_opt_args;       
    herr_t                           ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!size)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "size parameter cannot be NULL");
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "not a file ID");

    
    file_opt_args.get_size.size = size;
    vol_cb_args.op_type         = H5VL_NATIVE_FILE_GET_SIZE;
    vol_cb_args.args            = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to get file size");

done:
    FUNC_LEAVE_API(ret_value)
} 

ssize_t
H5Fget_file_image(hid_t file_id, void *buf , size_t buf_len)
{
    H5VL_object_t                   *vol_obj;       
    H5VL_optional_args_t             vol_cb_args;   
    H5VL_native_file_optional_args_t file_opt_args; 
    size_t                           image_len = 0; 
    ssize_t                          ret_value;     

    FUNC_ENTER_API((-1))

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, (-1), "not a file ID");

    
    file_opt_args.get_file_image.buf_size  = buf_len;
    file_opt_args.get_file_image.buf       = buf;
    file_opt_args.get_file_image.image_len = &image_len;
    vol_cb_args.op_type                    = H5VL_NATIVE_FILE_GET_FILE_IMAGE;
    vol_cb_args.args                       = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, (-1), "unable to get file image");

    
    ret_value = (ssize_t)image_len;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fget_mdc_config(hid_t file_id, H5AC_cache_config_t *config )
{
    H5VL_object_t                   *vol_obj = NULL;
    H5VL_optional_args_t             vol_cb_args;         
    H5VL_native_file_optional_args_t file_opt_args;       
    herr_t                           ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if ((NULL == config) || (config->version != H5AC__CURR_CACHE_CONFIG_VERSION))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "Bad config ptr");

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid file identifier");

    
    file_opt_args.get_mdc_config.config = config;
    vol_cb_args.op_type                 = H5VL_NATIVE_FILE_GET_MDC_CONF;
    vol_cb_args.args                    = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to get metadata cache configuration");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fset_mdc_config(hid_t file_id, const H5AC_cache_config_t *config_ptr)
{
    H5VL_object_t                   *vol_obj = NULL;
    H5VL_optional_args_t             vol_cb_args;         
    H5VL_native_file_optional_args_t file_opt_args;       
    herr_t                           ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid file identifier");

    
    file_opt_args.set_mdc_config.config = config_ptr;
    vol_cb_args.op_type                 = H5VL_NATIVE_FILE_SET_MDC_CONFIG;
    vol_cb_args.args                    = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "unable to set metadata cache configuration");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fget_mdc_hit_rate(hid_t file_id, double *hit_rate )
{
    H5VL_object_t                   *vol_obj;
    H5VL_optional_args_t             vol_cb_args;         
    H5VL_native_file_optional_args_t file_opt_args;       
    herr_t                           ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == hit_rate)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "NULL hit rate pointer");
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "not a file ID");

    
    file_opt_args.get_mdc_hit_rate.hit_rate = hit_rate;
    vol_cb_args.op_type                     = H5VL_NATIVE_FILE_GET_MDC_HR;
    vol_cb_args.args                        = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to get MDC hit rate");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fget_mdc_size(hid_t file_id, size_t *max_size , size_t *min_clean_size ,
                size_t *cur_size , int *cur_num_entries )
{
    H5VL_object_t                   *vol_obj;
    H5VL_optional_args_t             vol_cb_args;         
    H5VL_native_file_optional_args_t file_opt_args;       
    uint32_t                         index_len = 0;       
    herr_t                           ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "not a file ID");

    
    file_opt_args.get_mdc_size.max_size        = max_size;
    file_opt_args.get_mdc_size.min_clean_size  = min_clean_size;
    file_opt_args.get_mdc_size.cur_size        = cur_size;
    file_opt_args.get_mdc_size.cur_num_entries = &index_len;
    vol_cb_args.op_type                        = H5VL_NATIVE_FILE_GET_MDC_SIZE;
    vol_cb_args.args                           = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to get MDC size");

    
    if (cur_num_entries)
        *cur_num_entries = (int)index_len;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Freset_mdc_hit_rate_stats(hid_t file_id)
{
    H5VL_object_t       *vol_obj = NULL;
    H5VL_optional_args_t vol_cb_args;         
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid file identifier");

    
    vol_cb_args.op_type = H5VL_NATIVE_FILE_RESET_MDC_HIT_RATE;
    vol_cb_args.args    = NULL;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "can't reset cache hit rate");

done:
    FUNC_LEAVE_API(ret_value)
} 

ssize_t
H5Fget_name(hid_t obj_id, char *name , size_t size)
{
    H5VL_object_t       *vol_obj;     
    H5VL_file_get_args_t vol_cb_args; 
    H5I_type_t           type;
    size_t               file_name_len = 0;  
    ssize_t              ret_value     = -1; 

    FUNC_ENTER_API((-1))

    
    type = H5I_get_type(obj_id);
    if (H5I_FILE != type && H5I_GROUP != type && H5I_DATATYPE != type && H5I_DATASET != type &&
        H5I_ATTR != type)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, (-1), "not a file or file object");

    
    if (NULL == (vol_obj = H5VL_vol_object(obj_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, (-1), "invalid file identifier");

    
    vol_cb_args.op_type                     = H5VL_FILE_GET_NAME;
    vol_cb_args.args.get_name.type          = type;
    vol_cb_args.args.get_name.buf_size      = size;
    vol_cb_args.args.get_name.buf           = name;
    vol_cb_args.args.get_name.file_name_len = &file_name_len;

    
    if (H5VL_file_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, (-1), "unable to get file name");

    
    ret_value = (ssize_t)file_name_len;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fget_info2(hid_t obj_id, H5F_info2_t *finfo )
{
    H5VL_object_t                   *vol_obj = NULL;
    H5VL_optional_args_t             vol_cb_args;   
    H5VL_native_file_optional_args_t file_opt_args; 
    H5I_type_t                       type;
    herr_t                           ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!finfo)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file info pointer can't be NULL");

    
    type = H5I_get_type(obj_id);
    if (H5I_FILE != type && H5I_GROUP != type && H5I_DATATYPE != type && H5I_DATASET != type &&
        H5I_ATTR != type)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file or file object");

    
    if (NULL == (vol_obj = H5VL_vol_object(obj_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid object identifier");

    
    file_opt_args.get_info.type  = type;
    file_opt_args.get_info.finfo = finfo;
    vol_cb_args.op_type          = H5VL_NATIVE_FILE_GET_INFO;
    vol_cb_args.args             = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to retrieve file info");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fget_metadata_read_retry_info(hid_t file_id, H5F_retry_info_t *info )
{
    H5VL_object_t                   *vol_obj = NULL;      
    H5VL_optional_args_t             vol_cb_args;         
    H5VL_native_file_optional_args_t file_opt_args;       
    herr_t                           ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!info)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no info struct");

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "not a file ID");

    
    file_opt_args.get_metadata_read_retry_info.info = info;
    vol_cb_args.op_type                             = H5VL_NATIVE_FILE_GET_METADATA_READ_RETRY_INFO;
    vol_cb_args.args                                = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't get metadata read retry info");

done:
    FUNC_LEAVE_API(ret_value)
} 

ssize_t
H5Fget_free_sections(hid_t file_id, H5F_mem_t type, size_t nsects, H5F_sect_info_t *sect_info )
{
    H5VL_object_t                   *vol_obj = NULL;
    H5VL_optional_args_t             vol_cb_args;     
    H5VL_native_file_optional_args_t file_opt_args;   
    size_t                           sect_count = 0;  
    ssize_t                          ret_value  = -1; 

    FUNC_ENTER_API((-1))

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, (-1), "invalid file identifier");
    if (sect_info && nsects == 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, (-1), "nsects must be > 0");

    
    file_opt_args.get_free_sections.type       = type;
    file_opt_args.get_free_sections.sect_info  = sect_info;
    file_opt_args.get_free_sections.nsects     = nsects;
    file_opt_args.get_free_sections.sect_count = &sect_count;
    vol_cb_args.op_type                        = H5VL_NATIVE_FILE_GET_FREE_SECTIONS;
    vol_cb_args.args                           = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, (-1), "unable to get file free sections");

    
    ret_value = (ssize_t)sect_count;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fclear_elink_file_cache(hid_t file_id)
{
    H5VL_object_t       *vol_obj;             
    H5VL_optional_args_t vol_cb_args;         
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "not a file ID");

    
    vol_cb_args.op_type = H5VL_NATIVE_FILE_CLEAR_ELINK_CACHE;
    vol_cb_args.args    = NULL;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTRELEASE, FAIL, "can't release external file cache");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fstart_swmr_write(hid_t file_id)
{
    H5VL_object_t       *vol_obj = NULL;      
    H5VL_optional_args_t vol_cb_args;         
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "hid_t identifier is not a file ID");

    
    if (H5CX_set_loc(file_id) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "can't set collective metadata read info");

    
    vol_cb_args.op_type = H5VL_NATIVE_FILE_START_SWMR_WRITE;
    vol_cb_args.args    = NULL;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_SYSTEM, FAIL, "unable to start SWMR writing");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fstart_mdc_logging(hid_t file_id)
{
    H5VL_object_t       *vol_obj;             
    H5VL_optional_args_t vol_cb_args;         
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "hid_t identifier is not a file ID");

    
    vol_cb_args.op_type = H5VL_NATIVE_FILE_START_MDC_LOGGING;
    vol_cb_args.args    = NULL;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_LOGGING, FAIL, "unable to start mdc logging");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fstop_mdc_logging(hid_t file_id)
{
    H5VL_object_t       *vol_obj;             
    H5VL_optional_args_t vol_cb_args;         
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "hid_t identifier is not a file ID");

    
    vol_cb_args.op_type = H5VL_NATIVE_FILE_STOP_MDC_LOGGING;
    vol_cb_args.args    = NULL;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_LOGGING, FAIL, "unable to stop mdc logging");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fget_mdc_logging_status(hid_t file_id, bool *is_enabled , bool *is_currently_logging )
{
    H5VL_object_t                   *vol_obj;             
    H5VL_optional_args_t             vol_cb_args;         
    H5VL_native_file_optional_args_t file_opt_args;       
    herr_t                           ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "hid_t identifier is not a file ID");

    
    file_opt_args.get_mdc_logging_status.is_enabled           = is_enabled;
    file_opt_args.get_mdc_logging_status.is_currently_logging = is_currently_logging;
    vol_cb_args.op_type                                       = H5VL_NATIVE_FILE_GET_MDC_LOGGING_STATUS;
    vol_cb_args.args                                          = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_LOGGING, FAIL, "unable to get logging status");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fset_libver_bounds(hid_t file_id, H5F_libver_t low, H5F_libver_t high)
{
    H5VL_object_t                   *vol_obj;             
    H5VL_optional_args_t             vol_cb_args;         
    H5VL_native_file_optional_args_t file_opt_args;       
    herr_t                           ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, FAIL, "not a file ID");

    
    if (H5CX_set_loc(file_id) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "can't set collective metadata read info");

    
    file_opt_args.set_libver_bounds.low  = low;
    file_opt_args.set_libver_bounds.high = high;
    vol_cb_args.op_type                  = H5VL_NATIVE_FILE_SET_LIBVER_BOUNDS;
    vol_cb_args.args                     = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "can't set library version bounds");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fformat_convert(hid_t file_id)
{
    H5VL_object_t       *vol_obj = NULL;      
    H5VL_optional_args_t vol_cb_args;         
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "file_id parameter is not a valid file identifier");

    
    if (H5CX_set_loc(file_id) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "can't set collective metadata read info");

    
    vol_cb_args.op_type = H5VL_NATIVE_FILE_FORMAT_CONVERT;
    vol_cb_args.args    = NULL;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTCONVERT, FAIL, "can't convert file format");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Freset_page_buffering_stats(hid_t file_id)
{
    H5VL_object_t       *vol_obj;             
    H5VL_optional_args_t vol_cb_args;         
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid file identifier");

    
    vol_cb_args.op_type = H5VL_NATIVE_FILE_RESET_PAGE_BUFFERING_STATS;
    vol_cb_args.args    = NULL;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "can't reset stats for page buffering");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fget_page_buffering_stats(hid_t file_id, unsigned accesses[2] , unsigned hits[2] ,
                            unsigned misses[2] , unsigned evictions[2] ,
                            unsigned bypasses[2] )
{
    H5VL_object_t                   *vol_obj;             
    H5VL_optional_args_t             vol_cb_args;         
    H5VL_native_file_optional_args_t file_opt_args;       
    herr_t                           ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "not a file ID");
    if (NULL == accesses || NULL == hits || NULL == misses || NULL == evictions || NULL == bypasses)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "NULL input parameters for stats");

    
    file_opt_args.get_page_buffering_stats.accesses  = accesses;
    file_opt_args.get_page_buffering_stats.hits      = hits;
    file_opt_args.get_page_buffering_stats.misses    = misses;
    file_opt_args.get_page_buffering_stats.evictions = evictions;
    file_opt_args.get_page_buffering_stats.bypasses  = bypasses;
    vol_cb_args.op_type                              = H5VL_NATIVE_FILE_GET_PAGE_BUFFERING_STATS;
    vol_cb_args.args                                 = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't retrieve stats for page buffering");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fget_mdc_image_info(hid_t file_id, haddr_t *image_addr , hsize_t *image_len )
{
    H5VL_object_t                   *vol_obj;             
    H5VL_optional_args_t             vol_cb_args;         
    H5VL_native_file_optional_args_t file_opt_args;       
    herr_t                           ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "hid_t identifier is not a file ID");

    
    file_opt_args.get_mdc_image_info.addr = image_addr;
    file_opt_args.get_mdc_image_info.len  = image_len;
    vol_cb_args.op_type                   = H5VL_NATIVE_FILE_GET_MDC_IMAGE_INFO;
    vol_cb_args.args                      = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't retrieve cache image info");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fget_eoa(hid_t file_id, haddr_t *eoa )
{
    H5VL_object_t *vol_obj;             
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "hid_t identifier is not a file ID");

    
    if (eoa) {
        H5VL_optional_args_t             vol_cb_args;   
        H5VL_native_file_optional_args_t file_opt_args; 

        
        file_opt_args.get_eoa.eoa = eoa;
        vol_cb_args.op_type       = H5VL_NATIVE_FILE_GET_EOA;
        vol_cb_args.args          = &file_opt_args;

        
        if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to get EOA");
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fincrement_filesize(hid_t file_id, hsize_t increment)
{
    H5VL_object_t                   *vol_obj;             
    H5VL_optional_args_t             vol_cb_args;         
    H5VL_native_file_optional_args_t file_opt_args;       
    herr_t                           ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "hid_t identifier is not a file ID");

    
    file_opt_args.increment_filesize.increment = increment;
    vol_cb_args.op_type                        = H5VL_NATIVE_FILE_INCR_FILESIZE;
    vol_cb_args.args                           = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "unable to increment file size");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fget_dset_no_attrs_hint(hid_t file_id, bool *minimize )
{
    H5VL_object_t                   *vol_obj;             
    H5VL_optional_args_t             vol_cb_args;         
    H5VL_native_file_optional_args_t file_opt_args;       
    herr_t                           ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == minimize)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "out pointer 'minimize' cannot be NULL");
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid file identifier");

    
    file_opt_args.get_min_dset_ohdr_flag.minimize = minimize;
    vol_cb_args.op_type                           = H5VL_NATIVE_FILE_GET_MIN_DSET_OHDR_FLAG;
    vol_cb_args.args                              = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "unable to set file's dataset header minimization flag");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Fset_dset_no_attrs_hint(hid_t file_id, bool minimize)
{
    H5VL_object_t                   *vol_obj;             
    H5VL_optional_args_t             vol_cb_args;         
    H5VL_native_file_optional_args_t file_opt_args;       
    herr_t                           ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid file identifier");

    
    file_opt_args.set_min_dset_ohdr_flag.minimize = minimize;
    vol_cb_args.op_type                           = H5VL_NATIVE_FILE_SET_MIN_DSET_OHDR_FLAG;
    vol_cb_args.args                              = &file_opt_args;

    
    if (H5VL_file_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "unable to set file's dataset header minimization flag");

done:
    FUNC_LEAVE_API(ret_value)
} 
