% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pulstran.R
\name{pulstran}
\alias{pulstran}
\title{Pulse train}
\usage{
pulstran(
  t,
  d,
  func,
  fs = 1,
  method = c("linear", "nearest", "cubic", "spline"),
  ...
)
}
\arguments{
\item{t}{Time values at which \code{func} is evaluated, specified as a
vector.}

\item{d}{Offset removed from the values of the array \code{t}, specified as a
real vector, matrix, or array. You can apply an optional gain factor to
each delayed evaluation by specifying \code{d} as a two-column matrix, with
offset defined in column 1 and associated gain in column 2. If you specify
\code{d} as a vector, the values are interpreted as delays only.}

\item{func}{Continuous function used to generate a pulse train based on its
samples, specified as 'rectpuls', 'gauspuls', 'tripuls', or a function
handle. If you use \code{func} as a function handle, you can pass the
function parameters as follows:\cr \code{y <- pulstran(t, d, 'gauspuls',
10e3, bw = 0.5)}.\cr This creates a pulse train using a 10 kHz Gaussian
pulse with 50\% bandwidth. Alternatively, \code{func} can be a prototype
function, specified as a vector. The interval of the function \code{0} to
\code{(length(p) - 1) / fs}, and its samples are identically zero outside
this interval. By default, linear interpolation is used for generating
delays.}

\item{fs}{Sample rate in Hz, specified as a real scalar.}

\item{method}{Interpolation method, specified as one of the following
  options:
\describe{
  \item{"linear" (default)}{Linear interpolation. The interpolated value at a
  query point is based on linear interpolation of the values at neighboring
  grid points in each respective dimension. This is the default interpolation
  method.}
  \item{"nearest"}{Nearest neighbor interpolation. The interpolated value at a
  query point is the value at the nearest sample grid point.}
  \item{"cubic"}{Shape-preserving piecewise cubic interpolation. The
  interpolated value at a query point is based on a shape-preserving
  piecewise cubic interpolation of the values at neighboring grid points.}
  \item{"spline"}{Spline interpolation using not-a-knot end conditions. The
  interpolated value at a query point is based on a cubic interpolation of
  the values at neighboring grid points in each respective dimension.}
}
Interpolation is performed by the function \code{'interp1'} function in the
library \code{'pracma'}, and any interpolation method accepted by the
function \code{'interp1'} can be specified here.}

\item{...}{Further arguments passed to \code{func}.}
}
\value{
Pulse train generated by the function, returned as a vector.
}
\description{
Generate a train of pulses based on samples of a continuous function.
}
\details{
Generate the signal \code{y <- sum(func(t + d, ...))} for each \code{d}. If
\code{d} is a matrix of two columns, the first column is the delay \code{d}
and the second column is the amplitude \code{a}, and \code{y <- sum(a *
func(t + d))} for each \code{d, a}. Clearly, \code{func} must be a function
which accepts a vector of times. Any extra arguments needed for the function
must be tagged on the end.

If instead of a function name you supply a pulse shape sampled at frequency
\code{fs} (default 1 Hz), an interpolated version of the pulse is added at
each delay \code{d}. The interpolation stays within the the time range of the
delayed pulse. The interpolation method defaults to linear, but it can be any
interpolation method accepted by the function \code{interp1}
}
\examples{

## periodic rectangular pulse
t <- seq(0, 60, 1/1e3)
d <- cbind(seq(0, 60, 2), sin(2 * pi * 0.05 * seq(0, 60, 2)))
y <- pulstran(t, d, 'rectpuls')
plot(t, y, type = "l", xlab = "Time (s)", ylab = "Waveform",
     main = "Periodic rectangular pulse")

## assymetric sawtooth waveform
fs <- 1e3
t <- seq(0, 1, 1/fs)
d <- seq(0, 1, 1/3)
x <- tripuls(t, 0.2, -1)
y <- pulstran(t, d, x, fs)
plot(t, y, type = "l", xlab = "Time (s)", ylab = "Waveform",
     main = "Asymmetric sawtooth waveform")

## Periodic Gaussian waveform
fs <- 1e7
tc <- 0.00025
t <- seq(-tc, tc, 1/fs)
x <- gauspuls(t, 10e3, 0.5)
plot(t, x, type="l", xlab = "Time (s)", ylab = "Waveform",
     main = "Gaussian pulse")
ts <- seq(0, 0.025, 1/50e3)
d <- cbind(seq(0, 0.025, 1/1e3), sin(2 * pi * 0.1 * (0:25)))
y <- pulstran(ts, d, x, fs)
plot(ts, y, type = "l", xlab = "Time (s)", ylab = "Waveform",
     main = "Gaussian pulse train")

# Custom pulse trains
fnx <- function(x, fn) sin(2 * pi * fn * x) * exp(-fn * abs(x))
ffs <- 1000
tp <- seq(0, 1, 1/ffs)
pp <- fnx(tp, 30)
plot(tp, pp, type = "l",xlab = 'Time (s)', ylab = 'Waveform',
     main = "Custom pulse")
fs <- 2e3
t <- seq(0, 1.2, 1/fs)
d <- seq(0, 1, 1/3)
dd <- cbind(d, 4^-d)
z <- pulstran(t, dd, pp, ffs)
plot(t, z, type = "l", xlab = "Time (s)", ylab = "Waveform",
     main = "Custom pulse train")

}
\author{
Sylvain Pelissier, \email{sylvain.pelissier@gmail.com}.\cr
Conversion to R by Geert van Boxtel \email{G.J.M.vanBoxtel@gmail.com}.
}
