% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/graph3d.R
\name{graph3d}
\alias{graph3d}
\title{3D chart}
\usage{
graph3d(
  data = NULL,
  x = ~x,
  y = ~y,
  z = ~z,
  frame = NULL,
  style = NULL,
  type = "surface",
  surfaceColors = c("#FF0000", "#FFF000", "#00FF00", "#68E8FB", "#000FFF"),
  dataColor = NULL,
  xBarWidth = NULL,
  yBarWidth = NULL,
  xlab = NULL,
  ylab = NULL,
  zlab = NULL,
  xValueLabel = NULL,
  yValueLabel = NULL,
  zValueLabel = NULL,
  width = "100\%",
  height = "100\%",
  backgroundColor = NULL,
  showPerspective = TRUE,
  showGrid = TRUE,
  showShadow = FALSE,
  showXAxis = TRUE,
  showYAxis = TRUE,
  showZAxis = TRUE,
  axisColor = NULL,
  axisFontSize = 30,
  gridColor = NULL,
  keepAspectRatio = TRUE,
  verticalRatio = 0.5,
  tooltip = TRUE,
  tooltipDelay = NULL,
  tooltipStyle = NULL,
  showLegend = TRUE,
  legendLabel = NULL,
  cameraPosition = list(horizontal = 1, vertical = 0.5, distance = 2.8),
  xCenter = NULL,
  yCenter = NULL,
  xMin = NULL,
  xMax = NULL,
  yMin = NULL,
  yMax = NULL,
  zMin = NULL,
  zMax = NULL,
  xStep = NULL,
  yStep = NULL,
  zStep = NULL,
  showAnimationControls = TRUE,
  animationInterval = 100,
  animationPreload = TRUE,
  frameLabel = NULL,
  onclick = NULL,
  elementId = NULL
)
}
\arguments{
\item{data}{dataframe containing the data for the chart; if not \code{NULL},
the variables passed to \code{x}, \code{y}, \code{z}, \code{frame} and
\code{style} are searched among the columns of \code{data}}

\item{x}{a right-sided formula giving the variable for the locations of the
points on the x-axis; required}

\item{y}{a right-sided formula giving the variable for the locations of the
points on the y-axis; required}

\item{z}{a right-sided formula giving the variable for the locations of the
points on the z-axis; required}

\item{frame}{a right-sided formula giving the variable for the frames of the
animation; optional}

\item{style}{a right-sided formula required for \code{type="dot-color"}
and \code{type="dot-size"}; the variable given by this formula can be a
numeric vector for the data value appearing in the legend, or a list of
style properties; see the examples}

\item{type}{the type of the chart, one of \code{"bar"}, \code{"bar-color"},
\code{"bar-size"}, \code{"dot"}, \code{"dot-line"}, \code{"dot-color"},
\code{"dot-size"}, \code{"line"}, \code{"grid"}, or \code{"surface"}}

\item{surfaceColors}{a vector of colors for \code{type="surface"}, or a list
of the form
\code{list(hue = list(start=-360, end=360, saturation=50, brightness=100, colorStops=8))};
see the vis-graph3d documentation for more information}

\item{dataColor}{a string or a list; see the \code{type="line"} example and
the vis-graph3d documentation}

\item{xBarWidth, yBarWidth}{the widths of bars in x and y directions for
\code{type="bar"} and \code{type="bar-color"};
by default, the width is equal to the smallest distance between the data points}

\item{xlab}{string, the label on the x-axis}

\item{ylab}{string, the label on the y-axis}

\item{zlab}{string, the label on the z-axis}

\item{xValueLabel}{JavaScript function for custom formatting of the labels
along the x-axis, for example \code{JS("function(x){return (x * 100) + '\%'}")}}

\item{yValueLabel}{same as \code{xValueLabel} for the y-axis}

\item{zValueLabel}{same as \code{xValueLabel} for the z-axis}

\item{width, height}{the dimensions of the chart given as strings, in pixels
(e.g. \code{"400px"}) or percentages (e.g. \code{"80\%"})}

\item{backgroundColor}{the background color of the chart, either a string
giving a HTML color (like \code{"red"} or \code{"#00CC00"}), or a list of
the form \code{list(fill="black", stroke="yellow", strokeWidth=3)};
\code{fill} is the chart fill color, \code{stroke} is the color of the chart
border, and \code{strokeWidth} is the border width in pixels}

\item{showPerspective}{logical; if \code{TRUE}, the graph is drawn in
perspective: points and lines which are further away are drawn smaller}

\item{showGrid}{logical; if \code{TRUE}, grid lines are drawn in the
x-y surface}

\item{showShadow}{logical, whether to show shadow on the graph}

\item{showXAxis}{logical; if \code{TRUE}, x-axis and x-axis labels are drawn}

\item{showYAxis}{logical; if \code{TRUE}, y-axis and y-axis labels are drawn}

\item{showZAxis}{logical; if \code{TRUE}, z-axis and z-axis labels are drawn}

\item{axisColor}{a HTML color given as a string, the color of the axis lines
and the text along the axes}

\item{axisFontSize}{a positive number, the font size of the axes labels}

\item{gridColor}{a HTML color given as a string, the color of the grid lines}

\item{keepAspectRatio}{logical; if \code{TRUE}, the x-axis and the y-axis
keep their aspect ratio; if \code{FALSE}, the axes are scaled such that they
both have the same, maximum width}

\item{verticalRatio}{value between 0.1 and 1 which scales the vertical
size of the graph; when \code{keepAspectRatio=FALSE} and
\code{verticalRatio=1}, the graph will be a cube}

\item{tooltip}{logical, whether to see the tooltips, or a JavaScript
function to customize the tooltips; see the barplot example}

\item{tooltipDelay}{a number, the delay time in ms for the tooltip to appear
when the mouse cursor hovers over an x-y grid tile}

\item{tooltipStyle}{a list of tooltip style properties; see the vis-graph3d
documentation}

\item{showLegend}{logical, whether to see the legend if the graph type
supports it}

\item{legendLabel}{a string, the label of the legend}

\item{cameraPosition}{a list with three fields to set the initial rotation
and position if the camera: \code{horizontal}, a value in radians,
\code{vertical}, a value in radians between 0 and pi/2, and \code{distance},
the distance between 0.71 and 5 from the camera to the center of the graph}

\item{xCenter}{a string giving the horizontal center position of the graph
as a percentage (like \code{"50\%"}) or in pixels (like \code{"100px"});
default to \code{"55\%"}}

\item{yCenter}{same as \code{xCenter} for the vertical center position of
the graph; default to \code{"45\%"}}

\item{xMin}{minimum value for the x-axis; if not set, the smallest value of
\code{x} is used}

\item{xMax}{maximum value for the x-axis; if not set, the largest value of
\code{x} is used}

\item{yMin}{minimum value for the y-axis; if not set, the smallest value of
\code{y} is used}

\item{yMax}{maximum value for the y-axis; if not set, the largest value of
\code{y} is used}

\item{zMin}{minimum value for the z-axis; if not set, the smallest value of
\code{z} is used}

\item{zMax}{maximum value for the z-axis; if not set, the largest value of
\code{z} is used}

\item{xStep}{a number, the step size for the grid on the x-axis}

\item{yStep}{a number, the step size for the grid on the y-axis}

\item{zStep}{a number, the step size for the grid on the z-axis}

\item{showAnimationControls}{logical, only applicable when the graph
contains an animation (i.e. \code{frame} is not \code{NULL}), whether to
show the animation controls (buttons previous, start/stop, next, and a slider)}

\item{animationInterval}{a number, the animation interval in milliseconds;
default to 1000}

\item{animationPreload}{logical; if \code{FALSE}, the animation frames are
loaded as soon as they are requested; if \code{TRUE}, the animation frames
are automatically loaded in the background}

\item{frameLabel}{string, the label for the animation slider}

\item{onclick}{a JavaScript function to handle the click event on a point;
see the vis-graph3d documentation and the second example in
\code{\link{graph3d-shiny}}}

\item{elementId}{an id for the widget}
}
\description{
Generate an interactive 3D chart.
}
\details{
See the
\href{https://visjs.github.io/vis-graph3d/docs/graph3d/index.html#Configuration_Options}{vis-graph3d}
documentation.
}
\examples{
# 3d bar plot ####
dat <- data.frame(x = c(1,1,2,2), y = c(1,2,1,2), z = c(1,2,3,4))
graph3d(dat, type = "bar", zMin = 0)
# change bar widths
graph3d(dat, type = "bar", zMin = 0, xBarWidth = 0.3, yBarWidth = 0.3)
# with custom tooltips
graph3d(dat, type = "bar", zMin = 0,
        tooltip = JS(c("function(xyz){",
                       "  var x = 'X: ' + xyz.x.toFixed(2);",
                       "  var y = 'Y: ' + xyz.y.toFixed(2);",
                       "  var z = 'Z: ' + xyz.z.toFixed(2);",
                       "  return  x + '<br/>' + y + '<br/>' + z;",
                       "}"))
)

# bivariate Gaussian density ####
dat <- expand.grid(
  x = seq(-4,4,length.out=100),
  y = seq(-4,4,length.out=100)
)
dat <- transform(dat, density = dnorm(x)*dnorm(y))
graph3d(dat, z = ~density, keepAspectRatio = FALSE, verticalRatio = 1)

# animation ####
f <- function(x, y) sin(x/50) * cos(y/50) * 50 + 50
t_ <- seq(0, 2*pi, length.out = 90)[-90]
x_ <- y_ <- seq(0, 314, length.out = 50)
dat <- expand.grid(x = x_, y = y_, t = t_)
dat <- transform(dat, z = f(x*cos(t) - y*sin(t), x*sin(t) + y*cos(t)))
graph3d(dat, frame = ~t, tooltip = FALSE)

# scatterplot ####
dat <- iris
dat$style <- I(lapply(iris$Species, function(x){
  switch(as.character(x),
         setosa     = list(fill="red",   stroke="#'000"),
         versicolor = list(fill="green", stroke="#'000"),
         virginica  = list(fill="blue",  stroke="#'000"))
}))
graph3d(dat, x = ~Sepal.Length, y = ~Sepal.Width, z = ~Petal.Length,
        style = ~style, type = "dot-color", showLegend = FALSE)

# line ####
t_ <- seq(0, 2*pi, length.out = 200)
dat <- data.frame(
  x = cos(t_),
  y = sin(t_),
  z = 2 * cos(3*t_)
)
graph3d(dat, type = "line", dataColor = list(strokeWidth = 5, stroke = "red"),
        verticalRatio = 1)

# a complex function ####
dat <- expand.grid(
  x = seq(-1, 1, length.out = 100),
  y = seq(-1, 1, length.out = 100)
)
dat <- transform(dat, sine = sin(x + 1i*y))
dat <- transform(dat, modulus = Mod(sine), phase = Arg(sine))
graph3d(dat, z = ~modulus, style = ~phase, type = "dot-color",
        legendLabel = "phase")
}
