% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fpc.R
\name{fpc_utils}
\alias{fpc_utils}
\alias{fpc_coefs}
\alias{beta_fpc_coefs}
\alias{fpc_to_fdata}
\alias{fpc_to_beta}
\title{Utilities for functional principal components}
\usage{
fpc_coefs(X_fdata, X_fpc, ind_X_fpc = 1:3, int_rule = "trapezoid")

beta_fpc_coefs(beta, X_fpc, Y_fpc, ind_X_fpc = 1:3, ind_Y_fpc = 1:3,
  int_rule = "trapezoid")

fpc_to_fdata(coefs, X_fpc, ind_coefs = seq_len(ncol(coefs)))

fpc_to_beta(beta_coefs, X_fpc, Y_fpc,
  ind_coefs_X = seq_len(nrow(beta_coefs)),
  ind_coefs_Y = seq_len(ncol(beta_coefs)))
}
\arguments{
\item{X_fdata}{sample of functional data as an
\code{\link[fda.usc]{fdata}} object of length \code{n}.}

\item{X_fpc, Y_fpc}{\code{"fpc"} objects as resulted from calling
\code{\link{fpc}}.}

\item{ind_X_fpc, ind_Y_fpc}{vectors giving the FPC indexes for whom the
coefficients are computed. Their lengths must be smaller than the number of
FPC in \code{X_fpc} and \code{Y_fpc}, respectively. Default to \code{1:3}.}

\item{int_rule}{quadrature rule for approximating the definite
unidimensional integral: trapezoidal rule (\code{int_rule = "trapezoid"})
and extended Simpson rule (\code{int_rule = "Simpson"}) are available.
Defaults to \code{"trapezoid"}.}

\item{beta}{a matrix containing the bivariate kernel \eqn{\beta} evaluated
on a grid. Must be of size \code{c(length(X_fpc$rotation$argvals),
length(Y_fpc$rotation$argvals))}.}

\item{coefs}{a vector of coefficients to combine linearly the FPC. Its
length must be smaller than the number of FPC in \code{X_fpc}.}

\item{ind_coefs, ind_coefs_X, ind_coefs_Y}{indexes of FPC to associate to the
provided coefficients. By default, from the first FPC to the sizes of
\code{coefs} or \code{beta_coefs}.}

\item{beta_coefs}{a matrix of coefficients to combine linearly the tensor
products of FPC. Its size must be smaller than the number of FPC in
\code{X_fpc} and \code{Y_fpc}.}
}
\value{
\item{fpc_coefs}{a vector of the same length as \code{coefs} containing
the coefficients of \code{X_fdata} in the FPC of \code{X_fpc}.}
\item{beta_fpc_coefs}{a matrix of the same size as \code{beta_coefs}
containing the coefficients of \eqn{\beta} in the tensor product of
the FPC in \code{X_fpc} and \code{Y_fpc}.}
\item{fpc_to_fdata}{an \code{\link[fda.usc]{fdata}} object of the same
type as \code{X_fpc$rotation}.}
\item{fpc_to_beta}{a matrix with the reconstructed kernel and size\cr
\code{c(length(X_fpc$rotation$argvals), length(Y_fpc$rotation$argvals))}.}
}
\description{
Computation of coefficients and reconstructions based on
Functional Principal Components (FPC). The function \code{fpc_coefs} allows
to project a functional data sample into a basis of FPC; the reconstruction
of the sample from its projections and the FPC is done with
\code{fpc_to_fdata}. The functions \code{beta_fpc_coefs} and
\code{fpc_to_beta} do analogous operations but for the
\link[=flm_est]{bivariate kernel} \eqn{\beta} and the tensor product
of two FPC bases.
}
\examples{
## Compute FPC coefficients and reconstruct data

# Sample data
X_fdata <- r_ou(n = 200, t = seq(2, 4, l = 201))

# Compute FPC
X_fpc <- fpc(X_fdata = X_fdata, n_fpc = 50)

# FPC coefficients are the same if the data is centered
fpc_coefs(X_fdata = fdata_cen(X_fdata), X_fpc = X_fpc)[1:4, ]
X_fpc$scores[1:4, 1:3]

# Reconstruct the first two curves for an increasing number of FPC
plot(X_fdata[1:2, ], col = 1)
n_fpc <- c(2, 5, 10, 25, 50)
for (j in 1:5) {
  lines(fpc_to_fdata(X_fpc = X_fpc,
                     coefs = X_fpc$scores[, 1:n_fpc[j]])[1:2, ], col = j + 1)
}

## Project and reconstruct beta

# Surface
beta_fun <- function(s, t) sin(6 * pi * s) + cos(6 * pi * t)
s <- seq(0, 1, l = 101)
t <- seq(0, 1, l = 201)
beta_surf <- outer(s, t, FUN = beta_fun)

# Functional data as zero-mean Gaussian process with exponential variogram
X_fdata <- fda.usc::rproc2fdata(n = 100, t = s, sigma = "vexponential",
                                list = list(scale = 2.5))
Y_fdata <- flm_term(X_fdata = X_fdata, beta = beta_surf, t = t) +
  r_ou(n = 100, t = t, sigma = sqrt(0.075) * 2)

# FPC
X_fpc <- fpc(X_fdata = X_fdata, n_fpc = 50)
Y_fpc <- fpc(X_fdata = Y_fdata, n_fpc = 50)

# Coefficients
beta_coefs <- beta_fpc_coefs(beta = beta_surf, X_fpc = X_fpc, Y_fpc = Y_fpc,
                             ind_X_fpc = 1:50, ind_Y_fpc = 1:50)

# Reconstruction
beta_surf1 <- fpc_to_beta(beta_coefs = beta_coefs[1:2, 1:5],
                          X_fpc = X_fpc, Y_fpc = Y_fpc)
beta_surf2 <- fpc_to_beta(beta_coefs = beta_coefs[1:15, 1:10],
                          X_fpc = X_fpc, Y_fpc = Y_fpc)
beta_surf3 <- fpc_to_beta(beta_coefs = beta_coefs[1:50, 1:50],
                          X_fpc = X_fpc, Y_fpc = Y_fpc)

# Show reconstructions
old_par <- par(mfrow = c(2, 2))
col <- viridisLite::viridis(20)
image(s, t, beta_surf, col = col, zlim = c(-2.5, 2.5), main = "Original")
image(s, t, beta_surf1, col = col, zlim = c(-2.5, 2.5), main = "2 x 5")
image(s, t, beta_surf2, col = col, zlim = c(-2.5, 2.5), main = "15 x 10")
image(s, t, beta_surf3, col = col, zlim = c(-2.5, 2.5), main = "50 x 50")
par(old_par)
}
\references{
Jolliffe, I. T. (2002). Principal Component Analysis. Springer-Verlag,
New York.
}
\author{
Eduardo García-Portugués.
}
