
#----------------------------- Fitting GLM with Missing Values ------------------------
# Implementation is for the logit link, but should work for other link.
# Missing data could be present in the response variable y or in x variables
# Missing values in the covriate are assumed to be missing at random (MAR)
# Missing values in the response variable y are assumed to be nonignorable
# ------------------------------------------------- Missing -------------------------

#' Fitting binary regression model with missing responses based on Ibrahim and Lipsitz (1996)
#'
#' @description This function enables users to fit generalized linear models when handling incomplete data in the response variable. The missing responses are assumed to be nonignorable. The model is fitted using a novel likelihood-based method proposed by Ibrahim and Lipsitz(1996).
#' @param formula a formula expression as for regression models, of the form \code{response ~ predictors}. The response should be a numeric binary variable with missing values, and predictors can be any variables. A predictor with categorical values with missing can be used in the model. See the documentation of formula for other details.
#' @param data an optional data frame in which to interpret the variables occurring in formula.
#' @param adtnlCovforR an optional list of covariates to be used to fit the logistic regression \code{logit(R) ~ response+predictors+adtnlCovforR}. \code{adtnlCovforR} has to be supplied as a vector. Default is \code{NULL}.
#' @param eps0 arguments to be used to for the convergence criteria of the maximum likelihood computation of the joint likelihood function. The default is 1e-3.
#' @param maxit arguments to be used to for the maximization of the joint likelihood function. The default is 50.
#' @param family A character string specifying the type of model family.
#' @param method a method="brglmFit" or method="glm.fit" will be used for fitting model. The method="brglmFit" fits generalized linear models using bias reduction methods (Kosmidis, 2014), and other penalized maximum likelihood methods.

#' @details The `family` parameter in the `emil` function allows you to specify the probability distribution and link function for the response variable in the linear model. It determines the nature of the relationship between the predictors and the response variable.
#' The `family` argument is particularly important when working with binary data, where the response variable has only two possible outcomes. In such cases, you typically want to fit a binary regression model with an appropriate link.
#'
#' Currently the package only supports family=binomial for binary or dichotomous response variables.
#'
#' You can also specify different link functions within the family=binomial. The default link function is the logit function, which models the log-odds of success. Other available link functions include:
#'
#' - "probit" for the probit link function, which models the cumulative standard normal distribution.
#'
#' - "cloglog" for the complementary log-log link function, which models the complementary log-log of the survival function.
#'
#' It is important to choose the appropriate `family` and `link` function based on the specific characteristics and assumptions of your binary data. The default "binomial" family with the logit link function is often a good starting point, but alternative link functions might be more appropriate depending on the research question and the nature of the data.
#'
#' @return return the generalized linear model estimates
#' @importFrom stats terms glm coef fitted binomial
#' @importFrom dplyr %>%
#' @export
#'
#' @references
#' Firth, D. (1993). Bias reduction of maximum likelihood estimates, Biometrika, 80, 27-38. doi:10.2307/2336755.
#'
#' Ibrahim, J. G. (1990). Incomplete data in generalized linear models. Journal of the American Statistical Association 85, 765–769.
#'
#' Ibrahim, J. G., and Lipsitz, S. R. (1996). Parameter Estimation from Incomplete Data in Binomial Regression when the Missing Data Mechanism is Nonignorable, Biometrics, 52, 1071–1078.
#'
#' Kosmidis, I., Firth, D. (2021). Jeffreys-prior penalty, finiteness and shrinkage in binomial-response generalized linear models. Biometrika, 108, 71-82. doi:10.1093/biomet/asaa052.
#'
#' Louis, T. A. (1982). Finding the observed information when using the EM algorithm. Proceedings of the Royal Statistical Society, Ser B, 44, 226-233.
#'
#' Maity, A., Pradhan, V., Das U (2019). Bias reduction in logistic regression with missing responses when the missing data mechanism is nonignorable. The American Statistician, (73) 340-349.
#'
#' Pradhan V, Nychka DW, Bandyopadhyay S (2025).  Addressing Missing Responses and Categorical Covariates in Binary Regression Modeling: An Integrated Framework  (to be submitted).
#'

#' @examples
#' # using incontinence data
#' fit <- emil(y~x1+x2+x3,
#'                    data=incontinence,
#'                    family=binomial,
#'                    method="brglmFit")
#' summary(fit$fit_y)
#'
#' @importFrom utils globalVariables
#' @importFrom stats glm
#' @importFrom brglm2 brglmFit
#' @importFrom abind abind
#' @importFrom dplyr arrange group_by mutate summarise syms distinct
emil <- function(formula, data, adtnlCovforR = NULL, eps0 = 1e-5, maxit = 75, family = "binomial", method = "brglmFit") {


  # suppressMessages(library("dplyr"))
  # suppressMessages(require("abind"))
  # suppressMessages(library("MASS"))
  # suppressMessages(library("brglm2"))

  required_packages <- c("dplyr", "abind", "brglm2", "MASS")
  for (pkg in required_packages) {
    if (!requireNamespace(pkg, quietly = TRUE)) {
      stop(sprintf("Package '%s' needed for this function to work. Please install it.", pkg), call. = FALSE)
    }
  }
  

  formula_vars <- all.vars(formula)
  fn_data <- if (length(adtnlCovforR) > 0) {
    subset(data, select = c(formula_vars, adtnlCovforR))
  } else {
    subset(data, select = formula_vars)
  }

  resp <- all.vars(formula)[1]
  predictor_vars <- attr(terms(formula), "term.labels")
  VarWithMissingVal <- predictor_vars[colSums(is.na(data[, predictor_vars])) > 0]

  augmented <- dataAugmentation(fn_data, formula, adtnlCovforR = adtnlCovforR)
  df_augmented <- augmented$augData
  dfn <- augmented$distptrn

  theta <- rep(1 / dfn, dfn)
  beta <- rep(0.5, length(predictor_vars) + 1)

  data_all <- df_augmented
  xname <- predictor_vars
  xname_new <- if (length(adtnlCovforR) > 0) c(xname, adtnlCovforR) else xname
  yname_new <- resp
  rname_new <- "R"
  form_r <- form_gen(rname_new, c(yname_new, xname_new))
  N <- nrow(data_all)
  data_all$ry_weight <- 1

  fit_r <- suppressWarnings(stats::glm(form_r, data = data_all, family = stats::binomial(link = "logit"), method = if (method == "brglmFit") brglm2::brglmFit else method, weights = ry_weight))
  if (fit_r$converged) {
    oldalpha <- initial_coef_alpha <- fit_r$coefficients
    initial_se_alpha <- coef(summary(fit_r))[, 2]
    prob_r_1 <- fitted(fit_r)
    prob_r_0 <- rep(1, N) - prob_r_1
    prob_r <- sapply(seq_len(N), function(i) ifelse(data_all$R[i] == 0, prob_r_0[i], prob_r_1[i]))
  } else {
    return(list(converged = FALSE))
  }

  fit_y <- suppressWarnings(stats::glm(formula, data = data, family = family, method = if (method == "brglmFit") brglm2::brglmFit else method))
  initial_coef_beta <- fit_y$coefficients
  initial_se_beta <- coef(summary(fit_y))[, 2]
  init_omit <- length(fit_y$na.action)

  fit_y <- suppressWarnings(stats::glm(formula, data = data_all, family = family, method = if (method == "brglmFit") brglm2::brglmFit else method))
  if (fit_y$converged) {
    oldbeta <- fit_y$coefficients

    prob_y_1 <- fitted(fit_y)
    prob_y_0 <- rep(1, N) - prob_y_1
    prob_y <- sapply(seq_len(N), function(i) ifelse(eval(parse(text = paste0("data_all$", resp, "[i]"))) == 0, prob_y_0[i], prob_y_1[i]))
  } else {
    return(list(converged = FALSE))
  }

  data_all$prod <- prob_r * prob_y
  data_all <- dplyr::arrange(data_all, grp)
  tt.wgt <- data_all %>% dplyr::group_by(R, grp) %>% dplyr::mutate(wgt = prod / sum(prod))
  data_all$ry_weight <- ifelse(tt.wgt$R == 0, 1, tt.wgt$wgt)
  data_all$prod <- NULL
  oldpar <- c(oldalpha, oldbeta)

  eps <- 1
  i <- 0
  while (eps > eps0 && i <= maxit) {

    fit_y <- suppressWarnings(stats::glm(formula, data = data_all, family = family, method = if (method == "brglmFit") brglm2::brglmFit else method, weights = ry_weight))
    if (fit_y$converged) {
      newbeta <- fit_y$coefficients
      prob_y_1 <- fitted(fit_y)
      prob_y_0 <- rep(1, N) - prob_y_1
      prob_y <- sapply(seq_len(N), function(i) ifelse(eval(parse(text = paste0("data_all$", resp, "[i]"))) == 0, prob_y_0[i], prob_y_1[i]))
    } else {
      return(list(converged = FALSE))
    }

    fit_r <- suppressWarnings(stats::glm(form_r, data = data_all, family = stats::binomial(link = "logit"), method = if (method == "brglmFit") brglm2::brglmFit else method, weights = ry_weight))
    if (fit_r$converged) {
      newalpha <- fit_r$coefficients
      prob_r_1 <- fitted(fit_r)
      prob_r_0 <- rep(1, N) - prob_r_1
      prob_r <- sapply(seq_len(N), function(i) ifelse(data_all$R[i] == 0, prob_r_0[i], prob_r_1[i]))
    } else {
      return(list(converged = FALSE))
    }

    data_all$prod <- prob_r * prob_y
    data_all <- dplyr::arrange(data_all, grp)
    tt.wgt <- data_all %>% dplyr::group_by(R, grp) %>% dplyr::mutate(wgt = prod / sum(prod))
    data_all$ry_weight <- ifelse(tt.wgt$R == 0, 1, tt.wgt$wgt)
    data_all$prod <- NULL

    newpar <- c(newalpha, newbeta)
    diff <- abs(newpar - oldpar)
    eps <- sum(diff)
    oldpar <- newpar
    i <- i + 1
  }

  fit_y <- suppressWarnings(stats::glm(formula, family = family, data = data_all, method = if (method == "brglmFit") brglm2::brglmFit else method, weights = ry_weight))
  newbeta.1 <- fit_y$coefficients

  alpha <- data.frame(initial_coef_alpha, newalpha)
  names(alpha) <- c("Initial Est", "Final Est")
  beta <- data.frame(initial_coef_beta, newbeta.1)
  names(beta) <- c("Initial Est", "Final Est")

  temp_data_all <- data_all
  temp_data_all$wgt <- data_all$ry_weight
  vcov_beta <- louisvcov(formula, data = temp_data_all)
  se_beta_em <- sqrt(diag(vcov_beta))

  conv_em_se_beta <- coef(summary(fit_y))[, 2]
  conv_em_se_alpha <- coef(summary(fit_r))[, 2]
  #
  se_beta <- cbind(initial_se_beta, se_beta_em, conv_em_se_beta)
  colnames(se_beta) <- c("se_beta_ini", "se_beta_em", "se_beta_conv")
  #
  if (eps <= eps0 && i <= maxit) {
    return (list(fit_r = fit_r, fit_y = fit_y, beta_se_hat = se_beta_em, beta_hat = newbeta.1, converged=fit_y$converged))
  }
  else{
    return(list(converged = fit_y$converged))
  }
}

