/* glm.deploy: 'C' and 'Java' Source Code Generator for Fitted Glm Objects

# Copyright (c) 2018, Oscar J. Castro López, Inés F. Vega-López
#
# This file is part of the glm.deploy package for R.
#
# The glm.deploy package is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# The glm.deploy package is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (http://www.gnu.org/licenses/).
######################################################################################
 */
#include <string>
#include <fstream>
#include <Rcpp.h>
using namespace std;
using namespace Rcpp;

enum outlang {
  CANSI = 0,
  JAVA = 1
};

std::string getMultiHeader(outlang lang, string filename);
std::string getFunctionDeclaration(outlang lang, string functionName, CharacterVector arguments);
std::string getDataType(outlang lang, string type);
std::string getVarDeclaration(outlang lang,   CharacterVector fields,   CharacterVector types,   CharacterVector coefficients,   CharacterVector factors,   CharacterVector factorsname);
std::string getRegOperation(outlang lang, CharacterVector fields, CharacterVector coefficients);
std::string getFunctionArguments(outlang lang, CharacterVector arguments);
std::string getResponseFunction(outlang lang,  string functionNam, CharacterVector arguments,string linkfunction);
void saveFile(string filename, vector<string> filecontents);
std::string getMultiFooter(outlang lang);

//Main function that receives all the information needed to process a glm object and output the generated code in a file
// [[Rcpp::export]]
void glmdeploy_cpp(DataFrame dataFields,
                  CharacterVector arguments,
                  CharacterVector family,
                  CharacterVector link_function,
                  CharacterVector interceptname,
                  bool hasfactors,
                  CharacterVector filename_in,
                  int outlanguage,
                  CharacterVector path_in){

  CharacterVector fields =  dataFields[0];
  CharacterVector coefficients =  dataFields[1];
  CharacterVector types =  dataFields[2];
  CharacterVector factors =  dataFields[3];
  CharacterVector factorsname =  dataFields[4];
  std::string filename = Rcpp::as<std::string>(filename_in);
  std::string path = Rcpp::as<std::string>(path_in);

  outlang lang = static_cast<outlang>(outlanguage);

  std::string functionName = Rcpp::as<std::string>(interceptname);

  std::string inisignature = "/* This file was generated by the glm.deploy R package */\n";
  inisignature += "/* Thanks for using our tool which is free under the GPL3.0, please cite the package or put a link to our project. */\n";
  inisignature += "/* https://github.com/oscarcastrolopez/glm.deploy */";

  vector<string> filecontents;

  filecontents.push_back(inisignature);
  filecontents.push_back(getMultiHeader(lang,filename));
  std::string functionprototype_link = getFunctionDeclaration(lang,functionName+"_link",arguments);
  filecontents.push_back(functionprototype_link+"{");
  filecontents.push_back(getVarDeclaration(lang,fields,types,coefficients,factors,factorsname));
  filecontents.push_back(getRegOperation(lang,fields,coefficients));
  filecontents.push_back("}");
  std::string functionprototype_response = getFunctionDeclaration(lang,functionName+"_response",arguments);
  filecontents.push_back(functionprototype_response+"{");
  filecontents.push_back(getResponseFunction(lang,functionName, arguments,Rcpp::as<std::string>(link_function)));
  filecontents.push_back("}");


  Rcout << "GLM model family: "<<family << "With Link Function:"<< link_function <<"\n";
  //Variable needed if link function needs to compute erf()
  std::string erffunction;


  vector<string> filelibrary;
  switch(lang){
  case CANSI:
    filecontents.push_back(getMultiFooter(lang));
    saveFile(path + '/' + filename + ".c", filecontents);
    Rcout << "File saved: " << path + '/' + filename + ".c" <<"\n";
    //Make .h library of the code

    filelibrary.push_back(inisignature);
    filelibrary.push_back(functionprototype_link + ";");
    filelibrary.push_back(functionprototype_response + ";");
    saveFile(path + '/' + filename + ".h", filelibrary);
    Rcout << "File saved: " << path + '/' + filename + ".h" <<"\n";
    break;
  case JAVA:
    //Add the erf(x) function if the link function needs it
    if(Rcpp::as<std::string>(link_function).compare("probit") == 0 || Rcpp::as<std::string>(link_function).compare("1/mu^2") == 0){
      erffunction += "/* Low-order Chebyshev approximation, return the erf(x) with fractional error less than 1.2 x 10^-7\n";
      erffunction += "* From the book: Numerical recipes, 6.2.2\n";
      erffunction += "*/\n";
      erffunction += "public static double erf(double x) {\n";
      erffunction += "double t,ans;\n";
      erffunction += "double z=Math.abs(x);\n";
      erffunction += "t = 2/(2+z);\n";
      erffunction += "ans = 1- t * Math.exp(-z * z - 1.26551223 + t * ( 1.00002368 + t * ( 0.37409196 + t * ( 0.09678418 + t * (-0.18628806 + t * ( 0.27886807 + t * (-1.13520398 + t * ( 1.48851587 + t * (-0.82215223 + t * ( 0.17087277))))))))));\n";
      erffunction += "return (x>=0.0)?ans:2.0-ans;\n";
      erffunction += "}\n";
    }
    filecontents.push_back(erffunction);
    filecontents.push_back(getMultiFooter(lang));
    saveFile(path + '/' + filename + "_class.java", filecontents);
    Rcout << "File saved: " << path + '/' + filename + "_class.java" <<"\n";
    break;
  default:
    break;
  }

}

std::string getMultiHeader(outlang lang,
                           string filename){
  std::string header;
  switch(lang){
  case CANSI:
    header += "#include <stdlib.h>\n";
    header += "#include <stdio.h>\n";
    header += "#include <string.h>\n";
    header += "#include <math.h>\n";
    break;
  case JAVA:
    header += "//Change to the name of your package\n";
    header += "package test;\n";
    header += "public class "+filename+"_class{\n";
    break;
  default:
    break;
  }
  return header;
}

std::string getMultiFooter(outlang lang){
  std::string footer;
  switch(lang){
  case CANSI:
    break;
  case JAVA:
    footer += "}";
    break;
  default:
    break;
  }
  return footer;
}

/*Get list of argument names and types */
/*All numeric fields are treated as Double*/
/*All factor fields are treated as Strings*/
std::string getFunctionDeclaration(outlang lang, string functionName, CharacterVector arguments){
  std::string fundec = "NULL";
  switch(lang){
  case CANSI:
    return "double glm_"+ functionName + "(" + getFunctionArguments(lang,arguments) + ")";
  case JAVA:
    return "public static double glm_"+ functionName + "(" + getFunctionArguments(lang,arguments) + ")";
  default:
    break;
  }
  return fundec;
}
//Build the list of arguments separated by a comma
// Ex: (type var1, type var2, type var3)
std::string getFunctionArguments(outlang lang,
                                 CharacterVector arguments){
  CharacterVector argumentsnames = arguments.names();
  int a = arguments.size();
  std::string functionArguments;
  for ( int i = 1; i < a; ++i ) {
    std::string temparg = Rcpp::as<std::string>(arguments[i]);
    std::string tempargname = Rcpp::as<std::string>(argumentsnames[i]);
    switch(lang){
    case CANSI:
      temparg = getDataType(lang, temparg);
      functionArguments += temparg +  " " + tempargname;
      if(i+1 != a)
        functionArguments += ", ";
      break;
    case JAVA:
      temparg = getDataType(lang, temparg);
      functionArguments += temparg +  " " + tempargname;
      if(i+1 != a)
        functionArguments += ", ";
      break;
    default:
      break;
    }
  }
  return functionArguments;
}

//factors are treated as strings
//numeric is treated as double
std::string getDataType(outlang lang,
                        string type){
  switch(lang){
  case CANSI:
    if(type.compare("factor") == 0){
      return "char*";
    }else if(type.compare("numeric") == 0){
      return "double";
    }else{
      stop("Data Type not contemplated: %s", type);
    }
    break;
  case JAVA:
    if(type.compare("factor") == 0){
      return "String";
    }else if(type.compare("numeric") == 0){
      return "double";
    }else{
      stop("Data Type not contemplated: %s", type);
    }
    break;
  default:
    break;
  }
  return "NULL";
}

//Builds the var declaration section of the function.
//If the model has factor variables they are treated as a string, and they match the right variable and coefficient for the formula.
//If the model has numeric variables they are treated as a double.
std::string getVarDeclaration(outlang lang,
                              CharacterVector fields,
                              CharacterVector types,
                              CharacterVector coefficients,
                              CharacterVector factors,
                              CharacterVector factorsname){
  int a = fields.size();
  std::string vardeclaration;
  for ( int i = 1; i < a; ++i ) {
    std::string tempfield = Rcpp::as<std::string>(fields[i]);
    std::string temptype = Rcpp::as<std::string>(types[i]);
    std::string tempcoef = Rcpp::as<std::string>(coefficients[i]);
    switch(lang){
    case CANSI:
      if(temptype.compare("factor")==0){
        std::string tempfactor = Rcpp::as<std::string>(factors[i]);
        std::string tempfactorname = Rcpp::as<std::string>(factorsname[i]);
        vardeclaration += "double new_" + tempfield + " = (strcmp(" + tempfactorname + ",\"" + tempfactor + "\")==0)?" + tempcoef+ ":0;\n";
      }else if(temptype.compare("numeric")==0){
        vardeclaration += "double new_" + tempfield + " = " + tempcoef + " * " + tempfield + ";\n";
      }
      break;
    case JAVA:
      if(temptype.compare("factor")==0){
        std::string tempfactor = Rcpp::as<std::string>(factors[i]);
        std::string tempfactorname = Rcpp::as<std::string>(factorsname[i]);
        vardeclaration += "double new_" + tempfield + " = (" + tempfactorname + ".equals(\"" + tempfactor + "\"))?" + tempcoef+ ":0;\n";
      }else if(temptype.compare("numeric")==0){
        vardeclaration += "double new_" + tempfield + " = " + tempcoef + " * " + tempfield + ";\n";
      }
      break;
    default:
      break;
    }
  }
  return vardeclaration;
}

//Builds the regression operation
std::string getRegOperation(outlang lang,
                            CharacterVector fields,
                            CharacterVector coefficients){
  int a = fields.size();
  std::string regoperation;
  for ( int i = 0; i < a; ++i ) {
    std::string tempfield = Rcpp::as<std::string>(fields[i]);
    std::string tempcoef = Rcpp::as<std::string>(coefficients[i]);
    switch(lang){
    case CANSI:
      if(i==0){
        regoperation += tempcoef;
      }else{
        regoperation += "+new_" + tempfield;
      }
      break;
    case JAVA:
      if(i==0){
        regoperation += tempcoef;
      }else{
        regoperation += "+new_" + tempfield;
      }
      break;
    default:
      break;
    }

  }
  regoperation ="return " + regoperation +";";
  return regoperation;
}

//Builds the _response function, same as predict with type="response"
std::string getResponseFunction(outlang lang,
                                string functionName,
                                CharacterVector arguments,
                                string linkfunction){
  CharacterVector argumentsnames = arguments.names();
  int a = arguments.size();
  std::string functionArguments;
  for ( int i = 1; i < a; ++i ) {
    std::string tempargname = Rcpp::as<std::string>(argumentsnames[i]);
    switch(lang){
    case CANSI:
      functionArguments += tempargname;
      if(i+1 != a)
        functionArguments += ", ";
      break;
    case JAVA:
      functionArguments += tempargname;
      if(i+1 != a)
        functionArguments += ", ";
      break;
    default:
      break;
    }
  }
  std::string funcall = "glm_" + functionName + "_link(" + functionArguments + ")";

  //Add the link function
  switch(lang){
  case CANSI:
    if(linkfunction.compare("identity") == 0){
      return "return " + funcall + ";";
    }else if(linkfunction.compare("logit") == 0){// =1/(1+EXP(Y))
      return "return 1/(1+exp(-" + funcall+"));";
    }else if(linkfunction.compare("probit") == 0){// =0.5*(1+ERF((Y-0)/1*SQRT(2)))
      return "return 0.5*(1+erf(("+funcall+"-0)/1*sqrt(2)));";
    }else if(linkfunction.compare("cloglog") == 0){// =1-EXP(-EXP(Y))
      return "return 1-exp(-exp("+funcall+"));";
    }else if(linkfunction.compare("log") == 0){// =EXP(Y)
      return "return exp("+funcall+");";
    }else if(linkfunction.compare("inverse") == 0){// =Y^-1
      return "return pow("+funcall+",-1);";
    }else if(linkfunction.compare("sqrt") == 0){// =SQRT(O9)
      return "return sqrt("+funcall+");";
    }else if(linkfunction.compare("cauchit") == 0){// = 0.5 + (1/π) arctan(Y) .... n = pi
      return "return 0.5 + (1/3.141593)*arctan("+funcall+");";
    }else if(linkfunction.compare("loglog") == 0){// = exp( -exp(-Y))
      return "return exp(-exp(-"+funcall+"));";
    }else if(linkfunction.compare("1/mu^2") == 0){// =0.5* 1+ERF((0/2*Y)*(Y/1-1))+0.5*EXP(2*0/1)* 1-ERF((0/2*Y)*(Y/1+1))
      return "double Y = " +funcall+";\nreturn (0.5* 1+erf((0/2*Y)*(Y/1-1))+0.5*exp(2*0/1)* 1-erf((0/2*Y)*(Y/1+1)));\n";
    }else{
      stop("Link function not supported: %s", linkfunction);
    }
    break;
  case JAVA:
    if(linkfunction.compare("identity") == 0){
      return "return " + funcall + ";";
    }else if(linkfunction.compare("logit") == 0){// =1/(1+EXP(Y))
      return "return 1/(1+Math.exp(-" + funcall+"));";
    }else if(linkfunction.compare("probit") == 0){// =0.5*(1+ERF((Y-0)/1*SQRT(2)))
      return "return 0.5*(1+erf(("+funcall+"-0)/1*Math.sqrt(2)));";
    }else if(linkfunction.compare("cloglog") == 0){// =1-EXP(-EXP(Y))
      return "return 1-Math.exp(-Math.exp("+funcall+"));";
    }else if(linkfunction.compare("log") == 0){// =EXP(Y)
      return "return Math.exp("+funcall+");";
    }else if(linkfunction.compare("inverse") == 0){// =Y^-1
      return "return Math.pow("+funcall+",-1);";
    }else if(linkfunction.compare("sqrt") == 0){// =SQRT(O9)
      return "return Math.sqrt("+funcall+");";
    }else if(linkfunction.compare("cauchit") == 0){// = 0.5 + (1/π) arctan(Y) .... n = pi
      return "return 0.5 + (1/3.141593)*Math.arctan("+funcall+");";
    }else if(linkfunction.compare("loglog") == 0){// = exp( -exp(-Y))
      return "return Math.exp(-Math.exp(-"+funcall+"));";
    }else if(linkfunction.compare("1/mu^2") == 0){// =0.5* 1+ERF((0/2*Y)*(Y/1-1))+0.5*EXP(2*0/1)* 1-ERF((0/2*Y)*(Y/1+1))
      return "double Y = " +funcall+";\nreturn (0.5* 1+erf((0/2*Y)*(Y/1-1))+0.5*Math.exp(2*0/1)* 1-erf((0/2*Y)*(Y/1+1)));\n";
    }else{
      stop("Link function not supported: %s", linkfunction);
    }
    break;
  default:
    break;
  }
  return funcall;
}

//Function that saves the output file.
void saveFile(string outfilename,
              vector<string> filecontents){
  std::ofstream file;
  file.open(outfilename.c_str());
  for (std::vector<string>::const_iterator i = filecontents.begin(); i != filecontents.end(); ++i)
    file << *i << '\n';
  file.close();
}
