% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kw.R
\name{pkw}
\alias{pkw}
\title{Cumulative Distribution Function (CDF) of the Kumaraswamy (Kw) Distribution}
\usage{
pkw(q, alpha = 1, beta = 1, lower.tail = TRUE, log.p = FALSE)
}
\arguments{
\item{q}{Vector of quantiles (values generally between 0 and 1).}

\item{alpha}{Shape parameter \code{alpha} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{beta}{Shape parameter \code{beta} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{lower.tail}{Logical; if \code{TRUE} (default), probabilities are
\eqn{P(X \le q)}, otherwise, \eqn{P(X > q)}.}

\item{log.p}{Logical; if \code{TRUE}, probabilities \eqn{p} are given as
\eqn{\log(p)}. Default: \code{FALSE}.}
}
\value{
A vector of probabilities, \eqn{F(q)}, or their logarithms/complements
depending on \code{lower.tail} and \code{log.p}. The length of the result
is determined by the recycling rule applied to the arguments (\code{q},
\code{alpha}, \code{beta}). Returns \code{0} (or \code{-Inf} if
\code{log.p = TRUE}) for \code{q <= 0} and \code{1} (or \code{0} if
\code{log.p = TRUE}) for \code{q >= 1}. Returns \code{NaN} for invalid
parameters.
}
\description{
Computes the cumulative distribution function (CDF), \eqn{P(X \le q)}, for the
two-parameter Kumaraswamy (Kw) distribution with shape parameters \code{alpha}
(\eqn{\alpha}) and \code{beta} (\eqn{\beta}). This distribution is defined
on the interval (0, 1).
}
\details{
The cumulative distribution function (CDF) of the Kumaraswamy (Kw)
distribution is given by:
\deqn{
F(x; \alpha, \beta) = 1 - (1 - x^\alpha)^\beta
}
for \eqn{0 < x < 1}, \eqn{\alpha > 0}, and \eqn{\beta > 0}.

The Kw distribution is a special case of several generalized distributions:
\itemize{
\item Generalized Kumaraswamy (\code{\link{pgkw}}) with \eqn{\gamma=1, \delta=0, \lambda=1}.
\item Exponentiated Kumaraswamy (\code{\link{pekw}}) with \eqn{\lambda=1}.
\item Kumaraswamy-Kumaraswamy (\code{\link{pkkw}}) with \eqn{\delta=0, \lambda=1}.
}
The implementation uses the closed-form expression for efficiency.
}
\examples{
\donttest{
# Example values
q_vals <- c(0.2, 0.5, 0.8)
alpha_par <- 2.0
beta_par <- 3.0

# Calculate CDF P(X <= q) using pkw
probs <- pkw(q_vals, alpha_par, beta_par)
print(probs)

# Calculate upper tail P(X > q)
probs_upper <- pkw(q_vals, alpha_par, beta_par, lower.tail = FALSE)
print(probs_upper)
# Check: probs + probs_upper should be 1
print(probs + probs_upper)

# Calculate log CDF
logs <- pkw(q_vals, alpha_par, beta_par, log.p = TRUE)
print(logs)
# Check: should match log(probs)
print(log(probs))

# Compare with pgkw setting gamma = 1, delta = 0, lambda = 1
probs_gkw <- pgkw(q_vals, alpha_par, beta_par,
  gamma = 1.0, delta = 0.0,
  lambda = 1.0
)
print(paste("Max difference:", max(abs(probs - probs_gkw)))) # Should be near zero

# Plot the CDF for different shape parameter combinations
curve_q <- seq(0.001, 0.999, length.out = 200)
plot(curve_q, pkw(curve_q, alpha = 2, beta = 3),
  type = "l",
  main = "Kumaraswamy CDF Examples", xlab = "q", ylab = "F(q)",
  col = "blue", ylim = c(0, 1)
)
lines(curve_q, pkw(curve_q, alpha = 3, beta = 2), col = "red")
lines(curve_q, pkw(curve_q, alpha = 0.5, beta = 0.5), col = "green")
lines(curve_q, pkw(curve_q, alpha = 5, beta = 1), col = "purple")
lines(curve_q, pkw(curve_q, alpha = 1, beta = 3), col = "orange")
legend("bottomright",
  legend = c("a=2, b=3", "a=3, b=2", "a=0.5, b=0.5", "a=5, b=1", "a=1, b=3"),
  col = c("blue", "red", "green", "purple", "orange"), lty = 1, bty = "n", ncol = 2
)
}

}
\references{
Kumaraswamy, P. (1980). A generalized probability density function for
double-bounded random processes. \emph{Journal of Hydrology}, \emph{46}(1-2), 79-88.

Jones, M. C. (2009). Kumaraswamy's distribution: A beta-type distribution
with some tractability advantages. \emph{Statistical Methodology}, \emph{6}(1), 70-81.
}
\seealso{
\code{\link{pgkw}}, \code{\link{pekw}}, \code{\link{pkkw}} (related generalized CDFs),
\code{\link{dkw}}, \code{\link{qkw}}, \code{\link{rkw}} (other Kw functions),
\code{\link[stats]{pbeta}}
}
\author{
Lopes, J. E.
}
\keyword{cumulative}
\keyword{distribution}
\keyword{kumaraswamy}
