## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse  = TRUE,
  comment   = "#>",
  fig.width = 7,
  fig.height = 4.5,
  out.width = "100%"
)

## ----install, eval = FALSE----------------------------------------------------
# # Install from CRAN
# install.packages("ggvariant")
# 
# # Or install the development version from GitHub
# # remotes::install_github("yourname/ggvariant")

## ----load---------------------------------------------------------------------
library(ggvariant)

## ----read-vcf-----------------------------------------------------------------
vcf_file <- system.file("extdata", "example.vcf", package = "ggvariant")
variants  <- read_vcf(vcf_file)

head(variants)

## ----coerce, eval = FALSE-----------------------------------------------------
# # Example: data exported from a custom pipeline or Excel
# my_df <- read.csv("my_variants.csv")
# 
# variants <- coerce_variants(my_df,
#   chrom       = "Chr",
#   pos         = "Position",
#   ref         = "Ref_Allele",
#   alt         = "Alt_Allele",
#   consequence = "Variant_Class",
#   gene        = "Hugo_Symbol",
#   sample      = "Tumor_Sample"
# )

## ----lollipop-basic-----------------------------------------------------------
plot_lollipop(variants, gene = "TP53")

## ----lollipop-domains---------------------------------------------------------
tp53_domains <- data.frame(
  name  = c("Transactivation", "DNA-binding", "Tetramerization"),
  start = c(1,   102, 323),
  end   = c(67,  292, 356)
)

# Scale genomic positions to protein coordinates
tp53 <- variants[variants$gene == "TP53", ]
tp53$pos <- round(
  (tp53$pos - min(tp53$pos)) /
  (max(tp53$pos) - min(tp53$pos)) * 393
) + 1

plot_lollipop(tp53, gene = "TP53",
              domains        = tp53_domains,
              protein_length = 393)

## ----lollipop-sample----------------------------------------------------------
plot_lollipop(variants, gene = "TP53", color_by = "sample")

## ----lollipop-custom----------------------------------------------------------
library(ggplot2)

plot_lollipop(variants, gene = "KRAS") +
  labs(subtitle = "KRAS mutations across TUMOR_S1 and TUMOR_S2") +
  theme(legend.position = "bottom")

## ----consequence-sample-------------------------------------------------------
plot_consequence_summary(variants)

## ----consequence-fill---------------------------------------------------------
plot_consequence_summary(variants, position = "fill")

## ----consequence-gene---------------------------------------------------------
plot_consequence_summary(variants, group_by = "gene", top_n = 7)

## ----spectrum-----------------------------------------------------------------
plot_variant_spectrum(variants)

## ----spectrum-facet-----------------------------------------------------------
plot_variant_spectrum(variants, facet_by_sample = TRUE)

## ----interactive, eval = FALSE------------------------------------------------
# # Requires the plotly package
# # install.packages("plotly")
# 
# p <- plot_lollipop(variants, gene = "TP53", interactive = TRUE)
# p  # opens in RStudio viewer or browser
# 
# # Save as a standalone HTML file
# htmlwidgets::saveWidget(p, "TP53_lollipop.html")

## ----palette------------------------------------------------------------------
# See the consequence colour palette
gv_palette("consequence")

# See the COSMIC SBS spectrum palette
gv_palette("spectrum")

## ----theme, eval = FALSE------------------------------------------------------
# ggplot(my_data, aes(x, y)) +
#   geom_point() +
#   theme_ggvariant()

## ----session------------------------------------------------------------------
sessionInfo()

